/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.brooklyn.entity.zookeeper;

import java.util.List;

import org.apache.brooklyn.api.catalog.Catalog;
import org.apache.brooklyn.api.entity.ImplementedBy;
import org.apache.brooklyn.api.sensor.AttributeSensor;
import org.apache.brooklyn.config.ConfigKey;
import org.apache.brooklyn.core.config.ConfigKeys;
import org.apache.brooklyn.core.sensor.BasicAttributeSensorAndConfigKey;
import org.apache.brooklyn.core.sensor.Sensors;
import org.apache.brooklyn.entity.group.DynamicCluster;
import org.apache.brooklyn.util.core.flags.SetFromFlag;
import org.apache.brooklyn.util.guava.Suppliers;

import com.google.common.base.Predicates;
import com.google.common.base.Supplier;
import com.google.common.reflect.TypeToken;

@Catalog(name="ZooKeeper ensemble", description="A cluster of ZooKeeper servers. "
        + "Apache ZooKeeper enables highly reliable distributed coordination.")
@ImplementedBy(ZooKeeperEnsembleImpl.class)
public interface ZooKeeperEnsemble extends DynamicCluster {

    @SetFromFlag("clusterName")
    BasicAttributeSensorAndConfigKey<String> CLUSTER_NAME = new BasicAttributeSensorAndConfigKey<String>(String.class,
            "zookeeper.cluster.name", "Name of the Zookeeper cluster", "BrooklynZookeeperCluster");

    @SetFromFlag("initialSize")
    ConfigKey<Integer> INITIAL_SIZE = ConfigKeys.newConfigKeyWithDefault(DynamicCluster.INITIAL_SIZE, 3);

    ConfigKey<Supplier<Integer>> NODE_ID_SUPPLIER = ConfigKeys.builder(new TypeToken<Supplier<Integer>>() {})
            .name("zookeeper.nodeId.supplier")
            .description("Supplies values for members id in zoo.cfg")
            .defaultValue(Suppliers.incrementing())
            .constraint(Predicates.notNull())
            .build();

    AttributeSensor<List<String>> ZOOKEEPER_SERVERS = Sensors.newSensor(new TypeToken<List<String>>() { },
            "zookeeper.servers", "Hostnames to connect to cluster with");

    AttributeSensor<String> ZOOKEEPER_ENDPOINTS = Sensors.newStringSensor(
            "zookeeper.endpoints", "A comma-separated host:port list of members of the ensemble");

    /** @deprecated since 0.10.0 use <code>sensors().get(ZooKeeperEnsemble.CLUSTER_NAME)</code> instead */
    @Deprecated
    String getClusterName();
}
