/*
 * TemplateHandler.java
 *
 * Brazil project web application toolkit,
 * export version: 2.1 
 * Copyright (c) 1998-2004 Sun Microsystems, Inc.
 *
 * Sun Public License Notice
 *
 * The contents of this file are subject to the Sun Public License Version 
 * 1.0 (the "License"). You may not use this file except in compliance with 
 * the License. A copy of the License is included as the file "license.terms",
 * and also available at http://www.sun.com/
 * 
 * The Original Code is from:
 *    Brazil project web application toolkit release 2.1.
 * The Initial Developer of the Original Code is: suhler.
 * Portions created by suhler are Copyright (C) Sun Microsystems, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s): cstevens, drach, suhler.
 *
 * Version:  2.5
 * Created by suhler on 98/09/14
 * Last modified by suhler on 04/12/30 12:38:25
 */

package sunlabs.brazil.template;

import sunlabs.brazil.handler.MatchString;
import sunlabs.brazil.server.FileHandler;
import sunlabs.brazil.server.Handler;
import sunlabs.brazil.server.Request;
import sunlabs.brazil.server.Server;
import sunlabs.brazil.util.http.HttpUtil;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.Properties;

/**
 * The <code>TemplateHandler</code> reads a template file from
 * the document root, based on the URL, and passes the content through
 * one or more template filters.
 * <p>
 * The following configuration parameters are used to initialize this
 * <code>Handler</code>: <dl class=props>
 * <dt>prefix, suffix, glob, match
 * <dd>Specify the URL that triggers this handler.
 *	By default, all URL's are considered.
 *	Only documents that are mime sub-types of <code>text</code>
 *	are processed. See {@link sunlabs.brazil.server.FileHandler} for a
 *	description of how to set mime types for url suffixes.
 * (See {@link MatchString}).
 * <dt> <code>templates</code>
 * <dd> A list of template names.  For each name in the list, the property
 *      <code><i>name</i>.class</code> is examined to determine which class to
 *      use for each template.  Then <code>name</code> is used as the prefix
 *      for other template specific properties if any.  If
 *      <code><i>name</i>.class</code> does not exist, then <code>name</code>
 *      is assumed to be the class name, and there are no template specific
 *      properties for the template.  Methods in the template classes will be
 *      invoked to process the XML/HTML tags present in the content.
 * <dt> <code>session</code>
 * <dd> The name of the request property that the Session ID will be found
 *	in, used to identify the proper template instance.
 *	The default value is "SessionID".  Typically, a sessionHandler, 
 *	such as {@link sunlabs.brazil.handler.CookieSessionHandler} is used
 *	upstream to create the sessionID.  If no id is found, then the
 *	session named "common" is used instead.  Exactly one instance of
 *	each template class is created for each session.
 * <dt> <code>default</code>
 * <dd> The default file in the directory to use as a template if
 *	a directory name is specified.  Defaults to index[suffix], 
 *	or "index.html" if no suffix is provided.
 * <dt> <code>encoding</code>
 * <dd> The character encoding to use to interpret the template.
 *	If no encoding is specified, the default encoding is used.
 *      The template is read from the filesystem, and converted into a 
 *	String using this encoding.  All template processing is done
 *	using the String representation.
 * <dt> <code>outputEncoding</code>
 * <dd> The character encoding to use to interpret the template results.
 *	If no "outputEncoding" is specified, then "encoding" is used.
 *      Once template processing is complete, the results are converted
 *      into a byte stream for transmission to the client using the
 *	"outputEncoding", if specified.  If not specified then the HTTP
 *	default (8-bit ASCII) encoding is used.
 * <dt> <code>modified</code>
 * <dd> if present (e.g. set to any value) an HTTP <code>
 *	last-modified</code> header is added to the response with the
 *	current time.
 * <dt> <code>debug</code>
 *	if set to "true", template debugging is enabled: templates will emit
 *	their pre-processed markup as an HTML comment.  This parameter only takes effect
 *	if the <code>debug</code> option is not specified for an individual template.
 * <dd>
 * <dt> <code>tagPrefix</code>
 * <dd> If specified, all tag names defined for each template class
 *      are prefixed with <i>tagPrefix</i>.
 *	This parameter only takes effect
 *	if the <code>tagPrefix</code> option is not specified for an individual template.
 * <dd>
 * </dl>
 * <p>The request properties <code>DirectoryName</code> and
 *    <code>fileName</code> may be set as a convenience for downstream handlers.
 * <p>
 * This handler duplicates some of the functionality of the
 * {@link  sunlabs.brazil.handler.DefaultFileHandler template filter}, 
 * so that it may be used by itself in simple configurations.  As such, 
 * if issues re-directs if directories are given without a trailing "/", 
 * and uses an "index" file (see <code>default</code> above) if a directory
 * name is specified.
 * <p>
 * To filter content other than from the file system, use the
 * {@link  sunlabs.brazil.filter.TemplateFilter template filter} instead.
 *
 * @author	Stephen Uhler (stephen.uhler@sun.com)
 * @author	Colin Stevens (colin.stevens@sun.com)
 * @version	2.5 04/12/30
 */

public class TemplateHandler implements Handler {
    static final String HANDLER = "handler";
    static final String TEMPLATES = "templates";      // the class to process these templates
    static final String SESSION = "session";  
    static final String ENCODING = "encoding";  
    static final String DEFAULT = "default";    // property for default document, given directory

    String propsPrefix;		// our perfix in the config file
    String sessionProperty;	// where to find the session name
    MatchString isMine;         // check for matching url
    boolean modified;		// if set, emit last-modified header
    String encoding;		// the page character encoding

    Handler handler;
    TemplateRunner runner;		// The template object for our class

    public boolean
    init(Server server, String propsPrefix) {

	this.propsPrefix = propsPrefix;
	isMine = new MatchString(propsPrefix, server.props);

	Properties props = server.props;
        sessionProperty = props.getProperty(propsPrefix + SESSION, "SessionID");
        encoding = props.getProperty(propsPrefix + ENCODING);
        modified = (props.getProperty(propsPrefix + "modified") != null);

	/*
	 * Gather the templates.
	 */

	String str;
	str = server.props.getProperty(propsPrefix + TEMPLATES);
	if (str == null) {
	    server.log(Server.LOG_ERROR, propsPrefix,
		    "no " + TEMPLATES + " property specified");
	    return false;
	}

	try {
	    runner = new TemplateRunner(server, propsPrefix, str);
	} catch (ClassCastException e) {
	    server.log(Server.LOG_ERROR, e.getMessage(), "not a Template");
	    return false;
	} catch (ClassNotFoundException e) {
	    server.log(Server.LOG_ERROR, e.getMessage(), "unknown class");
	    return false;
	}
	return true;
    }

    /**
     * Process an html template file, using the supplied template
     * processing classes.
     */

    public boolean
    respond(Request request) throws IOException {
	if (!isMine.match(request.url)) {
	    return false;
        }

	Properties props = request.props;
	String root = props.getProperty(propsPrefix + FileHandler.ROOT,
		props.getProperty(FileHandler.ROOT, "."));
	String name = FileHandler.urlToPath(request.url);
	File file = new File(root + name);
	String path = file.getPath();
	String outputEncoding = props.getProperty(propsPrefix +
		"outputEncoding", encoding);

	/*
	 * XXX This doesn't belong here.  The DefaultFileHandler should
	 * be used instead.  This allows "simple" configurations using only
	 * this handler.
	 */

	if (file.isDirectory()) {
	    if (request.url.endsWith("/") == false) {
		request.redirect(request.url + "/", null);
		return true;
	    }
	    props.put("DirectoryName", path);
	    String index = props.getProperty(propsPrefix+DEFAULT, "index.html");
	    file = new File(file, index);
	    path = file.getPath();
	}
	props.put("fileName", path);
	request.log(Server.LOG_DIAGNOSTIC, propsPrefix,
		"Looking for template file: " + path);

	int index = path.lastIndexOf('.');
	if (index < 0) {
	    request.log(Server.LOG_INFORMATIONAL, propsPrefix,
		    "no file suffix for: " + path);
	    return false;
	}

	String suffix = path.substring(index);
	String type = props.getProperty(propsPrefix + "mime" + suffix,
		props.getProperty("mime" + suffix));
	if (type == null) {
	    request.log(Server.LOG_INFORMATIONAL, propsPrefix,
		    "unknown file suffix: " + suffix);
	    return false;
	}

	// make sure this is a text file!

	if (!type.toLowerCase().startsWith("text/")) {
	    request.log(Server.LOG_INFORMATIONAL, propsPrefix,
		    "Not a text file: " + type);
	    return false;
	}

	// OK, now we can read in the file!

	String content;
	try {
	    content =  getContent(request, file, encoding);
	} catch (IOException e) {
	    request.log(Server.LOG_WARNING, propsPrefix, e.getMessage());
	    return false;
	}

        String session = props.getProperty(sessionProperty, "common");
	request.log(Server.LOG_DIAGNOSTIC, propsPrefix, 
		"Using session (" + session + ")");
	String result = runner.process(request,	content, session);
	if (result != null) {
	    if (modified) {
	       request.addHeader("Last-Modified", HttpUtil.formatTime());
	    }
	    if (outputEncoding == null) {
	        request.sendResponse(result, type);
	    } else {
	        request.sendResponse(result.getBytes(outputEncoding),
			  type + "; charset=" + outputEncoding);
	    }
	    return true;
	} else {
	    request.log(Server.LOG_INFORMATIONAL,propsPrefix,runner.getError());
	    return false;
	}
    }

    /**
     * get the content associated with this template.
     * This version reads it from a file.
     *
     * @param request	The standard request object
     * @param file	The file object to get the template from
     * @return		The content of the template to be processed
     */

    public String
    getContent(Request request, File file, String encoding) throws IOException {
	FileInputStream in = new FileInputStream(file);
	byte[] buf = new byte[in.available()];
	in.read(buf);
	in.close();
	String result;
	if (encoding != null) {
	    result = new String(buf, encoding);
	} else {
	    result = new String(buf);
	}
	return result;
    }
}
