#!BPY
"""
Name: 'Curveaceous Galore'
Blender: 248
Group: 'Wizards'
Tip: 'Create curves in many forms'
"""
__author__ = "Jimmy Hazevoet"
__version__ = "v.1.0 February 2007"
__bpydoc__ = """\

Curveaceous galore:

This script helps you to create curves in many forms.

"""
# --------------------------------------------------------------------------
# ***** BEGIN GPL LICENSE BLOCK *****
#
# Copyright (C) Jimmy Hazevoet
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software Foundation,
# Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#
# ***** END GPL LICENCE BLOCK *****
# --------------------------------------------------------------------------

# Alt-P to run script
#--------------------------------------------------------------------------
# CurveaceousGalore v.1.0, Jimmy Hazevoet February 2007
#--------------------------------------------------------------------------


#
##------------------------------------------------------------------------------------------------------------------------
###------------------------------------------------------------------------------------------------------------------------
####------------------------------------------------------------------------------------------------------------------------
##### Curveaceous galore
####------------------------------------------------------------------------------------------------------------------------
###------------------------------------------------------------------------------------------------------------------------
##------------------------------------------------------------------------------------------------------------------------
#


import Blender
from Blender import *
from Blender.Draw import *
from Blender.BGL import *
import string
from string import strip
from math import *
import sys




#------------------------------------------------------------
##------------------------------------------------------------
###------------------------------------------------------------
#### Some functions to use with others:
###------------------------------------------------------------
##------------------------------------------------------------
#------------------------------------------------------------



#------------------------------------------------------------
# Generate random number:
def randnum( low=0.0, high=1.0, seed=0 ):
	"""
	randnum( low=0.0, high=1.0, seed=0 )
	
	Create random number
	
		Parameters:
			low - lower range
				(type=float)
			high - higher range
				(type=float)
			seed - the random seed number, if seed is 0, the current time will be used instead
				(type=int)
		Returns:
			a random number
				(type=float)
	"""

	s = Noise.setRandomSeed( seed )
	rnum = Noise.random()
	rnum = rnum*(high-low)
	rnum = rnum+low
	return rnum



#------------------------------------------------------------
# Make some noise:
def vTurbNoise((x,y,z), iScale=0.25, Size=1.0, Depth=6, Hard=0, Basis=0, Seed=0 ):
	"""
	vTurbNoise((x,y,z), iScale=0.25, Size=1.0, Depth=6, Hard=0, Basis=0, Seed=0 )
	
	Create randomised vTurbulence noise
	
		Parameters:
			xyz - (x,y,z) float values.
				(type=3-float tuple)
			iScale - noise intensity scale
				(type=float)
			Size - noise size
				(type=float)
			Depth - number of noise values added.
				(type=int)
			Hard - noise hardness: 0 - soft noise; 1 - hard noise
				(type=int)
			basis - type of noise used for turbulence
				(type=int)
			Seed - the random seed number, if seed is 0, the current time will be used instead
				(type=int)
		Returns:
			the generated turbulence vector.
				(type=3-float list)
	"""

	rand = randnum(-100,100,Seed)
	if Basis ==9: Basis = 14
	vTurb = Noise.vTurbulence(( x/Size+rand, y/Size+rand, z/Size+rand ), Depth, Hard, Basis )
	tx = vTurb[0]*iScale
	ty = vTurb[1]*iScale
	tz = vTurb[2]*iScale
	return tx,ty,tz



#------------------------------------------------------------
# Axis: ( used in 3DCurve Turbulence )
def AxisFlip((x,y,z), x_axis=1, y_axis=1, z_axis=1, flip=0 ):
	if flip != 0:
		flip *= -1
	else: flip = 1
	x *= x_axis*flip
	y *= y_axis*flip
	z *= z_axis*flip
	return x,y,z



#------------------------------------------------------------
# ...:




#------------------------------------------------------------
##------------------------------------------------------------
###------------------------------------------------------------------------------------------------------------------------
#### 2D Curve shape functions:
###------------------------------------------------------------------------------------------------------------------------
##------------------------------------------------------------
#------------------------------------------------------------



##------------------------------------------------------------
# 2DCurve: Profile:  L, H, T, U, Z
def ProfileCurve( type=0, a=0.25, b=0.25 ):
	"""
	ProfileCurve( type=0, a=0.25, b=0.25 )
	
	Create profile curve 
		
		Parameters:
			type - select profile type, L, H, T, U, Z
				(type=int)
			a - a scaling parameter
				(type=float)
			b - b scaling parameter
				(type=float)
		Returns:
			a list with lists of x,y,z coordinates for curve points, [[x,y,z],[x,y,z],...n]
			(type=list)
	"""

	newpoints = []
	if type ==1:
		## H:
		a*=0.5
		b*=0.5
		newpoints = [ [ -1.0, 1.0, 0.0 ], [ -1.0+a, 1.0, 0.0 ], [ -1.0+a, b, 0.0 ], [ 1.0-a, b, 0.0 ], [ 1.0-a, 1.0, 0.0 ], [ 1.0, 1.0, 0.0 ],
						[ 1.0, -1.0, 0.0 ], [ 1.0-a, -1.0, 0.0 ], [ 1.0-a, -b, 0.0 ], [ -1.0+a, -b, 0.0 ], [ -1.0+a, -1.0, 0.0 ], [ -1.0, -1.0, 0.0 ] ]
	elif type ==2:
		## T:
		a*=0.5
		newpoints = [ [ -1.0, 1.0, 0.0 ], [ 1.0, 1.0, 0.0 ], [ 1.0, 1.0-b, 0.0 ], [ a, 1.0-b, 0.0 ], [ a, -1.0, 0.0 ], [ -a, -1.0, 0.0 ], [ -a, 1.0-b, 0.0 ], [ -1.0, 1.0-b, 0.0 ] ]
	elif type ==3:
		## U:
		a*=0.5
		newpoints = [ [ -1.0, 1.0, 0.0 ], [ -1.0+a, 1.0, 0.0 ], [ -1.0+a, -1.0+b, 0.0 ], [ 1.0-a, -1.0+b, 0.0 ], [ 1.0-a, 1.0, 0.0 ], [ 1.0, 1.0, 0.0 ], [ 1.0, -1.0, 0.0 ], [ -1.0, -1.0, 0.0 ] ]
	elif type ==4:
		## Z:
		a*=0.5
		newpoints = [ [ -0.5, 1.0, 0.0 ], [ a, 1.0, 0.0 ], [ a, -1.0+b, 0.0 ], [ 1.0, -1.0+b, 0.0 ], [ 1.0, -1.0, 0.0 ], [ -a, -1.0, 0.0 ], [ -a, 1.0-b, 0.0 ], [ -1.0, 1.0-b, 0.0 ], [ -1.0, 1.0, 0.0 ] ]
	else:
		## L:
		newpoints = [ [ -1.0, 1.0, 0.0 ], [ -1.0+a, 1.0, 0.0 ], [ -1.0+a, -1.0+b, 0.0 ], [ 1.0, -1.0+b, 0.0 ], [ 1.0, -1.0, 0.0 ], [ -1.0, -1.0, 0.0 ] ]
	return newpoints



##------------------------------------------------------------
# 2DCurve: Flower:
def FlowerCurve( petals=8, innerradius=0.5, outerradius=1.0, petalwidth=2.0 ):
	"""
	FlowerCurve( petals=8, innerradius=0.5, outerradius=1.0, petalwidth=2.0 )
	
	Create flower shaped curve
	
		Parameters:
			petals - the number of petals
				(type=int)
			innerradius - innerradius
				(type=float)
			outerradius - outerradius
				(type=float)
			petalwidth - width of petals
				(type=float)
		Returns:
			a list with lists of x,y,z coordinates for curve points, [[x,y,z],[x,y,z],...n]
			(type=list)
	"""

	newpoints = []
	step = (2.0/(petals))
	pet = (step/pi*2)*petalwidth
	i = 0
	while i < petals:
		t = (i*step)
		x1 = cos(t*pi-(pi/petals))*innerradius
		y1 = sin(t*pi-(pi/petals))*innerradius
		newpoints.append([x1,y1,0])
		x2 = cos(t*pi-pet)*outerradius
		y2 = sin(t*pi-pet)*outerradius
		newpoints.append([x2,y2,0])
		x3 = cos(t*pi+pet)*outerradius
		y3 = sin(t*pi+pet)*outerradius
		newpoints.append([x3,y3,0])
		i+=1
	return newpoints



##------------------------------------------------------------
# 2DCurve: Star:
def StarCurve( starpoints=8, innerradius=0.5, outerradius=1.0, twist=0.0 ):
	"""
	StarCurve( starpoints=8, innerradius=0.5, outerradius=1.0, twist=0.0 )
	
	Create star shaped curve
	
		Parameters:
			starpoints - the number of points
				(type=int)
			innerradius - innerradius
				(type=float)
			outerradius - outerradius
				(type=float)
			twist - twist amount
				(type=float)
		Returns:
			a list with lists of x,y,z coordinates for curve points, [[x,y,z],[x,y,z],...n]
			(type=list)
	"""

	newpoints = []
	step = (2.0/(starpoints))
	i = 0
	while i < starpoints:
		t = (i*step)
		x1 = cos(t*pi)*outerradius
		y1 = sin(t*pi)*outerradius
		newpoints.append([x1,y1,0])
		x2 = cos(t*pi+(pi/starpoints+twist))*innerradius
		y2 = sin(t*pi+(pi/starpoints+twist))*innerradius
		newpoints.append([x2,y2,0])
		i+=1
	return newpoints



##------------------------------------------------------------
# 2DCurve: Miscellaneous.: Diamond, Arrow1, Arrow2, Square, ....
def MiscCurve( type=1, a=1.0, b=0.5, c=90.0 ):
	"""
	MiscCurve( type=1, a=1.0, b=0.5, c=90.0 )
	
	Create miscellaneous curves
	
		Parameters:
			type - select type, Diamond, Arrow1, Arrow2, Square
				(type=int)
			a - a scaling parameter
				(type=float)
			b - b scaling parameter
				(type=float)
			c - c scaling parameter
				(type=float)
		Returns:
			a list with lists of x,y,z coordinates for curve points, [[x,y,z],[x,y,z],...n]
			(type=list)
	"""

	newpoints = []
	a*=0.5
	b*=0.5
	if type ==1:
		## diamond:
		newpoints = [ [ 0.0, b, 0.0 ], [ a, 0.0, 0.0 ], [ 0.0, -b, 0.0 ], [ -a, 0.0, 0.0 ]  ]
	elif type ==2:
		## Arrow1:
		newpoints = [ [ -a, b, 0.0 ], [ a, 0.0, 0.0 ], [ -a, -b, 0.0 ], [ 0.0, 0.0, 0.0 ]  ]
	elif type ==3:
		## Arrow2:
		newpoints = [ [ -1.0, b, 0.0 ], [ -1.0+a, b, 0.0 ], [ -1.0+a, 1.0, 0.0 ], [ 1.0, 0.0, 0.0 ], [ -1.0+a, -1.0, 0.0 ], [ -1.0+a, -b, 0.0 ], [ -1.0, -b, 0.0 ] ]
	elif type ==4:
		## Rounded square:
		newpoints = [ [ -a, b-b*0.2, 0.0 ], [ -a+a*0.05, b-b*0.05, 0.0 ], [ -a+a*0.2, b, 0.0 ],
							[ a-a*0.2, b, 0.0 ], [ a-a*0.05, b-b*0.05, 0.0 ], [ a, b-b*0.2, 0.0 ],
							[ a, -b+b*0.2, 0.0 ], [ a-a*0.05, -b+b*0.05, 0.0 ], [ a-a*0.2, -b, 0.0 ],
							[ -a+a*0.2, -b, 0.0 ], [ -a+a*0.05, -b+b*0.05, 0.0 ], [ -a, -b+b*0.2, 0.0 ] ]

	#elif type ==15:
		## :
		#newpoints = [ [ x,y,z ] ]
	else:
		## Square:
		newpoints = [ [ -a, b, 0.0 ], [ a, b, 0.0 ], [ a, -b, 0.0 ], [ -a, -b, 0.0 ]  ]
	return newpoints



##------------------------------------------------------------
# 2DCurve: Arc,Sector,Segment,Ring:
def ArcCurve( sides=6, startangle=0.0, endangle=90.0, innerradius=0.5, outerradius=1.0, type=3 ):
	"""
	ArcCurve( sides=6, startangle=0.0, endangle=90.0, innerradius=0.5, outerradius=1.0, type=3 )
	
	Create arc shaped curve
	
		Parameters:
			sides - number of sides
				(type=int)
			startangle - startangle
				(type=float)
			endangle - endangle
				(type=float)
			innerradius - innerradius
				(type=float)
			outerradius - outerradius
				(type=float)
			type - select type Arc,Sector,Segment,Ring
				(type=int)
		Returns:
			a list with lists of x,y,z coordinates for curve points, [[x,y,z],[x,y,z],...n]
			(type=list)
	"""

	newpoints = []
	sides += 1
	angle = (2.0*(1.0/360.0))
	endangle-=startangle
	step = ((angle*endangle)/(sides-1))
	i = 0
	while i < sides:
		t = (i*step) + angle*startangle
		x1 = sin(t*pi)*outerradius
		y1 = cos(t*pi)*outerradius
		newpoints.append([x1,y1,0])
		i+=1

	#if type ==0:
		# Arc: turn cyclic curve flag off!

	# Segment:
	if type ==2:
		newpoints.append([0,0,0])
	# Ring:
	elif type ==3:
		j=sides-1
		while j > -1:
			t = (j*step) + angle*startangle
			x2 = sin(t*pi)*innerradius
			y2 = cos(t*pi)*innerradius
			newpoints.append([x2,y2,0])
			j-=1
	return newpoints



##------------------------------------------------------------
# 2DCurve: Cog wheel: 
def CogCurve( theeth=8, innerradius=0.8, middleradius=0.95, outerradius=1.0, bevel=0.5 ):
	"""
	CogCurve( theeth=8, innerradius=0.8, middleradius=0.95, outerradius=1.0, bevel=0.5 )
	
	Create cog wheel shaped curve
	
		Parameters:
			theeth - number of theeth
				(type=int)
			innerradius - innerradius
				(type=float)
			middleradius - middleradius
				(type=float)
			outerradius - outerradius
				(type=float)
			bevel - bevel amount
				(type=float)
		Returns:
			a list with lists of x,y,z coordinates for curve points, [[x,y,z],[x,y,z],...n]
			(type=list)
	"""

	newpoints = []
	step = (2.0/(theeth))
	pet = (step/pi*2)
	bevel = 1.0-bevel
	i = 0
	while i < theeth:
		t = (i*step)
		x1 = cos(t*pi-(pi/theeth)-pet)*innerradius
		y1 = sin(t*pi-(pi/theeth)-pet)*innerradius
		newpoints.append([x1,y1,0])
		x2 = cos(t*pi-(pi/theeth)+pet)*innerradius
		y2 = sin(t*pi-(pi/theeth)+pet)*innerradius
		newpoints.append([x2,y2,0])
		x3 = cos(t*pi-pet)*middleradius
		y3 = sin(t*pi-pet)*middleradius
		newpoints.append([x3,y3,0])
		x4 = cos(t*pi-(pet*bevel))*outerradius
		y4 = sin(t*pi-(pet*bevel))*outerradius
		newpoints.append([x4,y4,0])
		x5 = cos(t*pi+(pet*bevel))*outerradius
		y5 = sin(t*pi+(pet*bevel))*outerradius
		newpoints.append([x5,y5,0])
		x6 = cos(t*pi+pet)*middleradius
		y6 = sin(t*pi+pet)*middleradius
		newpoints.append([x6,y6,0])
		i+=1
	return newpoints



##------------------------------------------------------------
# 2DCurve: nSide:
def nSideCurve( sides=6, radius=1.0 ):
	"""
	nSideCurve( sides=6, radius=1.0 )
	
	Create n-sided curve
	
		Parameters:
			sides - number of sides
				(type=int)
			radius - radius
				(type=float)
		Returns:
			a list with lists of x,y,z coordinates for curve points, [[x,y,z],[x,y,z],...n]
			(type=list)
	"""

	newpoints = []
	step = (2.0/(sides))
	i = 0
	while i < sides:
		t = (i*step)
		x = sin(t*pi)*radius
		y = cos(t*pi)*radius
		newpoints.append([x,y,0])
		i+=1
	return newpoints



##------------------------------------------------------------
# 2DCurve: Splat:
def SplatCurve( sides=24, scale=1.0, seed=0, basis=0, radius=1.0 ):
	"""
	SplatCurve( sides=24, scale=1.0, seed=0, basis=0, radius=1.0 )
	
	Create splat curve
	
		Parameters:
			sides - number of sides
				(type=int)
			scale - noise size
				(type=float)
			seed - noise random seed
				(type=int)
			basis - noise basis
				(type=int)
			radius - radius
				(type=float)
		Returns:
			a list with lists of x,y,z coordinates for curve points, [[x,y,z],[x,y,z],...n]
			(type=list)
	"""

	newpoints = []
	step = (2.0/(sides))
	i = 0
	while i < sides:
		t = (i*step)
		turb = vTurbNoise((t,t,t), 1.0, scale, 6, 0, basis, seed )
		turb = turb[2] * 0.5 + 0.5
		x = sin(t*pi)*radius * turb
		y = cos(t*pi)*radius * turb
		newpoints.append([x,y,0])
		i+=1
	return newpoints



##------------------------------------------------------------
# 2DCurve: Other2D


##------------------------------------------------------------
#



#------------------------------------------------------------
##------------------------------------------------------------
###------------------------------------------------------------------------------------------------------------------------
#### 3D curve shape functions:
###------------------------------------------------------------------------------------------------------------------------
##------------------------------------------------------------
#------------------------------------------------------------



###------------------------------------------------------------
# 3DCurve: Formula:
def FormulaCurve( number=24, length=2.0, input=( 't', '0.0', 'sin(t*pi)' ) ):
	'''
	FormulaCurve( number=24, length=2.0, input=( 't', '0.0', 'sin(t*pi)' ) )
	
	Create Formula Curve
	
	( x(t), y(t), z(t) )
	Enter a mathematical function,
	with dependant variable t,
	for each of these directions.
	
	Example:
	x(t) = 'sin(t*pi*2.0/l)'
	y(t) = 'cos(t*pi*2.0/l)'
	z(t) = 't'
	
		Parameters:
			number - the number of points
				(type=int)
			length - length of curve
				(type=float)
			input - formula input, 3 strings, x(t), y(t), z(t)
				(type=string)
		Returns:
			a list with lists of x,y,z coordinates for curve points, [[x,y,z],[x,y,z],...n]
			(type=list)
	'''

	newpoints = []
	step = (length/(number-1))
	n  = number
	l  = length
	x,y,z=0.0,0.0,0.0
	i = 0
	while i < n:
		t = (i*step)
		x = eval( input[0] )
		y = eval( input[1] )
		z = eval( input[2] )
		newpoints.append([x,y,z])
		i+=1
	return newpoints



###------------------------------------------------------------
# 3DCurve: Helix:
def HelixCurve( number=100, height=2.0, startangle=0.0, endangle=360.0, width=1.0, a=0.0, b=0.0 ):
	"""
	HelixCurve( number=100, height=2.0, startangle=0.0, endangle=360.0, width=1.0, a=0.0, b=0.0 )
	
	Create helix curve
	
		Parameters:
			number - the number of points
				(type=int)
			height - height
				(type=float)
			startangle - startangle
				(type=float)
			endangle - endangle
				(type=float)
			width - width
				(type=float)
			a - a
				(type=float)
			b - b
				(type=float)
		Returns:
			a list with lists of x,y,z coordinates for curve points, [[x,y,z],[x,y,z],...n]
			(type=list)
	"""

	newpoints = []
	angle = (2.0/360.0)*(endangle-startangle)
	step = angle/(number-1)
	h = height/angle	
	start = (startangle*2.0/360.0)
	a/=angle
	i = 0
	while i < number:
		t = ( i*step+start )
		x = sin( (t*pi) ) * ( 1.0 + cos( t * pi * a - ( b * pi ) ) ) * ( 0.25 * width )
		y = cos( (t*pi) ) * ( 1.0 + cos( t * pi * a - ( b * pi ) ) ) * ( 0.25 * width )
		z = ( t * h ) -h*start
		newpoints.append([x,y,z])
		i+=1
	return newpoints



###------------------------------------------------------------
# 3DCurve: Turbulence: tnoise=[ iScale, Size, Depth, Hard, Basis, Seed ], axis=[x,y,z]
def TurbulenceCurve( number=24, length=2.0, tnoise=[0.25,1.0,6,0,0,0], axis=[1,1,1,0] ):
	"""
	TurbulenceCurve( number=24, length=2.0, tnoise=[0.25,1.0,6,0,0,0], axis=[1,1,1,0] )
	
	Create turbulence curve
	
		Parameters:
			number - the number of points
				(type=int)
			length - length of curve
				(type=float)
			tnoise - noise settings: Size, Depth, Hard, Basis, Seed
				(type=list)
			axis - enable/disable, or flip axis
				(type=list)
		Returns:
			a list with lists of x,y,z coordinates for curve points, [[x,y,z],[x,y,z],...n]
			(type=list)
	"""
	
	newpoints = []
	step = (length/(number-1))
	i = 0
	while i < number:
		t = (i*step)
		x,y,z = AxisFlip((t,t,t), axis[0], axis[1], axis[2], axis[3] )

		turb = vTurbNoise((x,y,z), tnoise[0], tnoise[1], tnoise[2], tnoise[3], tnoise[4], tnoise[5] )
		if i==0:
			newpoints.append([x,y,z])
		else:
			newpoints.append([x+turb[0], y+turb[1], z+turb[2] ])			
		i+=1
	return newpoints




###------------------------------------------------------------ ?
# 3DCurve: Cycloid: Cycloid, Epicycloid, Hypocycloid
def CycloidCurve( number=24, length=2.0, type=0, a=1.0, b=1.0, startangle=0.0, endangle=360.0 ):
	"""
	CycloidCurve( number=24, length=2.0, type=0, a=1.0, b=1.0, startangle=0.0, endangle=360.0 )
	
	Create a Cycloid, Epicycloid or Hypocycloid curve

		Parameters:
			number - the number of points
				(type=int)
			length - length of curve
				(type=float)
			type - types: Cycloid, Epicycloid, Hypocycloid
				(type=int)
		Returns:
			a list with lists of x,y,z coordinates for curve points, [[x,y,z],[x,y,z],...n]
			(type=list)
	"""

	newpoints = [[-1,0,0], [0,0,1], [1,0,0] ]
	return newpoints



###------------------------------------------------------------ ?
# 3DCurve: Cissoid:  Cissoid, Lemniscate, Strophoid
def CissoidCurve( number=24, length=2.0, type=0,  tension=2.0 ):
	"""
	CissoidCurve( number=24, length=2.0, type=0,  tension=2.0 )
	
	Create a Cissoid, Lemniscate or Strophoid curve
	
		Parameters:
			number - the number of points
				(type=int)
			length - length of curve
				(type=float)
			type - types: Cissoid, Lemniscate, Strophoid
				(type=int)
			tension - tension
				(type=float)
		Returns:
			a list with lists of x,y,z coordinates for curve points, [[x,y,z],[x,y,z],...n]
			(type=list)
	"""

	newpoints = [[0,0,0], [0,0,1], [0,0,2] ]
	return newpoints



###------------------------------------------------------------
# 3DCurve: Other3D



###------------------------------------------------------------
#



#------------------------------------------------------------------------------------------------------------------------
##------------------------------------------------------------------------------------------------------------------------
###------------------------------------------------------------------------------------------------------------------------
#### Make Curve from a list with lists of point x,y,z coordinates: [[x,y,z],[x,y,z],...]
###------------------------------------------------------------------------------------------------------------------------
##------------------------------------------------------------------------------------------------------------------------
#------------------------------------------------------------------------------------------------------------------------



####--------------------
# Make Curve from points:

def make_curve( curvepoints, curvename='Curve', flags=[0,1,0], curvesettings=[ 1.0, 0.0 , 0.0, 0, '','' ] ):
	"""
	make_curve( curvepoints, curvename='Curve', flags=[0,1,0], curvesettings=[ 1.0, 0.0 , 0.0, 0, '','' ] )
	
	Make Curve from a list with lists of points: [[x,y,z],[x,y,z],...]

	Example (triangle):
	curvepoints = [ [ 0.0, 1.0, 0.0 ], [ 1.0, -1.0, 0.0 ], [ -1.0, -1.0, 0.0 ]  ]
	
		Parameters:
			curvepoints - a list with lists of curve point coordinates
				(type=list)
			curvename - the new curve name
				(type=string)
		
			flags -  [ CurveType(Poly,Bez,Nurb), Cyclic, 3D ]

			curvesettings - [ Width, Extrude, BevDepth, BevRes, BevObName, TapObName  ]

		Returns:
			The generated curve object
	"""


	scn = Scene.GetCurrent()

	#--------------------
	# Deselect all other objects:
	#for obj in Object.GetSelected():
	#	obj.select( 0 )
	scn.objects.selected = []

	#--------------------
	# Make New Curve:
	NewCurve = Curve.New( 'Curve' )
	#CurveOb = Object.New( 'Curve', curvename )
	#CurveOb.link( NewCurve )
	#scn.link( CurveOb )
	CurveOb = scn.objects.new( NewCurve )

	#--------------------
	i = 0
	for p in curvepoints:
		#--------------------
		if i == 0:
			NewCurve.appendNurb( [ p[0], p[1], p[2], 1 ] )
			NewCurve[0].type = 4 # 0=poly, 1=Bezier, 4=nurb
		else:
			NewCurve[0].append( [ p[0], p[1], p[2], 1 ] )
		#--------------------
		i+=1

	#--------------------
	# set Flags:
	if curvesettings[4] !='':
		bob = Object.Get( curvesettings[4] ) # bevel object
		NewCurve.setBevOb(bob)
	if curvesettings[5] !='':
		tap = Object.Get( curvesettings[5] ) # taper object
		NewCurve.setTaperOb(tap)
	NewCurve[0].setFlagU( flags[1] )  # curve cyclic, 2=off, 1=on
	NewCurve[0].setFlagV( flags[1] )  # curve cyclic, 2=off, 1=on
	NewCurve[0].setType(  flags[0] )  # curve type, 0=poly, 1=Bezier, 4=nurb
	if flags[2] ==0: flags[2] = 6
	else: flags[2] = 1
	NewCurve.setFlag( flags[2] )      # 3D Flag: 6=Back/Front, 1=3D
	NewCurve.setWidth(    curvesettings[0] )  # Width
	NewCurve.setExt1(     curvesettings[1] )  # Extrude
	NewCurve.setExt2(     curvesettings[2] )  # Bevel depth
	NewCurve.setBevresol( curvesettings[3] )  # Bevel resolution
	#--------------------
	NewCurve.update()
	CurveOb.sel = 1
	CurveOb.setLocation( Window.GetCursorPos() )
	CurveOb.layers = Window.ViewLayers()
	#--------------------
	#Blender.Redraw(-1)
	Window.RedrawAll()
	#--------------------






#
##------------------------------------------------------------------------------------------------------------------------
###------------------------------------------------------------------------------------------------------------------------
####------------------------------------------------------------------------------------------------------------------------
##### Initialise values and Buttons:
####------------------------------------------------------------------------------------------------------------------------
###------------------------------------------------------------------------------------------------------------------------
##------------------------------------------------------------------------------------------------------------------------
#






#--------------------
# events:
Btn_Evt = 11
Shape_Evt = 3
Upd_Evt = 19
End_Evt = 73
No_Evt = 23
Col_Evt= 59
SaveF_Evt=321
LoadF_Evt=123

#--------------------
noisebasismenu = "Noise Basis %t|Blender Original%x0|Original Perlin%x1|Improved Perlin%x2|Voronoi_F1%x3|Voronoi_F2%x4|Voronoi_F3%x5|Voronoi_F4%x6|Voronoi_F2-F1%x7|Voronoi Crackle%x8|CellNoise%x9"

#--------------------
# 2D / 3D Mode:
ShapeMode = Create(0)

#--------------------
# Curve settings:

# [ ToggleG.U.I., Width, Extrude, BevDepth, BevRes, BevObName, TapObName  ]
CurveButs  = [ Create(0),  Create(1.0),  Create(0.0), Create(0.0), Create(0), Create(''), Create('') ]

# Flags = [ CurveType, Cyclic, 3D ]
CurveFlags = [ Create(0),  Create(1), Create(0) ]


#------------------------------------------------------------
##------------------------------------------------------------
###------------------------------------------------------------------------------------------------------------------------
####  2D:
###------------------------------------------------------------------------------------------------------------------------
##------------------------------------------------------------
#------------------------------------------------------------


##--------------------
# Buttons 2D:

ShapeMenu = Create(4)

ProfButs  = [ Create(0),  Create(1.0), Create(0.5) ]
FlowerButs= [ Create(10), Create(0.25), Create(1.0),  Create(1.5) ]
StarButs  = [ Create(5),  Create(0.375), Create(1.0),  Create(0.0) ]
MiscButs  = [ Create(0),  Create(2.0), Create(1.0),  Create(1.0) ]
ArcButs   = [ Create(18), Create(120.0), Create(400.0), Create(0.5), Create(1.0), Create(3) ]
CogButs   = [ Create(10), Create(0.8), Create(0.95), Create(1.0), Create(0.5) ]
nSideButs = [ Create(6),  Create(1.0) ]
SplatButs = [ Create(32), Create(0.5), Create(1), Create(0) ]

## OtherButs2D = Create(8)



##------------------------------------------------------------
## Update 2D:

##--------------------
# Make shape dictionary, and update shape values:
def Shape2DUpdate():
	global ProfButs, FlowerButs, StarButs, MiscButs, ArcButs, CogButs, nSideButs, SplatButs

	shapedict2D = {}
	shapedict2D = {
		'Profile':    ProfileCurve(   ProfButs[0].val,   ProfButs[1].val,   ProfButs[2].val ),
		'Flower':     FlowerCurve(    FlowerButs[0].val, FlowerButs[1].val, FlowerButs[2].val, FlowerButs[3].val ),
		'Star':       StarCurve(      StarButs[0].val,   StarButs[1].val,   StarButs[2].val,   StarButs[3].val ),
		'Misc':       MiscCurve(      MiscButs[0].val,   MiscButs[1].val,   MiscButs[2].val,   MiscButs[3].val ),
		'Arc':        ArcCurve(       ArcButs[0].val,    ArcButs[1].val,    ArcButs[2].val,    ArcButs[3].val, ArcButs[4].val, ArcButs[5].val ),
		'Cog wheel':  CogCurve(       CogButs[0].val,    CogButs[1].val,    CogButs[2].val,    CogButs[3].val, CogButs[4].val ),
		'n-Side':     nSideCurve(     nSideButs[0].val,  nSideButs[1].val ),
		'Splat':      SplatCurve(     SplatButs[0].val,  SplatButs[1].val,  SplatButs[2].val,  SplatButs[3].val, 1.0 ) #,
		##'Other2D':      [[-1.0,0.0,0.0], [0.0,1.0,0.0], [0.0,-1.0,0.0] ]
		}
	return shapedict2D



#------------------------------------------------------------
##------------------------------------------------------------
###------------------------------------------------------------------------------------------------------------------------
#### 3D:
###------------------------------------------------------------------------------------------------------------------------
##------------------------------------------------------------
#------------------------------------------------------------


###--------------------
# Buttons 3D:

ShapeMenu3D = Create(0)

NoiseButs   = [ Create(0.25), Create(1.0), Create(6), Create(0), Create(0), Create(0) ]
AxisButs    = [ Create(1), Create(0), Create(0), Create(0) ]

FormulaButs = [ Create(24),  Create(2.0), Create( 't' ), Create( '0.0' ), Create( 'sin(t*pi)' ) ]
HelixButs   = [ Create(100), Create(2.0), Create(0.0), Create(720.0), Create(2.0), Create(0.0), Create(0.0) ]
TurbButs    = [ Create(24),  Create(2.0), Create(1.0), Create(0.0) ]

#CycloidButs = [ Create(24),  Create(2.0), Create(0), Create(0.5), Create(0.75), Create(0.0), Create(360.0) ]
#CissoidButs = [ Create(24),  Create(2.0), Create(0), Create(2) ]

### OtherButs3D = Create(24)


###------------------------------------------------------------
### Update 3D:

###--------------------
# Make shape dictionary, and update shape values:
def Shape3DUpdate():
	global ShapeMenu3D, HelixButs, TurbButs, NoiseButs, AxisButs, CycloidButs, CissoidButs

	shapedict3D = {}
	shapedict3D = {
		'Formula':    FormulaCurve(    FormulaButs[0].val, FormulaButs[1].val, ( FormulaButs[2].val, FormulaButs[3].val, FormulaButs[4].val ) ),
		'Helix':      HelixCurve(      HelixButs[0].val,   HelixButs[1].val, HelixButs[2].val, HelixButs[3].val, HelixButs[4].val, HelixButs[5].val, HelixButs[6].val ),
		'Turbulence': TurbulenceCurve( TurbButs[0].val,    TurbButs[1].val, [ NoiseButs[0].val, NoiseButs[1].val, NoiseButs[2].val, NoiseButs[3].val, NoiseButs[4].val, NoiseButs[5].val ], [ AxisButs[0].val, AxisButs[1].val,AxisButs[2].val,AxisButs[3].val ] ) #,
		}
	return shapedict3D

		#'Cycloid':    CycloidCurve(    CycloidButs[0].val, CycloidButs[1].val, CycloidButs[2].val, CycloidButs[3].val, CycloidButs[4].val, CycloidButs[5].val, CycloidButs[6].val ),
		#'Cissoid':    CissoidCurve(    CissoidButs[0].val, CissoidButs[1].val, CissoidButs[2].val, CissoidButs[3].val ) #,

		#'Other3D':    [[0.0,0.0,0.0], [0.0,0.0,1.0]]




#------------------------------------------------------------
##------------------------------------------------------------
###------------------------------------------------------------------------------------------------------------------------
#### Initialise Menu's and points at script start-up:
###------------------------------------------------------------------------------------------------------------------------
##------------------------------------------------------------
#------------------------------------------------------------

##----------------------------------------
# Initialise 2D Shapes Menu at script startup:
shapedict2DStartup = Shape2DUpdate()
s2Dkeys = shapedict2DStartup.keys()

shape2Dtypesmenu = "2D Curve Shapes %t|"
i = 0
for key in s2Dkeys:
	shape2Dtypesmenu += '|' + key + '%x' + str(i)
	i += 1

## Initialise Shape points at startup:
points = shapedict2DStartup[ s2Dkeys[ ShapeMenu.val ] ]

##print '\n', shapedict2DStartup,'\n', shape2Dtypesmenu,'\n', points, '\n', s2Dkeys
del key, shapedict2DStartup

###----------------------------------------
# Initialise 3D Shapes Menu at script startup:
shapedict3DStartup = Shape3DUpdate()
s3Dkeys = shapedict3DStartup.keys()

shape3Dtypesmenu = "3D Curve Shapes %t|"
i = 0
for key in s3Dkeys:
	shape3Dtypesmenu += '|' + key + '%x' + str(i)
	i += 1

##print '\n', shapedict3DStartup,'\n', shape3Dtypesmenu,'\n', s3Dkeys
del shapedict3DStartup



#------------------------------------------------------------------------------------------------------------------------
##------------------------------------------------------------------------------------------------------------------------
###------------------------------------------------------------------------------------------------------------------------
#### G.U.I.:
###------------------------------------------------------------------------------------------------------------------------
##------------------------------------------------------------------------------------------------------------------------
#------------------------------------------------------------------------------------------------------------------------

# G.U.I. colours:
colourfull  = [ [1.0,0.5,0.0], [1.0,0.498,0.498], [1.0,0.698,0.498], [1.0,0.898,0.498], [0.898,1.0,0.498], [0.698,1.0,0.498], [0.498,1.0,0.498], [0.498,1.0,0.698], [0.498,1.0,0.898], [0.600,0.918,1.0], [0.6,0.757,1.0], [0.6,0.6,1.0], [0.757,0.6,1.0], [0.898,0.498,1.0], [1.0,0.498,0.898] ]
linecolours = colourfull[ 0 ] # G.U.I. 2D line colour: ( shuffle with: [SPACEKEY] )

background  = [  0.69,  0.69, 0.69, 1.0 ]
panelcolour = [ 0.74, 0.74, 0.74 ] 
black       = [  0.0,  0.0, 0.0 ]

#------------------------------------------------------------

# G.U.I. Draw 2D shape from curve points in G.U.I:
def draw_Line( linepoints, x=100, y=100, size=50, colour=[1.0,0.5,0.0] ):
	glColor3f(colour[0],colour[1],colour[2])
	glBegin(GL_LINE_LOOP)
	for lp in linepoints:
		glVertex2i( x+int(lp[0]*size), y+int(lp[1]*size) )
	glEnd()

# G.U.I. Panel:
def draw_Panel( x, y, w, h, colors ):
	glColor3f( colors[0], colors[1], colors[2] )
	glRecti( x,y, w,h )

# G.U.I. Rounded corners:
dcp = ArcCurve( sides=4, startangle=0.0, endangle=90.0, innerradius=0.25, outerradius=1.0, type=2 )
def draw_RoundCorner( x, y, size, xflip, yflip ):
	global dcp
	if xflip !=0: xflip*=-1
	else: xflip=1
	if yflip !=0: yflip*=-1
	else: yflip=1
	glBegin(GL_POLYGON)
	for cp in dcp:
		glVertex2i(x+int(cp[0]*size*xflip) , y+int(cp[1]*size*yflip) )
	glEnd()

# G.U.I. Back Panel:
def draw_BackPanel( text, x, y, w, h, colors ):
	glColor3f( colors[0]*0.8, colors[1]*0.8, colors[2]*0.8 )
	glRecti( x+20, h, w-20, h+20 )
	draw_RoundCorner( x+20, h, 20, 1, 0 )
	draw_RoundCorner( w-20, h, 20, 0, 0 )
	glColor3f( colors[0], colors[1], colors[2] )
	glRecti( x, y, w, h )
	glColor3f( colors[0]*1.2, colors[1]*1.2, colors[2]*1.2 )
	glRasterPos2d( x+10, h+6 )
	Text( text, 'small' )

# G.U.I. Text
def draw_Text( ( x, y ), text, color, size ):
	glColor3f( color[0],color[1],color[2] )
	glRasterPos2d(x,y)
	txtsize = 'small', 'normal', 'large'
	Text( text, txtsize[ size ] )


#------------------------------------------------------------
##------------------------------------------------------------
###------------------------------------------------------------
#### Draw G.U.I.:
###------------------------------------------------------------
##------------------------------------------------------------
#------------------------------------------------------------


##--------------------
# G.U.I.:
def drawgui():
	global ShapeMode, CurveButs, CurveFlags, ShapeMenu, ShapeMenu3D, points
	global ProfButs, FlowerButs, StarButs, MiscButs, ArcButs, CogButs, nSideButs, SplatButs
	global FormulaButs, HelixButs, TurbButs, NoiseButs, AxisButs, CycloidButs, CissoidButs
	global linecolours, s2Dkeys, s3Dkeys

	glClearColor( background[0], background[1], background[2], background[3] )
	glClear(GL_COLOR_BUFFER_BIT)

	draw_BackPanel( '    Curveaceous galore', 8, 8, 320, 160, panelcolour )

	PushButton("X",  End_Evt, 280, 136, 32, 16 , "Exit")
	PushButton("OK", Upd_Evt, 280,  16, 32, 112, "Generate new Curve")

	CurveButs[0] = Toggle("Curve", Btn_Evt, 16, 136, 56, 16, CurveButs[0].val, "Curve Settings" )

	if CurveButs[0].val !=0:

		#--------------------
		# Curve settings:

		CurveFlags[0] = Menu("Curve Type %t|Poly Curve %x0|Bezier Curve %x1|Nurb Curve %x4", Btn_Evt, 16, 136-96, 112, 16, CurveFlags[0].val, "Curve type" )
		CurveFlags[1] = Toggle("Cyclic",        Btn_Evt,  16,  136-72,  112, 16, CurveFlags[1].val,           "Draw cyclic (closed) curve" )
		CurveFlags[2] = Toggle("3D",            Btn_Evt,  16,  136-48,  112, 16, CurveFlags[2].val,           "Allow the curve to be 3D, it doesn't fill then" )
		CurveButs[1]  = Number("Width: ",       Btn_Evt, 140,     136,  128, 16, CurveButs[1].val,  0.0, 2.0, "Make interpolated result thinner or fatter" )
		CurveButs[2]  = Number("Extrude: ",     Btn_Evt, 140,  136-24,  128, 16, CurveButs[2].val,  0.0, 5.0, "Curve extrusion size when not using a bevel object" )
		CurveButs[3]  = Number("Bevel Depth: ", Btn_Evt, 140,  136-48,  128, 16, CurveButs[3].val,  0.0, 2.0, "Bevel depth when not using a bevel object" )
		CurveButs[4]  = Number("BevResol: ",    Btn_Evt, 140,  136-72,  128, 16, CurveButs[4].val,   0,  10,  "Bevel resolution when not using a bevel object" )
		CurveButs[5]  = String("BevOb: ",       Btn_Evt, 140,  136-96,  128, 16, CurveButs[5].val,  20,        "Curve object name that devines the bevel shape" )
		CurveButs[6]  = String("TaperOb: ",     Btn_Evt, 140, 136-120,  128, 16, CurveButs[6].val,  20,        "Curve object name that devines the taper (width)" )

		#------------------------------------------------------------

	else:

		#--------------------
		# Shape settings:

		#--------------------
		# Shape Mode: Toggle 2D / 3D Shapes:		
		ShapeMode = Toggle("2D / 3D", Shape_Evt, 72, 136, 56, 16, ShapeMode.val, "Toggle: 2D / 3D Curve Shapes" )

		if ShapeMode.val !=1: ## 2D:

			##--------------------
			# Draw 2D Shape In G.U.I.:

			draw_Panel( 16, 16, 128, 128, black )        # draw black panel
			draw_Line( points, 72, 72, 48, linecolours ) # draw 2D shape from points

			##--------------------
			# 2D Shape Buttons:

			ShapeMenu = Menu( shape2Dtypesmenu, Shape_Evt, 140, 136, 128, 16, ShapeMenu.val, "2D Curve Shapes" )

			if s2Dkeys[ ShapeMenu.val ] == 'Profile':
				ProfButs[0] = Number("Type: ", Shape_Evt, 140, 112,    128, 16, ProfButs[0].val,    0,  15, "Type" )
				ProfButs[1] = Number("A: ",    Shape_Evt, 140, 112-24, 128, 16, ProfButs[1].val,  0.0, 2.0, "" )
				ProfButs[2] = Number("B: ",    Shape_Evt, 140, 112-48, 128, 16, ProfButs[2].val,  0.0, 2.0, "" )

			elif s2Dkeys[ ShapeMenu.val ] == 'Flower':
				FlowerButs[0] = Number("Petals: ", Shape_Evt, 140, 112,    128, 16, FlowerButs[0].val,    3, 1000, "Number" )
				FlowerButs[1] = Number("Inner: ",  Shape_Evt, 140, 112-24, 128, 16, FlowerButs[1].val,  0.0, 1.0, "Inner radius" )
				FlowerButs[3] = Number("Width: ",  Shape_Evt, 140, 112-48, 128, 16, FlowerButs[3].val,  0.0, 2.5, "Petal width" )

			elif s2Dkeys[ ShapeMenu.val ] == 'Star':
				StarButs[0] = Number("Number: ", Shape_Evt, 140, 112,    128, 16, StarButs[0].val,    3, 1000, "Number" )
				StarButs[1] = Number("Inner: ",  Shape_Evt, 140, 112-24, 128, 16, StarButs[1].val,  0.0, 1.0, "Inner radius" )
				StarButs[3] = Number("Twist: ",  Shape_Evt, 140, 112-48, 128, 16, StarButs[3].val, -1.0, 1.0, "Twist" )

			elif s2Dkeys[ ShapeMenu.val ] == 'Misc':
				MiscButs[0] = Number("Type: ", Shape_Evt, 140, 112,    128, 16, MiscButs[0].val,    0, 15, "Type" )
				MiscButs[1] = Number("A: ",    Shape_Evt, 140, 112-24, 128, 16, MiscButs[1].val,  0.0, 2.0 )
				MiscButs[2] = Number("B: ",    Shape_Evt, 140, 112-48, 128, 16, MiscButs[2].val,  0.0, 2.0 )
				MiscButs[3] = Number("C: ",    Shape_Evt, 140, 112-72, 128, 16, MiscButs[3].val, -360.0, 360.0 )

			elif s2Dkeys[ ShapeMenu.val ] == 'Arc':
				ArcButs[0] = Number("Sides: ", Shape_Evt, 140, 112,    128, 16, ArcButs[0].val,    3, 1000, "Number" )
				ArcButs[1] = Number("Start: ", Shape_Evt, 140, 112-24, 128, 16, ArcButs[1].val,  -360.0+ArcButs[2].val, ArcButs[2].val, "Start angle" )
				ArcButs[2] = Number("End: ",   Shape_Evt, 140, 112-48, 128, 16, ArcButs[2].val,  ArcButs[1].val, 360.0+ArcButs[1].val, "End angle" )
				if ArcButs[5].val ==3:
					ArcButs[3] = Number("Inner: ", Shape_Evt, 140, 112-72, 128, 16, ArcButs[3].val,  0.0, 1.0, "Inner radius" )
				ArcButs[5] = Number("Type: ",     Shape_Evt, 140, 112-96, 128, 16, ArcButs[5].val,    0,   3, "Type" )

			elif s2Dkeys[ ShapeMenu.val ] == 'Cog wheel':
				CogButs[0] = Number("Theeth: ", Shape_Evt, 140, 112,    128, 16, CogButs[0].val,    3, 1000, "Number" )
				CogButs[1] = Number("Inner: ",  Shape_Evt, 140, 112-24, 128, 16, CogButs[1].val,  0.0, 1.0,  "Inner radius" )
				CogButs[2] = Number("Middle: ", Shape_Evt, 140, 112-48, 128, 16, CogButs[2].val,  0.0, 1.0,  "Middle radius" )
				CogButs[4] = Number("Bevel: ",  Shape_Evt, 140, 112-72, 128, 16, CogButs[4].val,  0.0, 1.0,  "Bevel amount" )

			elif s2Dkeys[ ShapeMenu.val ] == 'n-Side':
				nSideButs[0] = Number("Sides: ", Shape_Evt, 140, 112, 128, 16, nSideButs[0].val,  3, 100, "Number of sides" )

			elif s2Dkeys[ ShapeMenu.val ] == 'Splat':
				SplatButs[0] = Number("Sides: ", Shape_Evt, 140,    112, 128, 16, SplatButs[0].val,     3, 1000,"Number of sides" )
				SplatButs[1] = Number("Size: ",  Shape_Evt, 140, 112-24, 128, 16, SplatButs[1].val, 0.001, 6.0, "Noise size" )
				SplatButs[2] = Number("Seed: ",  Shape_Evt, 140, 112-48, 128, 16, SplatButs[2].val,     0, 255, "Noise set random seed, if seed is 0, the current time will be used instead" )
				#SplatButs[3] = Menu( noisebasismenu, Shape_Evt, 140, 112-72, 128, 16, SplatButs[3].val,         "Noise Basis" )

			#elif s2Dkeys[ ShapeMenu.val ] == 'Other2D':
			#	draw_Text( ( 16, 112+5-16 ), 'Just a test.', black, 0 )


			##------------------------------------------------------------

		else: ### 3D:

			###--------------------
			# 3D Shape Buttons:

			ShapeMenu3D = Menu( shape3Dtypesmenu, Shape_Evt, 140, 136, 128, 16, ShapeMenu3D.val, "3D Curve Shapes" )

			if s3Dkeys[ ShapeMenu3D.val ] == 'Formula':
				draw_Text( ( 16, 112+5-16 ), 'x(t), y(t), z(t) Enter a mathematical function,', black, 0 )
				draw_Text( ( 16, 112+5-32 ), 'with dependant variable t,for each of these directions. ', black, 0 )
				FormulaButs[0] = Number("Number: ", Shape_Evt, 16, 112,  112, 16, FormulaButs[0].val, 3, 1000, "Number of points" )
				FormulaButs[1] = Number("Length: ", Shape_Evt, 140, 112, 128, 16, FormulaButs[1].val, 0.01, 100.0, "Length of curve" )
				draw_Text( ( 16, 112+5-48 ), 'x(t) =', black, 0 )
				FormulaButs[2] = String("",   No_Evt, 48, 112-48, 220, 16, FormulaButs[2].val, 255 )
				draw_Text( ( 16, 112+5-72 ), 'y(t) =', black, 0 )
				FormulaButs[3] = String("",   No_Evt, 48, 112-72, 220, 16, FormulaButs[3].val, 255 )
				draw_Text( ( 16, 112+5-96 ), 'z(t) =', black, 0 )
				FormulaButs[4] = String("",   No_Evt, 48, 112-96, 220, 16, FormulaButs[4].val, 255 )

			elif s3Dkeys[ ShapeMenu3D.val ] == 'Helix':
				HelixButs[0] = Number("Number: ",     Shape_Evt,  16, 112,    112, 16, HelixButs[0].val,    3,  1000, "Number of points" )
				HelixButs[1] = Number("Heigth: ",     Shape_Evt, 140, 112,    128, 16, HelixButs[1].val, 0.01, 100.0, "Height of curve" )
				HelixButs[2] = Number("StartAngle: ", Shape_Evt, 140, 112-48, 128, 16, HelixButs[2].val,  0.0, HelixButs[3].val, "Start Angle" )
				HelixButs[3] = Number("EndAngle: ",   Shape_Evt, 140, 112-72, 128, 16, HelixButs[3].val,  HelixButs[2].val, 36000.0, "End Angle" )
				HelixButs[4] = Number("Width: ",      Shape_Evt, 140, 112-96, 128, 16, HelixButs[4].val,  0.0, 100.0, "Width" )
				HelixButs[5] = Number("a: ",          Shape_Evt,  16, 112-72, 112, 16, HelixButs[5].val, -1.0, 1.0, "Taper width" )
				HelixButs[6] = Number("b: ",          Shape_Evt,  16, 112-96, 112, 16, HelixButs[6].val, -1.0, 1.0, "" )

			elif s3Dkeys[ ShapeMenu3D.val ] == 'Turbulence':
				TurbButs[0] = Number("Number: ", Shape_Evt, 16, 112,  112, 16, TurbButs[0].val, 3, 1000, "Number of points" )
				TurbButs[1] = Number("Length: ", Shape_Evt, 140, 112, 128, 16, TurbButs[1].val, 0.01, 100.0, "Length of curve" )
				NoiseButs[0] = Number("iScale: ",  Shape_Evt, 140, 112-48, 128, 16, NoiseButs[0].val,  0.0, 10.0, "Noise intensity scale" )
				NoiseButs[1] = Number("Size: ",    Shape_Evt, 140, 112-72, 128, 16, NoiseButs[1].val, 0.01, 10.0, "Noise size" )
				NoiseButs[2] = Number("Depth: ",   Shape_Evt,  16, 112-48, 112, 16, NoiseButs[2].val, 1,12,    "Noise depth" )
				NoiseButs[3] = Toggle("Hard",      Shape_Evt,  16, 112-72, 112, 16, NoiseButs[3].val,             "SoftNoise / HardNoise" )
				NoiseButs[4] = Menu(noisebasismenu,Shape_Evt,  16, 112-96, 112, 16, NoiseButs[4].val,             "Noise basis type" )
				NoiseButs[5] = Number("Seed: ",    Shape_Evt, 140, 112-96, 128, 16, NoiseButs[5].val, 0,255,      "Set random seed, if seed=0, the current time will be used instead" )
				AxisButs[0] = Toggle("X", Shape_Evt, 140,    112-24, 32, 16, AxisButs[0].val, "X Axis" )
				AxisButs[1] = Toggle("Y", Shape_Evt, 140+32, 112-24, 32, 16, AxisButs[1].val, "Y Axis" )
				AxisButs[2] = Toggle("Z", Shape_Evt, 140+64, 112-24, 32, 16, AxisButs[2].val, "Z Axis" )
				AxisButs[3] = Toggle("I", Shape_Evt, 140+96, 112-24, 32, 16, AxisButs[3].val, "Flip" )

			'''
			### Cycloid: Not yet!
			elif s3Dkeys[ ShapeMenu3D.val ] == 'Cycloid':
				CycloidButs[0] = Number("Number: ", Shape_Evt,  16,    112, 112, 16, CycloidButs[0].val,     3,  1000,  "Number of points" )
				CycloidButs[1] = Number("Length: ", Shape_Evt, 140,    112, 128, 16, CycloidButs[1].val,  0.01, 100.0,  "Lenght of curve" )
				CycloidButs[2] = Number("Type: ",   Shape_Evt, 16,  112-96, 112, 16, CycloidButs[2].val,  0, 2,         "Type" )
				CycloidButs[3] = Number("r: ",      Shape_Evt, 140, 112-24, 128, 16, CycloidButs[3].val,   0.0, 1000.0, "" )
				CycloidButs[4] = Number("a: ",      Shape_Evt, 140, 112-48, 128, 16, CycloidButs[4].val,   0.0, 1000.0, "" )
				CycloidButs[5] = Number("Start: ",  Shape_Evt, 140, 112-72, 128, 16, CycloidButs[5].val, -36000.0, 36000.0, "StartAngle" )
				CycloidButs[6] = Number("End: ",    Shape_Evt, 140, 112-96, 128, 16, CycloidButs[6].val, -36000.0, 36000.0, "EndAngle" )
				draw_Text( ( 16, 112+5-48 ), 'Cycloid: Not yet!', black, 1 )

			### Cissoid: Not yet!
			elif s3Dkeys[ ShapeMenu3D.val ] == 'Cissoid':
				CissoidButs[0] = Number("Number: ",  Shape_Evt,  16,    112, 112, 16, CissoidButs[0].val,     3,  1000, "Number of points" )
				CissoidButs[1] = Number("Length: ",  Shape_Evt, 140,    112, 128, 16, CissoidButs[1].val,  0.01, 100.0, "Lenght of curve" )
				CissoidButs[2] = Number("Type: ",   Shape_Evt, 16,   112-96, 112, 16, CissoidButs[2].val,  0, 2,        "Type" )
				CissoidButs[3] = Number("Tension: ", Shape_Evt, 140, 112-24, 128, 16, CissoidButs[3].val,   0.0, 100.0, "Tension" )
				draw_Text( ( 16, 112+5-48 ), 'Cissoid: Not yet!', black, 1 )
			'''

			#elif s3Dkeys[ ShapeMenu3D.val ] == 'Other3D':
			#	draw_Text( ( 16, 112+5-16 ), 'Just a test.', black, 0 )


#------------------------------------------------------------------------------------------------------------------------
##------------------------------------------------------------------------------------------------------------------------
###------------------------------------------------------------------------------------------------------------------------
#### Events:
###------------------------------------------------------------------------------------------------------------------------
##------------------------------------------------------------------------------------------------------------------------
#------------------------------------------------------------------------------------------------------------------------


#--------------------------------------------------------------------------------
# events:
def events(evt, val):
	global ShapeMode, ShapeMenu3D, FormulaButs, linecolours

	#--------------------
	# ESCAPE:
	if evt in [ QKEY ] and not val:
		Exit()

	#--------------------
	# Random 2D gui line colours: use SPACEKEY in 2D mode
	if evt == SPACEKEY and not val:
		linecolours = colourfull[ int(randnum(0,14)) ]
		Draw()

	#--------------------
	# Load/Save Formula: use RIGHTMOUSE in 3D formula mode
	if evt == RIGHTMOUSE and not val:
		if ShapeMode.val !=0:
			if CurveButs[0].val !=1:
				if s3Dkeys[ ShapeMenu3D.val ] == 'Formula':
					name = "OK ?%t|Load Formula %x1|Save Formula %x2"
					result = Blender.Draw.PupMenu(name)
					if result==1:
						Window.FileSelector ( LoadFormula, "Load .txt File")
					elif result==2:
						Window.FileSelector ( SaveFormula, "Save .txt File")


#--------------------------------------------------------------------------------
# Button events:
def bevents(evt):
	global ShapeMode, ShapeMenu, ShapeMenu3D, points, CurveFlags, CurveButs

	# Quit:
	if evt == End_Evt:
		name = "Quit ?%t|YES %x1|NO %x2"
		result = Blender.Draw.PupMenu(name)
		if result==1:
			Exit()

	#
	if evt == Btn_Evt:
		Draw()

	#--------------------
	# Update Shape, Make Curve:
	if evt in [ Shape_Evt, Upd_Evt ]:
		try:
			points=[]
			if ShapeMode.val !=1:
				## 2D Shape
				newshape = Shape2DUpdate()
				sd = newshape.keys()
				points = newshape[ sd[ ShapeMenu.val ] ]
				if evt == Upd_Evt:
					make_curve( points, sd[ ShapeMenu.val ], [CurveFlags[0].val,2-CurveFlags[1].val,CurveFlags[2].val], [CurveButs[1].val,CurveButs[2].val,CurveButs[3].val,CurveButs[4].val,CurveButs[5].val,CurveButs[6].val ] )
				else:
					Draw()
			else:
				### 3D Shape
				newshape = Shape3DUpdate()
				sd = newshape.keys()
				points = newshape[ sd[ ShapeMenu3D.val ] ]
				if evt == Upd_Evt:
					make_curve( points, sd[ ShapeMenu3D.val ], [CurveFlags[0].val,2-CurveFlags[1].val,CurveFlags[2].val], [CurveButs[1].val,CurveButs[2].val,CurveButs[3].val,CurveButs[4].val,CurveButs[5].val,CurveButs[6].val ] )
				else:
					Draw()

			del sd, newshape

		except:
			Blender.Draw.PupMenu( "Unexpected error: " + str(sys.exc_info()[0]) )


#------------------------------------------------------------------------------------------------------------------------
##------------------------------------------------------------------------------------------------------------------------
###------------------------------------------------------------------------------------------------------------------------
#### Save / Load Formula to .txt file, use RightMouseButton when in Formula mode.
###------------------------------------------------------------------------------------------------------------------------
##------------------------------------------------------------------------------------------------------------------------
#------------------------------------------------------------------------------------------------------------------------
#__________________________________________________________________________
# A formula saved to .txt file looks like this:
#
# Curveaceous galore formula. ( Number(n), Length(l),  x(t), y(t), z(t) )
# 32
# 1.0
# sin(t*pi*2.0/l)
# cos(t*pi*2.0/l)
# t
#__________________________________________________________________________


def writeln(f,x):
  f.write(str(x))
  f.write("\n")
def readint(f):
  return int(f.readline())
def readfloat(f):
  return float(f.readline())
def readstr(f):
  s = (f.readline())
  return strip(s)

###-------------------------
# Write Formula to .txt file:
def SaveFormula(FName):
	global FormulaButs
	try:
		f = open(FName,'w')
		writeln(f,'Curveaceous galore formula. ( Number(n), Length(l),  x(t), y(t), z(t) )')
		writeln(f, FormulaButs[0].val )
		writeln(f, FormulaButs[1].val )	
		writeln(f, FormulaButs[2].val )
		writeln(f, FormulaButs[3].val )
		writeln(f, FormulaButs[4].val )
		f.close()
		print '\nCurveaceous galore: Formula saved to file:\n' + FName
	except:
		Blender.Draw.PupMenu( "Unexpected error: " + str(sys.exc_info()[0]) )

###-------------------------
# Read Formula from .txt file:
def LoadFormula(FName):
	global FormulaButs
	try:
		f = open(FName,'r')
		FVersion = readstr(f)
		FormulaButs[0].val = readint(f)
		FormulaButs[1].val = readfloat(f)
		FormulaButs[2].val = readstr(f)
		FormulaButs[3].val = readstr(f)
		FormulaButs[4].val = readstr(f)
		f.close()
		print '\nCurveaceous galore: Formula loaded from file:\n' + FName
	except:
		Blender.Draw.PupMenu( "Unexpected error: " + str(sys.exc_info()[0]) )

#------------------------------------------------------------------------------------------------------------------------
##------------------------------------------------------------------------------------------------------------------------
###------------------------------------------------------------------------------------------------------------------------
# Register:
Register( drawgui, events, bevents )