/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.table.store.file.utils;

import java.io.IOException;
import java.util.Iterator;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.function.BinaryOperator;
import javax.annotation.Nullable;
import org.apache.flink.core.fs.FileSystem;
import org.apache.flink.core.fs.Path;
import org.apache.flink.table.store.file.Snapshot;
import org.apache.flink.table.store.file.utils.FileUtils;
import org.apache.flink.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SnapshotManager {
    private static final Logger LOG = LoggerFactory.getLogger(SnapshotManager.class);
    private static final String SNAPSHOT_PREFIX = "snapshot-";
    public static final String EARLIEST = "EARLIEST";
    public static final String LATEST = "LATEST";
    private static final int READ_HINT_RETRY_NUM = 3;
    private static final int READ_HINT_RETRY_INTERVAL = 1;
    private final Path tablePath;

    public SnapshotManager(Path tablePath) {
        this.tablePath = tablePath;
    }

    public Path snapshotDirectory() {
        return new Path(this.tablePath + "/snapshot");
    }

    public Path snapshotPath(long snapshotId) {
        return new Path(this.tablePath + "/snapshot/" + SNAPSHOT_PREFIX + snapshotId);
    }

    public Snapshot snapshot(long snapshotId) {
        return Snapshot.fromPath(this.snapshotPath(snapshotId));
    }

    public boolean snapshotExists(long snapshotId) {
        Path path = this.snapshotPath(snapshotId);
        try {
            return path.getFileSystem().exists(path);
        }
        catch (IOException e) {
            throw new RuntimeException("Failed to determine if snapshot #" + snapshotId + " exists in path " + path, e);
        }
    }

    @Nullable
    public Long latestSnapshotId() {
        try {
            return this.findLatest();
        }
        catch (IOException e) {
            throw new RuntimeException("Failed to find latest snapshot id", e);
        }
    }

    @Nullable
    public Long earliestSnapshotId() {
        try {
            return this.findEarliest();
        }
        catch (IOException e) {
            throw new RuntimeException("Failed to find earliest snapshot id", e);
        }
    }

    @Nullable
    public Long latestCompactedSnapshotId() {
        Long latestId = this.latestSnapshotId();
        Long earliestId = this.earliestSnapshotId();
        if (latestId == null || earliestId == null) {
            return null;
        }
        for (long snapshotId = latestId.longValue(); snapshotId >= earliestId; --snapshotId) {
            Snapshot snapshot;
            if (!this.snapshotExists(snapshotId) || (snapshot = this.snapshot(snapshotId)).commitKind() != Snapshot.CommitKind.COMPACT) continue;
            return snapshot.id();
        }
        return null;
    }

    @Nullable
    public Long earlierThanTimeMills(long timestampMills) {
        Long earliest = this.earliestSnapshotId();
        Long latest = this.latestSnapshotId();
        if (earliest == null || latest == null) {
            return null;
        }
        for (long i = latest.longValue(); i >= earliest; --i) {
            long commitTime = this.snapshot(i).timeMillis();
            if (commitTime >= timestampMills) continue;
            return i;
        }
        return earliest - 1L;
    }

    @Nullable
    public Long earlierOrEqualTimeMills(long timestampMills) {
        Long earliest = this.earliestSnapshotId();
        Long latest = this.latestSnapshotId();
        if (earliest == null || latest == null) {
            return null;
        }
        for (long i = latest.longValue(); i >= earliest; --i) {
            long commitTime = this.snapshot(i).timeMillis();
            if (commitTime > timestampMills) continue;
            return i;
        }
        return null;
    }

    public long snapshotCount() throws IOException {
        return FileUtils.listVersionedFiles(this.snapshotDirectory(), SNAPSHOT_PREFIX).count();
    }

    public Iterator<Snapshot> snapshots() throws IOException {
        return FileUtils.listVersionedFiles(this.snapshotDirectory(), SNAPSHOT_PREFIX).map(this::snapshot).iterator();
    }

    public Optional<Snapshot> latestSnapshotOfUser(String user) {
        Long latestId = this.latestSnapshotId();
        if (latestId == null) {
            return Optional.empty();
        }
        long earliestId = (Long)Preconditions.checkNotNull((Object)this.earliestSnapshotId(), (String)"Latest snapshot id is not null, but earliest snapshot id is null. This is unexpected.");
        for (long id = latestId.longValue(); id >= earliestId; --id) {
            Snapshot snapshot = this.snapshot(id);
            if (!user.equals(snapshot.commitUser())) continue;
            return Optional.of(snapshot);
        }
        return Optional.empty();
    }

    @Nullable
    private Long findLatest() throws IOException {
        long nextSnapshot;
        Path snapshotDir = this.snapshotDirectory();
        FileSystem fs = snapshotDir.getFileSystem();
        if (!fs.exists(snapshotDir)) {
            return null;
        }
        Long snapshotId = this.readHint(LATEST);
        if (snapshotId != null && !this.snapshotExists(nextSnapshot = snapshotId + 1L)) {
            return snapshotId;
        }
        return this.findByListFiles(Math::max);
    }

    @Nullable
    private Long findEarliest() throws IOException {
        Path snapshotDir = this.snapshotDirectory();
        FileSystem fs = snapshotDir.getFileSystem();
        if (!fs.exists(snapshotDir)) {
            return null;
        }
        Long snapshotId = this.readHint(EARLIEST);
        if (snapshotId != null && this.snapshotExists(snapshotId)) {
            return snapshotId;
        }
        return this.findByListFiles(Math::min);
    }

    public Long readHint(String fileName) {
        Path snapshotDir = this.snapshotDirectory();
        Path path = new Path(snapshotDir, fileName);
        int retryNumber = 0;
        while (retryNumber++ < 3) {
            try {
                return Long.parseLong(FileUtils.readFileUtf8(path));
            }
            catch (Exception exception) {
                try {
                    TimeUnit.MILLISECONDS.sleep(1L);
                }
                catch (InterruptedException interruptedException) {}
            }
        }
        return null;
    }

    private Long findByListFiles(BinaryOperator<Long> reducer) throws IOException {
        Path snapshotDir = this.snapshotDirectory();
        return FileUtils.listVersionedFiles(snapshotDir, SNAPSHOT_PREFIX).reduce(reducer).orElse(null);
    }

    public void commitLatestHint(long snapshotId) throws IOException {
        this.commitHint(snapshotId, LATEST);
    }

    public void commitEarliestHint(long snapshotId) throws IOException {
        this.commitHint(snapshotId, EARLIEST);
    }

    private void commitHint(long snapshotId, String fileName) throws IOException {
        Path snapshotDir = this.snapshotDirectory();
        FileSystem fs = snapshotDir.getFileSystem();
        Path hintFile = new Path(snapshotDir, fileName);
        Path tempFile = new Path(snapshotDir, UUID.randomUUID() + "-" + fileName + ".temp");
        FileUtils.writeFileUtf8(tempFile, String.valueOf(snapshotId));
        fs.delete(hintFile, false);
        boolean success = fs.rename(tempFile, hintFile);
        if (!success) {
            fs.delete(tempFile, false);
        }
    }
}

