/*
 * Decompiled with CFR 0.152.
 */
package io.moquette.broker.unsafequeues;

import io.moquette.broker.unsafequeues.PagedFilesAllocator;
import io.moquette.broker.unsafequeues.QueueException;
import io.moquette.broker.unsafequeues.QueuePool;
import io.moquette.broker.unsafequeues.Segment;
import io.moquette.broker.unsafequeues.SegmentAllocator;
import io.moquette.broker.unsafequeues.SegmentPointer;
import io.moquette.broker.unsafequeues.VirtualPointer;
import java.nio.Buffer;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Queue {
    private static final Logger LOG = LoggerFactory.getLogger(Queue.class);
    public static final int LENGTH_HEADER_SIZE = 4;
    private final String name;
    private VirtualPointer currentHeadPtr;
    private Segment headSegment;
    private VirtualPointer currentTailPtr;
    private Segment tailSegment;
    private final QueuePool queuePool;
    private final SegmentAllocator allocator;
    private final PagedFilesAllocator.AllocationListener allocationListener;

    Queue(String name, Segment headSegment, VirtualPointer currentHeadPtr, Segment tailSegment, VirtualPointer currentTailPtr, SegmentAllocator allocator, PagedFilesAllocator.AllocationListener allocationListener, QueuePool queuePool) {
        this.name = name;
        this.headSegment = headSegment;
        this.currentHeadPtr = currentHeadPtr;
        this.currentTailPtr = currentTailPtr;
        this.tailSegment = tailSegment;
        this.allocator = allocator;
        this.allocationListener = allocationListener;
        this.queuePool = queuePool;
    }

    public void enqueue(ByteBuffer payload) throws QueueException {
        int messageSize = 4 + payload.remaining();
        if (this.headSegment.hasSpace(this.currentHeadPtr, messageSize)) {
            LOG.debug("Head segment has sufficient space for message length {}", (Object)(4 + payload.remaining()));
            this.writeData(this.headSegment, this.currentHeadPtr.plus(1), payload);
            this.currentHeadPtr = this.currentHeadPtr.moveForward(messageSize);
            return;
        }
        LOG.debug("Head segment doesn't have enough space");
        int dataSize = payload.remaining();
        ByteBuffer rawData = (ByteBuffer)ByteBuffer.allocate(4 + dataSize).putInt(dataSize).put(payload).flip();
        long bytesRemainingInHeaderSegment = Math.min((long)rawData.remaining(), this.headSegment.bytesAfter(this.currentHeadPtr));
        LOG.trace("Writing partial payload to offset {} for {} bytes", (Object)this.currentHeadPtr, (Object)bytesRemainingInHeaderSegment);
        if (bytesRemainingInHeaderSegment > 0L) {
            int copySize = (int)bytesRemainingInHeaderSegment;
            ByteBuffer slice = rawData.slice();
            slice.limit(copySize);
            this.writeDataNoHeader(this.headSegment, this.currentHeadPtr.plus(1), slice);
            this.currentHeadPtr = this.currentHeadPtr.moveForward(bytesRemainingInHeaderSegment);
            rawData.position(rawData.position() + copySize);
        }
        Segment newSegment = null;
        while (rawData.hasRemaining()) {
            newSegment = this.queuePool.nextFreeSegment();
            this.allocationListener.segmentedCreated(this.name, newSegment);
            int copySize = Math.min(rawData.remaining(), this.allocator.getSegmentSize());
            ByteBuffer slice = rawData.slice();
            slice.limit(copySize);
            this.currentHeadPtr = this.currentHeadPtr.moveForward(copySize);
            this.writeDataNoHeader(newSegment, newSegment.begin, slice);
            this.headSegment = newSegment;
            rawData.position(rawData.position() + copySize);
        }
    }

    private void writeDataNoHeader(Segment segment, SegmentPointer start, ByteBuffer data) {
        segment.write(start, data);
    }

    private void writeDataNoHeader(Segment segment, VirtualPointer start, ByteBuffer data) {
        segment.write(start, data);
    }

    private void writeData(Segment segment, VirtualPointer start, ByteBuffer data) {
        this.writeData(segment, start, data.remaining(), data);
    }

    private void writeData(Segment segment, VirtualPointer start, int size, ByteBuffer data) {
        ByteBuffer length = (ByteBuffer)ByteBuffer.allocate(4).putInt(size).flip();
        segment.write(start, length);
        segment.write(start.plus(4), data);
    }

    void force() {
        this.headSegment.force();
    }

    VirtualPointer currentHead() {
        return this.currentHeadPtr;
    }

    VirtualPointer currentTail() {
        return this.currentTailPtr;
    }

    public boolean isEmpty() {
        if (this.isTailFirstUsage(this.currentTailPtr)) {
            return this.currentHeadPtr.compareTo(this.currentTailPtr) == 0;
        }
        return this.currentHeadPtr.moveForward(1L).compareTo(this.currentTailPtr) == 0;
    }

    public Optional<ByteBuffer> dequeue() throws QueueException {
        if (!this.currentHeadPtr.isGreaterThan(this.currentTailPtr)) {
            if (this.currentTailPtr.isGreaterThan(this.currentHeadPtr)) {
                throw new QueueException("Current tail " + this.currentTailPtr + " is forward head " + this.currentHeadPtr);
            }
            return Optional.empty();
        }
        if (this.tailSegment == null) {
            this.tailSegment = this.queuePool.openNextTailSegment(this.name).get();
        }
        LOG.debug("currentTail is {}", (Object)this.currentTailPtr);
        if (Queue.containsHeader(this.tailSegment, this.currentTailPtr)) {
            VirtualPointer existingTail = this.isTailFirstUsage(this.currentTailPtr) ? this.currentTailPtr.plus(1) : this.currentTailPtr.copy();
            int payloadLength = this.tailSegment.readHeader(existingTail);
            int fullMessageSize = payloadLength + 4;
            long remainingInSegment = this.tailSegment.bytesAfter(existingTail) + 1L;
            if (remainingInSegment > (long)fullMessageSize) {
                this.currentTailPtr = existingTail.moveForward(fullMessageSize);
                VirtualPointer dataStart = existingTail.moveForward(4L);
                return Optional.of(this.readData(this.tailSegment, dataStart, payloadLength));
            }
            VirtualPointer dataStart = existingTail.moveForward(4L);
            if (remainingInSegment - 4L == 0L) {
                this.queuePool.consumedTailSegment(this.name);
                if (QueuePool.queueDebug) {
                    this.tailSegment.fillWith((byte)68);
                }
                this.tailSegment = this.queuePool.openNextTailSegment(this.name).get();
            }
            LOG.debug("Loading payload size {}", (Object)payloadLength);
            return Optional.of(this.loadPayloadFromSegments(payloadLength, this.tailSegment, dataStart));
        }
        CrossSegmentHeaderResult result = this.decodeCrossHeader(this.tailSegment, this.currentTailPtr);
        LOG.debug("Loading payload size {}", (Object)result.payloadLength);
        return Optional.of(this.loadPayloadFromSegments(result.payloadLength, result.segment, result.pointer));
    }

    private static boolean containsHeader(Segment segment, VirtualPointer tail) {
        return segment.bytesAfter(tail) + 1L >= 4L;
    }

    private CrossSegmentHeaderResult decodeCrossHeader(Segment segment, VirtualPointer pointer) throws QueueException {
        ByteBuffer lengthBuffer = ByteBuffer.allocate(4);
        ByteBuffer partialHeader = segment.readAllBytesAfter(pointer);
        int consumedHeaderSize = partialHeader.remaining();
        lengthBuffer.put(partialHeader);
        this.queuePool.consumedTailSegment(this.name);
        if (QueuePool.queueDebug) {
            segment.fillWith((byte)68);
        }
        int remainingHeaderSize = 4 - consumedHeaderSize;
        Segment nextTailSegment = this.queuePool.openNextTailSegment(this.name).get();
        lengthBuffer.put(nextTailSegment.read(nextTailSegment.begin, remainingHeaderSize));
        VirtualPointer dataStart = pointer.moveForward(4L);
        int payloadLength = ((ByteBuffer)lengthBuffer.flip()).getInt();
        return new CrossSegmentHeaderResult(nextTailSegment, dataStart, payloadLength);
    }

    private ByteBuffer loadPayloadFromSegments(int remaining, Segment segment, VirtualPointer tail) throws QueueException {
        ArrayList<ByteBuffer> createdBuffers = new ArrayList<ByteBuffer>(this.segmentCountFromSize(remaining));
        VirtualPointer scan = tail;
        do {
            LOG.debug("Looping remaining {}", (Object)remaining);
            int availableDataLength = Math.min(remaining, (int)segment.bytesAfter(scan) + 1);
            ByteBuffer buffer = segment.read(scan, availableDataLength);
            createdBuffers.add(buffer);
            boolean segmentCompletelyConsumed = segment.bytesAfter(scan) + 1L == (long)availableDataLength;
            scan = scan.moveForward(availableDataLength);
            if ((remaining -= buffer.remaining()) <= 0 && !segmentCompletelyConsumed) continue;
            this.queuePool.consumedTailSegment(this.name);
            if (QueuePool.queueDebug) {
                segment.fillWith((byte)68);
            }
            segment = this.queuePool.openNextTailSegment(this.name).orElse(null);
        } while (remaining > 0);
        this.tailSegment = segment;
        this.currentTailPtr = scan;
        LOG.debug("Moved currentTailPointer to {} from {}", (Object)scan, (Object)tail);
        return this.joinBuffers(createdBuffers);
    }

    private int segmentCountFromSize(int remaining) {
        return (int)Math.ceil((double)remaining / (double)this.allocator.getSegmentSize());
    }

    private boolean isTailFirstUsage(VirtualPointer tail) {
        return tail.isUntouched();
    }

    private ByteBuffer joinBuffers(List<ByteBuffer> buffers) {
        int neededSpace = buffers.stream().mapToInt(Buffer::remaining).sum();
        byte[] heapBuffer = new byte[neededSpace];
        int offset = 0;
        for (ByteBuffer buffer : buffers) {
            int readBytes = buffer.remaining();
            buffer.get(heapBuffer, offset, readBytes);
            offset += readBytes;
        }
        return ByteBuffer.wrap(heapBuffer);
    }

    private ByteBuffer readData(Segment source, VirtualPointer start, int length) {
        return source.read(start, length);
    }

    private static class CrossSegmentHeaderResult {
        private final Segment segment;
        private final VirtualPointer pointer;
        private final int payloadLength;

        private CrossSegmentHeaderResult(Segment segment, VirtualPointer pointer, int payloadLength) {
            this.segment = segment;
            this.pointer = pointer;
            this.payloadLength = payloadLength;
        }
    }
}

