// Metafile (WMF/EMF) to EPS Converter by Joost Verburg
// Version 1.5 - November 2008

// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.

// Parts of source code of the following GPL projects have been used:

// TpX Project http://tpx.sourceforge.net/
// EmfToEps by Dirk Struve http://www.projectory.de/emftoeps/index.html
// OLETeX Utility http://oletex.sourceforge.net/

unit Main;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, ExtDlgs, ExtCtrls, StdCtrls, Clipbrd, CheckLst, ComCtrls,
  ActnList, Menus, ToolWin, Printers, Buttons, ImgList, ShellAPI, CommDlg;

type

  TMainForm = class(TForm)
    ScrollBox: TScrollBox;
    Image: TImage;
    ActionList: TActionList;
    Open: TAction;
    SaveMetafile: TAction;
    ExitApp: TAction;
    ClipboardCopy: TAction;
    ClipboardPaste: TAction;
    Toolbar: TToolBar;
    ToolButtonOpen: TToolButton;
    ToolButtonSaveMetafile: TToolButton;
    ToolButtonSeparator: TToolButton;
    ToolButtonPaste: TToolButton;
    ToolButtonCopy: TToolButton;
    EmfFromList: TAction;
    StatusBar: TStatusBar;
    ExportEPS: TAction;
    ToolButtonExportEPS: TToolButton;
    ImageList: TImageList;

    procedure OpenExecute(Sender: TObject);
    procedure ClipboardCopyExecute(Sender: TObject);
    procedure ClipboardPasteExecute(Sender: TObject);
    procedure SaveMetafileExecute(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure ExitAppExecute(Sender: TObject);
    procedure ExportEPSExecute(Sender: TObject);
    procedure WMDropFiles(var message: TWMDropFiles); message WM_DROPFILES;

  protected 
    procedure CreateParams(var Params: TCreateParams); override;
    procedure WMSyscommand(var Message: TWmSysCommand); message WM_SYSCOMMAND;

  public
    Pic: TPicture;

    procedure GraphicToEps(const Pic: TPicture;
        const PostscriptPrinter: string; const FileName: string);
    procedure ShowGraphic;
    procedure ClipCopy;
    procedure ClipPaste;
    procedure CommandlineParameters;
  end;

var

  MainForm: TMainForm;
  PostscriptPrinter: string = 'Metafile to EPS Converter'; // Name of printer to use

implementation

uses VistaFuncs;

{$R *.dfm}

// EPS Printing
// ------------

procedure TMainForm.GraphicToEps(const Pic: TPicture;
  const PostscriptPrinter: string; const FileName: string);
var
  picWidth, picHeight: Integer;
  PxPerInchX, PxPerInchY: Integer;
  MarginsX, MarginsY: Integer;
  OffsetX, OffsetY: Integer;

  Driver, pFileName, pPrinter: array[0..MAX_PATH] of Char;
  DeviceMode: THandle;
  DevMode: PDeviceMode;

  Rect: TRect;

begin

  // Set printer so that it prints to a file and get device mode

  StrPCopy(pFileName, FileName);
  StrPCopy(pPrinter, PostscriptPrinter);

  Printer.SetPrinter(pPrinter, nil, pFileName, 0); // initialize printer
  Printer.GetPrinter(pPrinter, Driver, pFileName, DeviceMode); // get properties

  PxPerInchX := GetDeviceCaps(Printer.Handle, LOGPIXELSX); // pixels per inch
  PxPerInchY := GetDeviceCaps(Printer.Handle, LOGPIXELSY); // pixels per inch

  // Get page margins (should be 0 when the correct driver is used)

  OffsetX := GetDeviceCaps(Printer.Handle, PHYSICALOFFSETX);
  OffsetY := GetDeviceCaps(Printer.Handle, PHYSICALOFFSETY);
  MarginsX := GetDeviceCaps(Printer.Handle, PHYSICALWIDTH) -
    GetDeviceCaps(Printer.Handle, HORZRES);
  MarginsY := GetDeviceCaps(Printer.Handle, PHYSICALHEIGHT) -
    GetDeviceCaps(Printer.Handle, VERTRES);

  // Set picture dimensions

  picWidth := Round(Pic.Metafile.MMWidth / 1000 / 2.54 * PxPerInchX);
  picHeight := Round(Pic.Metafile.MMHeight / 1000 / 2.54 * PxPerInchY);

  DevMode := GlobalLock(DeviceMode);

  DevMode.dmPaperWidth := Round((Pic.Metafile.MMWidth / 10) +
   (MarginsX / PxPerInchX * 2.54 * 100));
  DevMode.dmPaperLength := Round((Pic.Metafile.MMHeight / 10) +
    (MarginsY / PxPerInchY * 2.54 * 100));

  DevMode.dmFields := DevMode.dmFields or (DM_PAPERLENGTH or DM_PAPERWIDTH);
  DevMode.dmFields := DevMode.dmFields and not (DM_PAPERSIZE or DM_FORMNAME);

  GlobalUnlock(DeviceMode);

  // Send picture to printer

  Printer.SetPrinter(pPrinter, Driver, pFileName, DeviceMode);
  Printer.BeginDoc;
  Rect.Left := OffsetX; // Offsets should be 0 with correct driver
  Rect.Top := OffsetY;
  Rect.Right := picWidth + OffsetX;
  Rect.Bottom := picHeight + OffsetY;

  Printer.Canvas.StretchDraw(Rect, Pic.Graphic);

  Printer.EndDoc;

end;


// Clipboard
// ---------

procedure TMainForm.ClipCopy;
begin

  // Assign loaded metafile to clipboard

  if Pic.Graphic <> nil then Clipboard.Assign(Pic.Graphic);

end;

procedure TMainForm.ClipPaste;
begin

  // Paste metafile from clipboard

  if Clipboard.HasFormat(CF_ENHMETAFILE) then
    begin
      // Enhanced metafile
      Pic.Metafile.Assign(Clipboard);
    end
  else if Clipboard.HasFormat(CF_METAFILEPICT) then
    begin
      // Old metafile
      Pic.Metafile.Assign(Clipboard);
    end
  else
    Exit;

  ShowGraphic;

end;


// Command line
// ------------

procedure TMainForm.CommandLineParameters();
begin

  if ParamCount = 1 then

    begin
      //ParamStr(1) = ouput file
      //Input from clipboard

      ClipPaste();
      if Pic.Graphic <> nil then
        GraphicToEps(Pic, PostscriptPrinter, ParamStr(1));

      Application.Terminate;
    end

  else if ParamCount = 2 then

    begin
      //ParamStr(1) = input file
      //ParamStr(2) = output file

      if FileExists(ParamStr(1)) then
      begin
        Pic.LoadFromFile(ParamStr(1));
        if Pic.Graphic <> nil then
          GraphicToEps(Pic, PostscriptPrinter, ParamStr(2));
      end;

      Application.Terminate;
    end
end;


// User interface
// --------------

procedure TMainForm.FormCreate(Sender: TObject);
begin
  ShowWindow(Application.Handle, SW_HIDE);
  SetWindowLong(Application.Handle, GWL_EXSTYLE,
    GetWindowLong(Application.Handle, GWL_EXSTYLE) and not WS_EX_APPWINDOW
    or WS_EX_TOOLWINDOW);
  ShowWindow(Application.Handle, SW_SHOW);

  SetVistaFonts(Self);

  Pic := TPicture.Create;
  CommandLineParameters();
  DragAcceptFiles(Handle, LongBool(True));

  TMetaFileCanvas.Create(Image.Picture.Metafile, 0).Free;
end;

procedure TMainForm.FormDestroy(Sender: TObject);
begin
  Pic.Free;
  DragAcceptFiles(Handle, LongBool(False));
end;

procedure TMainForm.CreateParams(var Params: TCreateParams);
begin  
  inherited CreateParams(Params);
  Params.ExStyle := Params.ExStyle and not WS_EX_TOOLWINDOW or
    WS_EX_APPWINDOW;
end;

procedure TMainForm.WMSyscommand(var Message: TWmSysCommand);
begin
  case (Message.CmdType and $FFF0) of
    SC_MINIMIZE:
    begin
      ShowWindow(Handle, SW_MINIMIZE);
      Message.Result := 0;
    end;
    SC_RESTORE:
    begin
      ShowWindow(Handle, SW_RESTORE);
      Message.Result := 0;
    end;
  else
    inherited;
  end;
end;

procedure TMainForm.WMDropFiles(var message: TWMDropFiles);
var FileName: String;
    FileExt: String;
    NameLen, Count: Integer;
begin
  // Get the number of files dropped...
  Count := DragQueryFile(message.Drop, $FFFFFFFF, nil, 0);
  if Count > 0 then
    // Only using first file
    begin
    FileName := #00;
    // Get the filename and extension
    NameLen := DragQueryFile(message.Drop, 0, nil, 0);
    SetLength(FileName, NameLen);
    DragQueryFile(message.Drop, 0, PChar(FileName), NameLen + 1);
    FileExt := UpperCase(ExtractFileExt(FileName));
    if FileExists(FileName) and ((FileExt = '.EMF') or (FileExt = '.WMF')) then
    begin
      // Load picture
      Pic.LoadFromFile(FileName);
      ShowGraphic;
    end;
  end;
  // Finish the drag'n'drop operation
  DragFinish(message.Drop);
  message.Result := 0;
end;


// Graphics drawing
// ----------------

procedure TMainForm.ShowGraphic;
begin
  StatusBar.Panels[0].Text := Format('%g x %g (cm)',
    [Pic.Metafile.MMWidth / 1000, Pic.Metafile.MMHeight / 1000]);

  Image.Picture.Metafile.Height := Round(Pic.Height);
  Image.Picture.Metafile.Width := Round(Pic.Width);
  with TMetaFileCanvas.Create(Image.Picture.Metafile, 0) do
  try
    StretchDraw(Rect(0, 0, Image.Picture.Metafile.Width - 1,
      Image.Picture.Metafile.Height - 1), Pic.Graphic);
  finally
    Free;
  end;

  ToolButtonSaveMetafile.Enabled := True;
  ToolButtonExportEPS.Enabled := True;
  ToolButtonCopy.Enabled := True;

end;


// User interface actions
// ----------------

function CharReplace(const Source: string; oldChar, newChar: Char): string;
var
  i: Integer;
begin
  Result := Source;
  for i := 1 to Length(Result) do
    if Result[i] = oldChar then
      Result[i] := newChar
end;

function OpenSaveFileDialog(Parent:
TWinControl; const Filter, DefExt, Title: string; var FileName:
string; DoOpen: Boolean): Boolean;
var
  ofn: TOpenFileName;
  szFile: array[0..MAX_PATH] of Char;
begin
  Result := False;
  FillChar(ofn, SizeOf(TOpenFileName), 0);
  with ofn do
  begin
    lStructSize := SizeOf(TOpenFileName);
    hwndOwner := Parent.Handle;
    lpstrFile := szFile;
    nMaxFile := SizeOf(szFile);
    if (Title <> '') then
      lpstrTitle := PChar(Title);
    StrPCopy(lpstrFile, FileName);
    lpstrFilter := PChar(CharReplace(Filter, '|', #0)+#0#0);
    if DefExt <> '' then
      lpstrDefExt := PChar(DefExt);
  end;

  if DoOpen then
  begin
    ofn.Flags := ofn.Flags or OFN_FILEMUSTEXIST;
    if GetOpenFileName(ofn) then
    begin
      Result := True;
      FileName := StrPas(szFile);
    end;
  end
  else
  begin
    if GetSaveFileName(ofn) then
    ofn.Flags := ofn.Flags or OFN_OVERWRITEPROMPT;
    begin
      Result := True;
      FileName := StrPas(szFile);
    end;
  end
end;

procedure TMainForm.OpenExecute(Sender: TObject);
var FileName: string;
begin
  OpenSaveFileDialog(MainForm, 'Metafiles|*.emf;*.wmf', '', 'Open Metafile', FileName, True);

  if FileName <> '' then
  begin
    Pic.LoadFromFile(FileName);
    ShowGraphic;
  end;
end;

procedure TMainForm.SaveMetafileExecute(Sender: TObject);
var FileName: string;
begin
  if Pic.Graphic = nil then Exit;

  OpenSaveFileDialog(MainForm, GraphicFilter(TMetaFile), GraphicExtension(TMetaFile), 'Save Metafile', FileName, False);

  if FileName <> '' then Pic.SaveToFile(FileName);
end;

procedure TMainForm.ExportEPSExecute(Sender: TObject);
var FileName: string;
begin
  if Pic.Graphic = nil then Exit;

  OpenSaveFileDialog(MainForm, 'Encapsulated PostScript (EPS)|*.eps', '*epa', 'Export EPS', FileName, False);

  if FileName <> '' then GraphicToEps(Pic, PostscriptPrinter, FileName);
end;

procedure TMainForm.ExitAppExecute(Sender: TObject);
begin
  Close;
end;

procedure TMainForm.ClipboardCopyExecute(Sender: TObject);
begin
  ClipCopy();
end;

procedure TMainForm.ClipboardPasteExecute(Sender: TObject);
begin
  ClipPaste();
end;


end.

