<?php
/*
Copyright (©) 2003-2013 Teus Benschop.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/


class Database_Bibles
{
  private static $instance;
  public $db;
  private function __construct() {
    $this->db = Database_SQLite::connect ("bibles");
  }
  public static function getInstance()
  {
    if (empty (self::$instance)) {
      self::$instance = new Database_Bibles();
    }
    return self::$instance;
  }


  public function create ()
  {
$sql = <<<'EOD'
CREATE TABLE IF NOT EXISTS names (
  bible integer,
  name text,
  versification text
);
EOD;
    Database_SQLite::exec ($this->db, $sql);

$sql = <<<'EOD'
CREATE TABLE IF NOT EXISTS data (
  id integer primary key autoincrement,
  bible integer,
  book integer,
  chapter integer,
  data text
);
EOD;
    Database_SQLite::exec ($this->db, $sql);

    $sql = "CREATE INDEX IF NOT EXISTS speedup ON data (bible, book, chapter)";
    Database_SQLite::exec ($this->db, $sql);
  }


  public function optimize () 
  {
    Database_SQLite::exec ($this->db, "REINDEX names;");
    Database_SQLite::exec ($this->db, "VACUUM names;");
    Database_SQLite::exec ($this->db, "REINDEX data;");
    Database_SQLite::exec ($this->db, "VACUUM data;");
  }


  // Returns an array with the available Bibles.
  public function getBibles ()
  {
    $bibles = array ();
    $query = "SELECT DISTINCT name FROM names ORDER BY name ASC;";
    $result = Database_SQLite::query ($this->db, $query);
    foreach ($result as $row) {
      $bibles [] = $row [0];
    }
    return $bibles;
  }


  // Returns the ID for a named Bible.
  public function getID ($name)
  {
    $name = Database_SQLiteInjection::no ($name);
    $query = "SELECT bible FROM names WHERE name = '$name';";
    $result = Database_SQLite::query ($this->db, $query);
    foreach ($result as $row) {
      return $row [0];
    }
    return 0;
  }


  // Returns the Bible name for a Bible ID.
  public function getName ($id)
  {
    $id = Database_SQLiteInjection::no ($id);
    $query = "SELECT name FROM names WHERE bible = $id;";
    $result = Database_SQLite::query ($this->db, $query);
    foreach ($result as $row) {
      return $row [0];
    }
    return "Unknown";
  }


  // Creates a new empty Bible. Returns its ID.
  public function createBible ($name)
  {
    // If the Bible already exists, return its ID.
    $id = $this->getID ($name);
    if ($id > 0) {
      return $id;
    }

    // Get the first free ID.
    $id = 0;
    $query = "SELECT bible FROM names ORDER BY bible DESC LIMIT 1;";
    $result = Database_SQLite::query ($this->db, $query);
    foreach ($result as $row) {
      $id = $row [0];
    }
    $id++;

    // Create the empty system.
    $name = Database_SQLiteInjection::no ($name);
    $query = "INSERT INTO names (bible, name, versification) VALUES ($id, '$name', 'English');";
    Database_SQLite::exec ($this->db, $query);

    // Return new ID.
    return $id;
  }


  // Deletes a Bible.
  public function deleteBible ($name)
  {
    // Store diff data prior to deletion.
    $database_modifications = Database_Modifications::getInstance ();
    $database_modifications->storeTeamDiffBible ($name);
    // Delete it.
    $id = $this->getID ($name);
    $query = "DELETE FROM names WHERE bible = $id;";
    Database_SQLite::exec ($this->db, $query);
    $query = "DELETE FROM data WHERE bible = $id;";
    Database_SQLite::exec ($this->db, $query);
  }


  // Gets the versification system of Bible $name.
  public function getVersification ($name)
  {
    $versification = "";
    $name = Database_SQLiteInjection::no ($name);
    $query = "SELECT versification FROM names WHERE name = '$name';";
    $result = Database_SQLite::query ($this->db, $query);
    foreach ($result as $row) {
      $versification = $row[0];
    }
    if ($versification == "") $versification = "English";
    return $versification;
  }


  // Sets the versification system of Bible $name.
  public function setVersification ($name, $versification)
  {
    $name = Database_SQLiteInjection::no ($name);
    $versification = Database_SQLiteInjection::no ($versification);
    $query = "UPDATE names SET versification = '$versification' WHERE name = '$name';";
    Database_SQLite::exec ($this->db, $query);
  }



  // Stores data of one chapter in Bible $name,
  public function storeChapter ($name, $book, $chapter_number, $chapter_text)
  {
    // Return if the Bible $name does not exist.
    $bible_id = $this->getID ($name);
    if ($bible_id == 0) {
      $database_logs = Database_Logs::getInstance();
      $database_logs->log ("Bible $name does not exist: Failed to store a chapter", Filter_Roles::ADMIN_LEVEL);
      return;
    }

    // Store the data of the current chapter in the diff table prior to storing the new data.
    $database_modifications = Database_Modifications::getInstance ();
    $database_modifications->storeTeamDiff ($name, $book, $chapter_number);

    // Store the chapter.
    $book = Database_SQLiteInjection::no ($book);
    $chapter_number = Database_SQLiteInjection::no ($chapter_number);
    $chapter_text = Database_SQLiteInjection::no ($chapter_text);
    $query = "DELETE FROM data WHERE bible = $bible_id AND book = $book AND chapter = $chapter_number;";
    Database_SQLite::exec ($this->db, $query);
    $query = "INSERT INTO data (bible, book, chapter, data) VALUES ($bible_id, $book, $chapter_number, '$chapter_text');";
    Database_SQLite::exec ($this->db, $query);
    // Update search fields.
    $this->updateSearchFields ($name, $book, $chapter_number);
  }


  public function updateSearchFields ($name, $book, $chapter)
  {
    $database_search = Database_Search::getInstance ();
    $database_search->updateSearchFields ($name, $book, $chapter);
  }


  // Returns an array with the available books in a Bible.
  public function getBooks ($bible)
  {
    // Read the books from the database.
    $books = array ();
    $bible = $this->getID ($bible);
    $query = "SELECT DISTINCT book FROM data WHERE bible = $bible;";
    $result = Database_SQLite::query ($this->db, $query);
    foreach ($result as $row) {
      $books[] = $row[0];
    }

    // Sort the books according to the sequence defined in the books database.
    $sequence = array ();
    $database_books = Database_Books::getInstance ();
    foreach ($books as $book) {
      $sequence [] = $database_books->getSequenceFromId ($book);
    }
    array_multisort ($sequence, $books);

    // Result.
    return $books;
  }


  public function deleteBook ($bible, $book)
  {
    // Store diff data prior to deleting the book.
    $database_modifications = Database_Modifications::getInstance ();
    $database_modifications->storeTeamDiffBook ($bible, $book);

    // Delete the book.
    $bible = $this->getID ($bible);
    $book = Database_SQLiteInjection::no ($book);
    $query = "DELETE FROM data WHERE bible = $bible AND book = $book;";
    Database_SQLite::exec ($this->db, $query);
  }


  // Returns an array with the available chapters in a $book in a Bible.
  public function getChapters ($bible, $book)
  {
    // Read the chapters from the database.
    $chapters = array ();
    $bible = $this->getID ($bible);
    $book = Database_SQLiteInjection::no ($book);
    $query = "SELECT DISTINCT chapter FROM data WHERE bible = $bible AND book = $book ORDER BY chapter ASC;";
    $result = Database_SQLite::query ($this->db, $query);
    foreach ($result as $row) {
      $chapters[] = $row[0];
    }
    return $chapters;
  }


  public function deleteChapter ($bible, $book, $chapter)
  {
    // Store diff data prior to deletion.
    $database_modifications = Database_Modifications::getInstance ();
    $database_modifications->storeTeamDiff ($bible, $book, $chapter);
    // Delete the chapter.
    $bible = $this->getID ($bible);
    $book = Database_SQLiteInjection::no ($book);
    $chapter = Database_SQLiteInjection::no ($chapter);
    $query = "DELETE FROM data WHERE bible = $bible AND book = $book AND chapter = $chapter;";
    Database_SQLite::exec ($this->db, $query);
  }


  // Gets the chapter data as a string.
  public function getChapter ($bible, $book, $chapter)
  {
    $bible = Database_SQLiteInjection::no ($bible);
    if (!is_numeric ($bible)) $bible = $this->getID ($bible);
    $book = Database_SQLiteInjection::no ($book);
    $chapter = Database_SQLiteInjection::no ($chapter);
    $query = "SELECT data FROM data WHERE bible = $bible AND book = $book AND chapter = $chapter;";
    $result = Database_SQLite::query ($this->db, $query);
    foreach ($result as $row) {
      return $row[0];
    }
    return "";
  }


  // Gets the chapter id.
  public function getChapterId ($bible, $book, $chapter)
  {
    if (!is_numeric ($bible)) $bible = $this->getID ($bible);
    $bible = Database_SQLiteInjection::no ($bible);
    $book = Database_SQLiteInjection::no ($book);
    $chapter = Database_SQLiteInjection::no ($chapter);
    $query = "SELECT rowid FROM data WHERE bible = $bible AND book = $book AND chapter = $chapter;";
    $result = Database_SQLite::query ($this->db, $query);
    foreach ($result as $row) {
      return $row[0];
    }
    return 0;
  }


}


?>
