/*
Copyright (©) 2003-2013 Teus Benschop.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 3 of the License, or
(at your option) any later version.
 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
  
You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/


$(document).ready (function () {
  rangy.init ();
  navigationNewPassage ();
  $ ("#editor").on ("paste cut keydown", editorChanged);
  $ (window).on ("unload", editorUnload);
  $ ("#editor").on ("paste", function (e) {
    var data = e.originalEvent.clipboardData.getData ('Text');
    e.preventDefault();
    document.execCommand ("insertHTML", false, data);
  });
  editorIdPoller ();
  $ ("#editor").on ("paste cut click", editorCaretChanged);
  $ ("#editor").on ("keydown", editorHandleKeyDown);
  $ ("#editor").on ("focus", editorWindowFocused);
  $ ("#editor").on ("blur", editorWindowBlurred);
  $ ("#editor").focus ();
  editorBindUnselectable ();
  $ ("#stylebutton").on ("click", handleStyleButton);
  $ (window).on ("keydown", editorPageKeyDown);
  $ (window).scroll (function () {
    $ ('#editorinnerheader').toggleClass('editorheaderscroll', $ (window).scrollTop () > $ ('#editorheader').offset ().top);
  });  
});


var editorBible;
var editorBook;
var editorNavigationBook;
var editorChapter;
var editorNavigationChapter;
var editorNavigationVerse;
var editorChangedTimeout;
var editorLoadedText;
var editorIdChapter = 0;
var editorIdTimeout;
var editorCaretTimeout;
var editorReload = false;
var editorCaretPosition = 0;
var editorClarifierTimeout;
var editorPreviousWidth;
var editorPreviousHeight;
var editorTextChanged = false;


function navigationNewPassage ()
{
  if (typeof navigationBook != 'undefined') {
    editorNavigationBook = navigationBook;
    editorNavigationChapter = navigationChapter;
    editorNavigationVerse = navigationVerse;
  } else if (parent.window.navigationBook != 'undefined') {
    editorNavigationBook = parent.window.navigationBook;
    editorNavigationChapter = parent.window.navigationChapter;
    editorNavigationVerse = parent.window.navigationVerse;
  } else {
    return;
  }
  editorSaveChapter ();
  editorReload = false;
  editorLoadChapter ();
  positionCaretViaAjax ();
}


function editorLoadChapter ()
{
  if ((editorNavigationBook != editorBook) || (editorNavigationChapter != editorChapter) || editorReload) {
    editorBible = navigationBible;
    editorBook = editorNavigationBook;
    editorChapter = editorNavigationChapter;
    editorIdChapter = 0;
    editorCaretPosition = getCaretPosition ();
    $ ("#editor").attr ("contenteditable", "false");
    $.ajax ({
      url: "load.php",
      type: "GET",
      data: { bible: editorBible, book: editorBook, chapter: editorChapter },
      success: function (response) {
        $ ("#editor").empty ();
        $ ("#editor").append (response);
        $ ("#editor").attr ("contenteditable", "true");
        editorStatus (editorChapterLoaded);
        editorLoadedText = response;
        if (editorReload) {
          positionCaret (editorCaretPosition);
        } else {
          positionCaretViaAjax ();
        }
        editorReload = false;
      },
    });
  }
}


function editorUnload ()
{
  editorSaveChapter (true);
}


function editorSaveChapter (sync)
{
  editorTextChanged = false;
  if (!editorBible) return;
  if (!editorBook) return;
  var html = $ ("#editor").html ();
  html = html.replace (' id="focus"', '');
  if (html == editorLoadedText) return;
  editorStatus (editorChapterSaving);
  editorLoadedText = editor;
  editorIdChapter = 0;
  var async = true;
  if (sync) async = false;
  $.ajax ({
    url: "save.php",
    type: "POST",
    async: async,
    data: { bible: editorBible, book: editorBook, chapter: editorChapter, html: html },
    success: function (response) {
      editorStatus (response);
    },
  });
}


function editorChanged ()
{
  editorTextChanged = true;
  if (editorChangedTimeout) {
    clearTimeout (editorChangedTimeout);
  }
  editorChangedTimeout = setTimeout (editorSaveChapter, 1000);
  restartFocusClarifier ();
}


function editorStatus (text)
{
  $ ("#editorstatus").empty ();
  $ ("#editorstatus").append (text);
}


function editorIdPoller ()
{
  if (editorIdTimeout) {
    clearTimeout (editorIdTimeout);
  }
  editorIdTimeout = setTimeout (editorEditorPollId, 1000);
}


function editorEditorPollId ()
{
  $.ajax ({
    url: "id.php",
    type: "GET",
    data: { bible: editorBible, book: editorBook, chapter: editorChapter },
    success: function (response) {
      if (editorIdChapter != 0) {
        if (response != editorIdChapter) {
          editorReload = true;
          editorLoadChapter ();
          editorIdChapter = 0;
        }
      }
      editorIdChapter = response;
    },
    complete: function (xhr, status) {
      editorIdPoller ();
    }
  });
}


function editorCaretChanged (event)
{
  if (editorCaretTimeout) {
    clearTimeout (editorCaretTimeout);
  }
  editorCaretTimeout = setTimeout (editorHandleCaret, 500);
  restartFocusClarifier ();
}


function editorHandleKeyDown (event)
{
  // Ctrl-G: No action.
  if ((event.ctrlKey == true) && (event.keyCode == 71)) {
    return;
  }
  // Alt / Ctrl / Shift: No action.
  if (event.keyCode == 18) return;
  if (event.keyCode == 17) return;
  if (event.keyCode == 16) return;

  editorCaretChanged ();
}


function editorPageKeyDown (event)
{
  // Ctrl-S: Style.
  if ((event.ctrlKey == true) && (event.keyCode == 83)) {
    handleStyleButton ();
    return false;
  }
  // Escape.
  if (event.keyCode == 27) {
    editorClearStyles ();
  }
}


function editorHandleCaret ()
{
  if (editorTextChanged) {
    editorCaretChanged ();
    return;
  }
  if ($ ("#editor").is (":focus")) {
    var offset = getCaretPosition ();
    $.ajax ({
      url: "offset.php",
      type: "GET",
      data: { bible: editorBible, book: editorBook, chapter: editorChapter, offset: offset },
      success: function (response) {
      }
    });
  }
}


function positionCaretViaAjax ()
{
  $.ajax ({
    url: "focus.php",
    type: "GET",
    data: { bible: editorBible, book: editorBook, chapter: editorChapter },
    success: function (response) {
      if (response != "") {
        response = $.parseJSON (response);
        var start = response ["start"];
        var end = response ["end"];
        var offset = getCaretPosition ();
        if ((offset < start) || (offset > end)) {
          positionCaret (start);
        }
      }
      restartFocusClarifier ();
    }
  });
}


function getCaretPosition ()
{
  var position = undefined;
  var editor = $ ("#editor");
  if (editor.is (":focus")) {
    var element = editor.get (0);
    position = getCaretCharacterOffsetWithin (element);
  }
  return position;
}


function getCaretCharacterOffsetWithin (element)
{
  var caretOffset = 0;
  if (typeof window.getSelection != "undefined") {
    var range = window.getSelection().getRangeAt(0);
    var preCaretRange = range.cloneRange();
    preCaretRange.selectNodeContents(element);
    preCaretRange.setEnd(range.endContainer, range.endOffset);
    caretOffset = preCaretRange.toString().length;
  } else if (typeof document.selection != "undefined" && document.selection.type != "Control") {
    var textRange = document.selection.createRange();
    var preCaretTextRange = document.body.createTextRange();
    preCaretTextRange.moveToElementText(element);
    preCaretTextRange.setEndPoint("EndToEnd", textRange);
    caretOffset = preCaretTextRange.text.length;
  }
  return caretOffset;
}


function positionCaret (position)
{
  var currentPosition = getCaretPosition ();
  if (currentPosition == undefined) return;
  if (position == undefined) return;
  var selection = rangy.getSelection ();
  selection.move ("character", position - currentPosition);
}


function editorWindowFocused ()
{
}


function editorWindowBlurred ()
{
}


function restartFocusClarifier ()
{
  if (editorClarifierTimeout) {
    clearTimeout (editorClarifierTimeout);
  }
  editorClarifierTimeout = setTimeout (clarifyFocus, 100);
}


function clarifyFocus ()
{
  $ (".v").each (function (index) {
    var element = $(this);
    var text = element[0].innerText;
    if (text == editorNavigationVerse) {
      element.attr ("id", "focus");
      var offset = element.offset ();
      var verseTop = offset.top;
      var viewportHeight = $(window).height ();
      var scrollTo = verseTop - (viewportHeight / 2);
      $ ("body").animate ({ scrollTop: scrollTo }, 500);
    } else {
      if (element.attr ("id") == "focus") {
        element.removeAttr ("id");
      }
    }
  });
}


function handleStyleButton ()
{
  $.get ("styles.php", function (response) {
    editorShowResponse (response);
    editorBindUnselectable ();
    dynamicClickHandlers ();
  });
  return false;
}


function editorBindUnselectable ()
{
  var elements = $ (".unselectable");
  elements.off ("mousedown");
  elements.on ("mousedown", function (event) {
    event.preventDefault();
  });
}


function editorShowResponse (response)
{
  $ ("#stylebutton").hide ();
  $ ("#nostyles").hide ();
  var area = $ ("#stylesarea");
  area.empty ();
  area.append (response);
}


function editorClearStyles ()
{
  var area = $ ("#stylesarea");
  area.empty ();
  $ ("#stylebutton").show ();
  $ ("#nostyles").show ();
}


function dynamicClickHandlers ()
{
  var elements = $ ("#stylesarea > a");
  elements.on ("click", function (event) {
    event.preventDefault();
    editorClearStyles ();
    $ ("#editor").focus ();
    var href = event.currentTarget.href;
    href = href.substring (href.lastIndexOf ('/') + 1);
    if (href == "cancel") return;
    if (href == "all") {
      displayAllStyles ();
    } else {
      requestStyle (href);
    }
  });

  $ ("select").on ("change", function (event) {
    var selection = $ ("select option:selected").text ();
    var style = selection.substring (0, selection.indexOf (" "));
    event.preventDefault();
    editorClearStyles ();
    $ ("#editor").focus ();
    requestStyle (style);
  });
}


function requestStyle (style)
{
  $.get ("styles.php?style=" + style, function (response) {
    response = $.parseJSON (response);
    var style = response ["style"];
    var action = response ["action"];
    if (action == "p") {
      applyParagraphStyle (style);
      editorChanged ();
    } else if (action == 'c') {
      applyCharacterStyle (style);
      editorChanged ();
    }
  });
}


function displayAllStyles ()
{
  $.get ("styles.php?all=", function (response) {
    editorShowResponse (response);
    editorBindUnselectable ();
    dynamicClickHandlers ();
  });
}


function applyParagraphStyle (style)
{
  $ ("#editor").focus ();
  var parent = rangy.getSelection().anchorNode.parentElement;
  parent = $ (parent);
  var paragraph = parent.closest ("p");
  paragraph.removeClass ();
  paragraph.addClass (style);
}


function applyCharacterStyle (style)
{
  $ ("#editor").focus ();
  var cssApplier = rangy.createCssClassApplier (style);
  cssApplier.toggleSelection ();
}

