/**CHeaderFile*************************************************************
  PackageName [Biddy]
  Synopsis    [Biddy provides data structures and algorithms
               for the representation and manipulation of Boolean
               functions with ROBDDs. A hash table is used for quick
               search of nodes. Complement edges decreases the number
               of nodes. Function ITE is given for manipulation of
               boolean functions. An automatic garbage collection with
               a formulae counter is implemented.]

  FileName    [bdd.h]
  Revision    [$Revision: 37 $]
  Date        [$Date: 2011-12-27 12:06:35 +0100 (tor, 27 dec 2011) $]
  Authors     [Robert Meolic (meolic@uni-mb.si),
               Ales Casar (casar@uni-mb.si)]
  Description [The file bdd.h contains declaration of all external
               data structures.]
  SeeAlso     [bddInt.h]

  Copyright   [This file is part of Biddy.
               Copyright (C) 2006, 2012 UM-FERI
               UM-FERI, Smetanova ulica 17, SI-2000 Maribor, Slovenia

               Biddy is free software; you can redistribute it and/or modify
               it under the terms of the GNU General Public License as
               published by the Free Software Foundation; either version 2
               of the License, or (at your option) any later version.

               Biddy is distributed in the hope that it will be useful,
               but WITHOUT ANY WARRANTY; without even the implied warranty of
               MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
               GNU General Public License for more details.

               You should have received a copy of the GNU General Public
               License along with this program; if not, write to the Free
               Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
               Boston, MA 02110-1301 USA.]
  ************************************************************************/

#ifndef _BIDDY
#define _BIDDY

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <stdarg.h>

#include <unistd.h>
#include <math.h>

/* ON WIN32 THERE HAS TO BE DEFINED WIN32 */
/* ON UNIX THERE HAS TO BE DEFINED UNIX */
/* ON MACOSX THERE HAS TO BE DEFINED MACOSX */

#ifdef UNIX
#  ifndef EXTERN
#    define EXTERN extern
#  endif
#endif

#ifdef WIN32
#  ifdef BUILD_BIDDY
#    undef EXTERN
#    define EXTERN __declspec (dllexport)
#  else
#    ifdef USE_BIDDY
#      undef EXTERN
#      define EXTERN __declspec (dllimport)
#    else
#      define EXTERN extern
#    endif
#  endif
#endif

#ifdef MACOSX
#  ifndef EXTERN
#    define EXTERN extern
#  endif
#endif

/*-----------------------------------------------------------------------*/
/* Constant definitions                                                  */
/*-----------------------------------------------------------------------*/

#define FALSE 0
#define TRUE !FALSE

/*-----------------------------------------------------------------------*/
/* Macro definitions                                                     */
/*-----------------------------------------------------------------------*/

/* Biddy_Eqv return TRUE iff given BDDs are equal */

#define Biddy_isEqv(f,g) (((f).p == (g).p) && ((f).mark == (g).mark))

/* Biddy_Terminal return TRUE iff given BDD is a constant */

#define Biddy_isTerminal(f) ((f).p == biddy_termTrue.p)

/* Biddy_Null return TRUE iff given BDD is a NULL edge */

#define Biddy_isNull(f) ((f).p == biddy_termNull.p)

/*-----------------------------------------------------------------------*/
/* Type declarations                                                     */
/*-----------------------------------------------------------------------*/

typedef char Biddy_Boolean;
typedef char *Biddy_String;
typedef int Biddy_Variable;
typedef void (*Biddy_UserFunction)();

/*-----------------------------------------------------------------------*/
/* Structure declarations                                                */
/*-----------------------------------------------------------------------*/

/* EDGE */
typedef struct Biddy_Edge {
  void *p;
  Biddy_Boolean mark;
} Biddy_Edge;

/*-----------------------------------------------------------------------*/
/* Variable declarations                                                 */
/*-----------------------------------------------------------------------*/

EXTERN Biddy_Edge biddy_termTrue;            /* terminal node */

EXTERN Biddy_Edge biddy_termFalse;           /* terminal node */

EXTERN Biddy_Edge biddy_termNull;                /* null edge */

/**AutomaticStart*********************************************************/

/*-----------------------------------------------------------------------*/
/* Function prototypes                                                   */
/*-----------------------------------------------------------------------*/

/*-----------------------------------------------------------------------*/
/* FUNCTIONS EXPORTED FROM biddyMain.c                                   */
/*-----------------------------------------------------------------------*/

EXTERN void Biddy_Init();

EXTERN void Biddy_Exit();

EXTERN char *Biddy_About();

EXTERN Biddy_Boolean Biddy_isOK(Biddy_Edge f);

EXTERN Biddy_Edge Biddy_GetThen(Biddy_Edge f);

EXTERN Biddy_Edge Biddy_GetElse(Biddy_Edge f);

EXTERN Biddy_Variable Biddy_GetVariable(Biddy_Edge f);

EXTERN Biddy_String Biddy_GetVariableName(Biddy_Edge f);

EXTERN char Biddy_GetVariableChar(Biddy_Edge f);

EXTERN Biddy_String Biddy_GetIthVariableName(int i);

EXTERN int Biddy_GetVariableOrder(Biddy_Edge f);

EXTERN void Biddy_SelectNode(Biddy_Edge f);

EXTERN void Biddy_DeselectNode(Biddy_Edge f);

EXTERN Biddy_Boolean Biddy_isSelected(Biddy_Edge f);

EXTERN Biddy_Edge Biddy_FindTerminal(Biddy_String x);

EXTERN Biddy_Edge Biddy_FoaTerminal(Biddy_String x);

EXTERN Biddy_Edge Biddy_FoaNode(Biddy_Variable v, Biddy_Edge pf, Biddy_Edge pt);

EXTERN Biddy_Edge Biddy_NOT(Biddy_Edge f);

EXTERN Biddy_Edge Biddy_TransferMark(Biddy_Edge f, Biddy_Boolean mark);

EXTERN Biddy_Edge Biddy_ITE(Biddy_Edge f, Biddy_Edge g, Biddy_Edge h);

EXTERN Biddy_Edge Biddy_Restrict(Biddy_Edge f, Biddy_Variable v, Biddy_Boolean value);

EXTERN Biddy_Edge Biddy_Compose(Biddy_Edge f, Biddy_Variable v, Biddy_Edge g);

EXTERN Biddy_Edge Biddy_E(Biddy_Edge f, Biddy_Variable v);

EXTERN Biddy_Edge Biddy_A(Biddy_Edge f, Biddy_Variable v);

EXTERN void Biddy_IncCounter();

EXTERN void Biddy_Fresh(Biddy_Edge f);

EXTERN void Biddy_Fortify(Biddy_Edge f);

EXTERN void Biddy_Garbage();

EXTERN void Biddy_AddCache(Biddy_UserFunction gc);

EXTERN void Biddy_NodeSelect(Biddy_Edge f);

EXTERN void Biddy_NodeRepair(Biddy_Edge f);

/*-----------------------------------------------------------------------*/
/* FUNCTIONS EXPORTED FROM biddyStat.c                                   */
/*-----------------------------------------------------------------------*/

EXTERN int Biddy_NodeTableSize();

EXTERN int Biddy_NodeTableMax();

EXTERN int Biddy_NodeTableNum();

EXTERN int Biddy_NodeTableNumF();

EXTERN int Biddy_NodeTableFOA();

EXTERN int Biddy_NodeTableCompare();

EXTERN int Biddy_NodeTableAdd();

EXTERN int Biddy_NodeTableGarbage();

EXTERN int Biddy_NodeTableGenerated();

EXTERN int Biddy_NodeTableBlockNumber();

EXTERN int Biddy_ListUsed();

EXTERN int Biddy_ListMaxLength();

EXTERN float Biddy_ListAvgLength();

EXTERN int Biddy_IteCacheFind();

EXTERN int Biddy_IteCacheOverwrite();

EXTERN int Biddy_NodeNumber(Biddy_Edge f);

EXTERN int Biddy_NodeMaxLevel(Biddy_Edge f);

EXTERN float Biddy_NodeAvgLevel(Biddy_Edge f);

/**AutomaticEnd***********************************************************/

#endif  /* _BIDDY */
