/*
 * hla.cxx - FlightGear interface to HLA 1.3 
 * http://virtualair.sourceforge.net 
 *
 * Copyright (C) 2008  Petr Gotthard <petr.gotthard@centrum.cz>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * $Id: hla.cxx 6 2008-06-01 12:21:09Z gotthardp $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <AIModel/AIManager.hxx>
#include <FDM/flight.hxx>
#include "hla.hxx"

// These constants are provided so that the ident 
// command can list file versions
const char sFG_HLA_BID[] = "$Id: hla.cxx 6 2008-06-01 12:21:09Z gotthardp $";
const char sFG_HLA_HID[] = FG_HLA_HID;

FGHLA::FGHLA (const int rate, const std::string &federateName,
  const std::string &federationName, const std::string &federationModel)
{
  set_hz(rate);

  mFederateName = federateName;
  mFederationName = federationName;

  if (!federationModel.empty())
    mFederationModel = federationModel;
  else
    mFederationModel = federationName + ".fed"; // default filename
}

FGHLA::~FGHLA() 
  throw (RTI::FederateInternalError)
{
}

bool FGHLA::open()
{
  try
  {
    mRtiAmbassador.createFederationExecution(mFederationName.c_str(), mFederationModel.c_str());
    SG_LOG(SG_NETWORK,SG_INFO,"FGHLA::open() new federation executed");
  }
  catch (RTI::FederationExecutionAlreadyExists& e)
  {
    SG_LOG(SG_NETWORK,SG_DEBUG,"FGHLA::open() execution already exists");
  }

  // join the federation
  mRtiAmbassador.joinFederationExecution(mFederateName.c_str(), mFederationName.c_str(), this);
  SG_LOG(SG_NETWORK,SG_INFO,"FGHLA::open() federation joined");

  mClassHandle = mRtiAmbassador.getObjectClassHandle("Aircraft");

  std::auto_ptr<RTI::AttributeHandleSet> attributeSet(RTI::AttributeHandleSetFactory::create(0));

  // attributes according to RPR-FOM (SISO-STD-001.1-1999)
  attributeSet->add(mAttrLocation =
    mRtiAmbassador.getAttributeHandle("WorldLocation", mClassHandle));
  attributeSet->add(mAttrOrientation =
    mRtiAmbassador.getAttributeHandle("Orientation", mClassHandle));
  attributeSet->add(mAttrLinearVelocity =
    mRtiAmbassador.getAttributeHandle("VelocityVector", mClassHandle));
  attributeSet->add(mAttrLinearAcceleration =
    mRtiAmbassador.getAttributeHandle("AccelerationVector", mClassHandle));
  attributeSet->add(mAttrAngularVelocity =
    mRtiAmbassador.getAttributeHandle("AngularVelocityVector", mClassHandle));
  // proprietary attributes
  attributeSet->add(mAttrVisualModel =
    mRtiAmbassador.getAttributeHandle("VisualModel", mClassHandle));

  mRtiAmbassador.subscribeObjectClassAttributes(mClassHandle, *attributeSet, RTI::RTI_TRUE);

  mCallSign = fgGetString("/sim/multiplay/callsign");
  if (!mCallSign.empty())
  {
    SG_LOG(SG_NETWORK,SG_INFO,"FGHLA::open() callsign='" << mCallSign << "'");

    mRtiAmbassador.publishObjectClass(mClassHandle, *attributeSet);
    mObjectHandle = mRtiAmbassador.registerObjectInstance(mClassHandle, mCallSign.c_str());
  }

  set_enabled(true);
  return true;
}

bool FGHLA::close()
{
  set_enabled(false);

  mRtiAmbassador.unsubscribeObjectClass(mClassHandle);

  // did we publish own objects?
  if (!mCallSign.empty())
  {
    mRtiAmbassador.deleteObjectInstance(mObjectHandle, NULL);
    mRtiAmbassador.unpublishObjectClass(mClassHandle);
  }

  mRtiAmbassador.resignFederationExecution(
    RTI::DELETE_OBJECTS_AND_RELEASE_ATTRIBUTES);

  return true;
}

//  Description: Sends the position data for the local position.
bool FGHLA::process()
{
  if (mCallSign.empty())
  {
    // running in a stealth mode
    mRtiAmbassador.tick(0.0, 0.2);

    return true;
  }

  FGInterface *ifce = cur_fdm_state;
  std::auto_ptr<RTI::AttributeHandleValuePairSet> attributeValues(RTI::AttributeSetFactory::create(0));

  double lon = ifce->get_Longitude();
  double lat = ifce->get_Latitude();
  // calculate geodetic coordinates
  SGGeod geodPosition = SGGeod::fromRadFt(lon, lat, ifce->get_Altitude());
  // convert to cartesian coordinates
  SGVec3d cartPosition = SGVec3d::fromGeod(geodPosition);

  double position[3]; // X,Y,Z [m]
  for (unsigned i = 0; i < 3; i++)
    position[i] = cartPosition(i);
  attributeValues->add(mAttrLocation, (char*)position, sizeof(position));

  float orientation[3]; // X,Y,Z [rad]
  orientation[0] = ifce->get_Phi(); // roll(x)
  orientation[1] = ifce->get_Theta(); // pitch(y)
  orientation[2] = ifce->get_Psi(); // heading(z)
  attributeValues->add(mAttrOrientation, (char*)orientation, sizeof(orientation));

  float linearVel[3]; // U,V,W [m/s]
  linearVel[0] = SG_FEET_TO_METER*ifce->get_U_body();
  linearVel[1] = SG_FEET_TO_METER*ifce->get_V_body();
  linearVel[2] = SG_FEET_TO_METER*ifce->get_W_body();
  attributeValues->add(mAttrLinearVelocity, (char*)linearVel, sizeof(linearVel));

  float linearAccel[3];
  linearAccel[0] = SG_FEET_TO_METER*ifce->get_U_dot_body();
  linearAccel[1] = SG_FEET_TO_METER*ifce->get_V_dot_body();
  linearAccel[2] = SG_FEET_TO_METER*ifce->get_W_dot_body();
  attributeValues->add(mAttrLinearAcceleration, (char*)linearAccel, sizeof(linearAccel));

  float angularVel[3]; // P,Q,R [rad/s]
  angularVel[0] = SG_FEET_TO_METER*ifce->get_P_body();
  angularVel[1] = SG_FEET_TO_METER*ifce->get_Q_body();
  angularVel[2] = SG_FEET_TO_METER*ifce->get_R_body();
  attributeValues->add(mAttrAngularVelocity, (char*)angularVel, sizeof(angularVel));

  const char* model = fgGetString("/sim/model/path");
  attributeValues->add(mAttrVisualModel, model, ::strlen(model));

  mRtiAmbassador.updateAttributeValues(mObjectHandle, *attributeValues, mCallSign.c_str());

  mRtiAmbassador.tick(0.0, 0.2);
  return true;
}

void FGHLA::discoverObjectInstance(RTI::ObjectHandle object_handle,
  RTI::ObjectClassHandle class_handle, const char *event_tag)
throw (RTI::CouldNotDiscover, RTI::ObjectClassNotKnown, RTI::FederateInternalError)
{
  std::auto_ptr<RTI::AttributeHandleSet> attributeSet(RTI::AttributeHandleSetFactory::create(0));

  // attributes according to RPR-FOM (SISO-STD-001.1-1999)
  attributeSet->add(mAttrLocation);
  attributeSet->add(mAttrOrientation);
  attributeSet->add(mAttrLinearVelocity);
  attributeSet->add(mAttrLinearAcceleration);
  attributeSet->add(mAttrAngularVelocity);
  // proprietary attributes
  attributeSet->add(mAttrVisualModel);

  mRtiAmbassador.requestObjectAttributeValueUpdate(object_handle, *attributeSet);
}

static RTI::ULong getAttributeIndex(const RTI::AttributeHandleValuePairSet &values, RTI::ParameterHandle handle)
{
  RTI::ULong idx = 0;
  while (idx < values.size() && values.getHandle(idx) != handle)
    idx++;

  return idx;
}

void FGHLA::reflectAttributes(
  RTI::ObjectHandle object_handle, const RTI::AttributeHandleValuePairSet &values)
{
  FGAIMultiplayer* player;

  MultiPlayerMap::iterator pos = mMultiPlayerMap.find(object_handle);
  if (pos == mMultiPlayerMap.end())
  {
    player = new FGAIMultiplayer;

    char* object_name = mRtiAmbassador.getObjectInstanceName(object_handle); 
    player->setCallSign(object_name);
    free(object_name);
  }
  else
    player = pos->second.get();

  FGExternalMotionData motionInfo;
  // initialize all values to zero
  memset(&motionInfo, 0, sizeof(motionInfo));

  SGTimeStamp timestamper;
  timestamper.stamp();
  // TODO: this should be simulation time
  motionInfo.time = timestamper.get_seconds();

  RTI::ULong locationIdx = getAttributeIndex(values, mAttrLocation);
  if (locationIdx < values.size())
  {
    RTI::ULong length;
    double* position = (double*)values.getValuePointer(locationIdx, length);

    for (unsigned i = 0; i < 3; ++i)
      motionInfo.position(i) = position[i]; // X,Y,Z [m]
  }
  else
    SG_LOG(SG_NETWORK,SG_ALERT,"FGHLA::reflectAttributes() mission location information");

  RTI::ULong orientationIdx = getAttributeIndex(values, mAttrOrientation);
  if (orientationIdx < values.size())
  {
    RTI::ULong length;
    float* orientation = (float*)values.getValuePointer(orientationIdx, length);

    SGGeod geodPosition = SGGeod::fromCart(motionInfo.position);
    // the rotation from the earth centered frame to the horizontal local frame
    SGQuatf qEc2Hl = SGQuatf::fromLonLatRad(geodPosition.getLongitudeRad(), geodPosition.getLatitudeRad());

    // the orientation of the body frame relative to the vehicle-carried NED frame
    SGQuatf hlOr = SGQuatf::fromYawPitchRoll(orientation[2], orientation[1], orientation[0]);
    // the orientation of the body frame relative the earth centered frame
    motionInfo.orientation = qEc2Hl*hlOr; // [rad]
  }
  else
    SG_LOG(SG_NETWORK,SG_ALERT,"FGHLA::reflectAttributes() mission orientation information");

  RTI::ULong linearVelocityIdx = getAttributeIndex(values, mAttrLinearVelocity);
  if (linearVelocityIdx < values.size())
  {
    RTI::ULong length;
    float* linearVel = (float*)values.getValuePointer(linearVelocityIdx, length);

    for (unsigned i = 0; i < 3; ++i)
      motionInfo.linearVel(i) = linearVel[i]; // U,V,W [m/s]
  }

  RTI::ULong angularVelocityIdx = getAttributeIndex(values, mAttrAngularVelocity);
  if (angularVelocityIdx < values.size())
  {
    RTI::ULong length;
    float* angularVel = (float*)values.getValuePointer(angularVelocityIdx, length);

    for (unsigned i = 0; i < 3; ++i)
      motionInfo.angularVel(i) = angularVel[i];
  }

  RTI::ULong linearAccelerationIdx = getAttributeIndex(values, mAttrLinearAcceleration);
  if (linearAccelerationIdx < values.size())
  {
    RTI::ULong length;
    float* linearAccel = (float*)values.getValuePointer(linearAccelerationIdx, length);

    for (unsigned i = 0; i < 3; ++i)
      motionInfo.linearAccel(i) = linearAccel[i]; // P,Q,R [rad/s]
  }

  RTI::ULong visualModelIdx = getAttributeIndex(values, mAttrVisualModel);
  if (visualModelIdx < values.size())
  {
    RTI::ULong length;
    char* value = (char*)values.getValuePointer(visualModelIdx, length);

    std::string modelName(value, length);
    player->setPath(modelName.c_str());
  }

  long stamp = timestamper.get_seconds();
  player->addMotionInfo(motionInfo, stamp);

  if (pos == mMultiPlayerMap.end())
  {
    mMultiPlayerMap[object_handle] = player;

    FGAIManager *aiMgr = (FGAIManager*)globals->get_subsystem("ai_model");
    if (aiMgr)
      aiMgr->attach(player);

    SG_LOG(SG_NETWORK,SG_INFO,"FGHLA::reflectAttributeValues() new player created");
  }
}

void FGHLA::removeObjectInstance(RTI::ObjectHandle object_handle, const RTI::FedTime &time,
  const char *event_tag, RTI::EventRetractionHandle retraction_handle)
throw (RTI::ObjectNotKnown, RTI::InvalidFederationTime, RTI::FederateInternalError)
{
  MultiPlayerMap::iterator pos = mMultiPlayerMap.find(object_handle);
  if (pos == mMultiPlayerMap.end())
  {
    SG_LOG(SG_NETWORK, SG_WARN, "FGHLA::removeObjectInstance() unknown object");
    return;
  }

  pos->second->setDie(true);
  mMultiPlayerMap.erase(object_handle);
}

void FGHLA::removeObjectInstance(RTI::ObjectHandle object_handle, const char *event_tag)
throw (RTI::ObjectNotKnown, RTI::FederateInternalError)
{
  MultiPlayerMap::iterator pos = mMultiPlayerMap.find(object_handle);
  if (pos == mMultiPlayerMap.end())
  {
    SG_LOG(SG_NETWORK, SG_WARN, "FGHLA::removeObjectInstance() unknown object");
    return;
  }

  pos->second->setDie(true);
  mMultiPlayerMap.erase(object_handle);
}

// end of file

