# Copyright 2009 Ben Escoto
#
# This file is part of Explicansubstr.

# Explicans is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# Explicans is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with Explicans.  If not, see <http://www.gnu.org/licenses/>.

"""extable.py - Code for the ExTable object"""

import objects, lazyarray

class ExTable(objects.ExObject):
	"""Two dimensional lazy table"""
	def __init__(self, row_axis_name, col_axis_name, rownames, colnames):
		"""Initialize with python lists/tuples of names of rows and columns
		
		Each element in row/col names should be explicans scalars (like
		ExString), not python objects.
		"""
		objects.ExObject.__init__(self, 'table', None)
		self.row_axis_name, self.col_axis_name = row_axis_name, col_axis_name
		self.rownames, self.colnames = rownames, colnames
		self.make_blank_values()
		self.memo_dict = {} # dictionary to hold memoized values
		self.row_axis_flag, self.col_axis_flag = self.get_axis_flags()

	def make_blank_values(self):
		"""Set self.values to return ExBlanks"""
		def get_blank(): return objects.ExBlank()
		rownum, colnum = self.get_size()
		self.values = [[None]*self.get_num_cols()
					   for i in range(self.get_num_rows())]
		for i in range(rownum):
			for j in range(colnum):
				self.set_value_thunk(i, j, get_blank)

	def set_value_thunk(self, i, j, thunk):
		"""Set the value thunk for row i, column j"""
		val_list = []
		def memoized_thunk():
			if val_list: return val_list[0]
			val = thunk()
			val_list.append(val)
			return val
		self.values[i][j] = memoized_thunk

	def get_row_name(self, i): return self.rownames[i]
	def get_col_name(self, i): return self.colnames[i]
	def get_value_thunk(self, i,j): return self.values[i][j]

	def get_axis_flags(self):
		"""Return (row_axis_flag, col_axis_flag)
		
		These flags are stuck onto other values so that relative reference
		functions like next(foo) can pick out specific axis foo.
		
		"""
		# FIXME: replace the below with something usable outside the table
		return (objects.ExBlank(), objects.ExBlank())

	def get_size(self):
		"""Return (number of rows, number of columns) in self"""
		return (len(self.rownames), len(self.colnames))
	def get_num_rows(self): return len(self.rownames)
	def get_num_cols(self): return len(self.colnames)
	
	def __getitem__(self, name):
		"""Return an ExArray corresponding to the given row or column name"""
		try: j = self.colnames.index(name)
		except ValueError:
			return self.row_to_exarray(self.rownames.index(name))
		return self.col_to_exarray(j)

	def col_to_exarray(self, j):
		"""Return the thunks in column j as an ExArray"""
		rownum = self.get_num_rows()
		la = lazyarray.LazyArray(rownum)
		la.set_keys(self.rownames)
		for i in range(rownum):
			la.set_value(i, self.get_value_thunk(i,j))
		return objects.ExArray(la)

	def row_to_exarray(self, i):
		"""Return the thunks in row i as an ExArray"""
		colnum = self.get_num_cols()
		la = lazyarray.LazyArray(colnum)
		la.set_keys(self.colnames)
		for j in range(colnum):
			la.set_value(j, self.get_value_thunk(i, j))
		return objects.ExArray(la)


def make_blank_table(rowcount, colcount):
	"""Create a new blank table with given ExNum of rows and columns"""
	py_rownum, py_colnum = int(rowcount.obj), int(colcount.obj)
	return ExTable('', '', [objects.Blank]*py_rownum, [objects.Blank]*py_colnum)
