"""testmodule.py - some functions useful in the Explicans test suite"""

import objects, extable

class Table:
	def __init__(self, rownames, colnames, values):
		"""Initialize with rows.  Values should just be a big tuple of tuples"""
		self.rownames = tuple(rownames)
		self.colnames = tuple(colnames)
		self.values = tuple(values)
		assert len(values) == len(rownames), (values, rownames)
		for row in self.values:
			assert len(row) == len(colnames), (values, colnames)

	def __eq__(self, other):
		if not isinstance(other, Table): return False
		if (self.rownames != other.rownames or self.colnames != other.colnames):
			return False
		return self.values == other.values
		
	def __str__(self):
		l = ['Table: rownames = %s' % (self.rownames,),
			 '       colnames = %s' % (self.colnames,),
			 '       values = %s' % (self.values,)]
		return '\n'.join(l)

	def __repr__(self): return str(self)

T = Table((1,2), (3,4), ((1,2), (3,4)))
assert T.values == ((1,2), (3,4))
assert T == Table((1,2), (3,4), ((1,2), (3,4)))
assert T != Table((1,2), (3,5), ((1,2), (3,4)))

def result_to_tuple(result):
	"""Convert an Explicans object to a python checkable object recursively

	This is used when you need to compare an Explicans object with its python
	equivalent.

	"""
	assert isinstance(result, objects.ExObject), result
	if result.isscalar(): return result.obj
	elif isinstance(result, objects.ExArray):
		return tuple(result_to_tuple(thunk()) for thunk in result.obj.values)
	assert isinstance(result, extable.ExTable), result

	rownames = tuple(result_to_tuple(rowname) for rowname in result.rownames)
	colnames = tuple(result_to_tuple(colname) for colname in result.colnames)
	values = []
	for value_row in result.values:
		row_tuple = tuple(result_to_tuple(thunk()) for thunk in value_row)
		values.append(row_tuple)
	return Table(rownames, colnames, tuple(values))