;;; "fresneleq.scm" Solve EM waves in parallel layers of dielectrics and metals -*-scheme-*-
;;; Copyright (C) 2003, 2004 Aubrey Jaffer

;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation, either version 3 of the License, or
;;; (at your option) any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.

;;; I can't find a free implementation of thin-film optical
;;; calculations.  So I rolled my own.

;; http://swiss.csail.mit.edu/~jaffer/FreeSnell

;; Optics is all wavelength based; so this is also.  These routines
;; work out the complex voltages in the forward and reverse directions
;; to find the transmitted and reflected amplitudes.  This works only
;; for intensities where the layers act linearly (superposition).
;; Each layer 0:n has an index of refraction and height.

;; Square the absolute value of numbers returned to get the power
;; ratios.

;; Because the P and S polarizations are independent, calculate them
;; separately.

;;; Fresnel's equations from
;;; http://chsfpc5.chem.ncsu.edu/CH795Z/lecture/lecture8/fresnel/fresnel.html
;;; Altered signs to match
;;; http://hyperphysics.phy-astr.gsu.edu/hbase/phyopt/freseq.html

;;; Rewrote using matrix method from
;;; http://www.ifm.liu.se/~boser/elma/Lect13.pdf

;;; The transmitted voltage (E-Field)
(define (E_T n1 n2 cos-i cos-t s-polarization?)
  (if s-polarization?
      (/ (* 2 n1 cos-i)
	 (+ (* n1 cos-i) (* n2 cos-t)))
      (/ (* 2 n1 cos-i)
	 (+ (* n1 cos-t) (* n2 cos-i)))))
;;; The reflected voltage (E-Field)
(define (E_R n1 n2 cos-i cos-t s-polarization?)
  (if s-polarization?
      (/ (- (* n1 cos-i) (* n2 cos-t))
	 (+ (* n1 cos-i) (* n2 cos-t)))
      (/ (- (* n2 cos-i) (* n1 cos-t))
	 (+ (* n2 cos-i) (* n1 cos-t)))))

;;; Returns 2x2 matrix:
;;;
;;;            (   1      r      )
;;;    1       (           n-1,n )
;;; -------- * (                 )
;;;  t         ( r          1    )
;;;   n-1,n    (  n-1,n          )
(define (layer-interface n1 n2 th1 s-polarization?)
  (define cos-i (cos th1))
  (define cos-t (cos (Snell-law n1 n2 th1)))
  (let ((transmit (E_T n1 n2 cos-i cos-t s-polarization?))
	(reflect (E_R n1 n2 cos-i cos-t s-polarization?)))
    (let ((r/t (/ reflect transmit))
	  (tinv (/ transmit)))
      (list (list tinv r/t) (list r/t tinv)))))

;;; Returns 2x2 matrix coding phase difference between reflected and
;;; transmitted paths.
;;;
;;;  (  -i*d_n    0    )
;;;  ( e               )
;;;  (           i*d_n )
;;;  (    0     e      )
(define (layer-phase h_j n_j th_j w)
  (define phase (exp (/ (* +2i pi h_j n_j (cos th_j)) w)))
  (list (list (/ phase) 0) (list 0 phase)))

;;; LAYERS are lists: (index-of-refraction height).  The
;;; index-of-refraction may be a complex number or a procedure of
;;; wavelength (W) returning a complex number.
;;;
;;; IR_0 is the index-of-refraction of the medium on the top of the
;;; stack of LAYERS.  The bottom layer has thickness 0.
;;; W is probe wavelength.
;;; TH_I is the angle of the incident ray.
;;;
;;; `combine-layers' returns a list of the transmitted and forward and
;;; reverse reflected power ratios.  The transmitted and each
;;; reflected power ratio sum to 1.0 if stack is all dielectric
;;; (lossless)
;;;
;;; Negative W computes the S-polarization, else the P-polarization.
(define (combine-layers th_i w layers)
  (define (numberize x) (if (procedure? x) (x w) x))
  (define IR_0 (caar layers))
  (define (add-next-0-layer stack)
    (cons (cons (caadr stack) (cons 0 (cddadr stack)))
	  (cdr stack)))
  (set! IR_0 (numberize IR_0))
  (let ((th_t th_i)
	(IR_n #f)
	(M '((1 0) (0 1)))
	(M2 '((1 0) (0 1)))
	(s-polarization? (negative? w)))
    (define (loop stack IR_n-1)
      (cond
       ((null? stack)
	;;(check-determinant M2)
	(matrix->powers (matrix-product
			 M2 (map (lambda (row) (map squmag row)) M))
			(abs (/ (real-part IR_n)
				(real-part IR_0)))))
       (else
	(let ((layer (car stack)))
	  (set! IR_n (numberize (car layer)))
	  (cond
	   ((negative? (cadr layer))	; layer*
	    (set! M2
		  (matrix-product
		   M2 (map (lambda (row) (map squmag row)) M)))
	    (set! th_t (Snell-law IR_n-1 IR_n th_t))
	    (set! M2
		  (matrix-product
		   M2 (map (lambda (row) (map squmag row))
			   (layer-phase (abs (cadr layer)) IR_n th_t w))))
	    (set! M '((1 0) (0 1)))
	    (loop (if (negative? (cadadr stack)) ; thickness of next layer
		      ;; then add 0-thickness layer with next IR
		      (add-next-0-layer stack)
		      ;; otherwise, next layer.
		      (cdr stack))
		  IR_n))
	   (else			; layer
	    (set! M (matrix-product
		     M (layer-interface IR_n-1 IR_n th_t s-polarization?)))
	    (set! th_t (Snell-law IR_n-1 IR_n th_t))
	    (set! M (matrix-product M (layer-phase (cadr layer) IR_n th_t w)))
	    (loop (if (and (positive? (cadr layer))
			   (negative? (cadadr stack)))
		      (add-next-0-layer stack)
		      (cdr stack))
		  IR_n)))))))
    (set! w (abs w))
    (loop (if (negative? (cadadr layers))
	      (add-next-0-layer layers)
	      (cdr layers))
	  IR_0)))

(define (matrix->powers M IR_ratio)
  (list (/ IR_ratio (caar M))
	(/ (caadr M) (caar M))
	(/ (cadar M) (caar M))))

(define (matrix-product mat1 mat2)
  (map (lambda (arow)
	 (apply map
		(lambda bcol (apply + (map * bcol arow)))
		mat2))
       mat1))

(define (check-determinant M)
  (let ((det (- (* (caar M) (cadadr M)) (* (cadar M) (caadr M)))))
    (if (and (not (< .99 (real-part det) 1.01))
	     (not (< -.01 (imag-part det) +.01)))
	(slib:warn 'determinant det M)))
  M)

;;; Square of the magnitude
(define (squmag x)
  (define mag (magnitude x))
  (* mag mag))

;;; Maxwell Garnett theory from p178 of
;;;	Heavens, O. S.
;;;	Optical Properties of Thin Solid Films
;;;      ISBN: 0486669246, Dover Pubns, Dec 1991
;;;
;;; q is the volume fraction of spheres of index IR in a medium of
;;; refractive index 1.
;;(define (granular-IR IR q)
;;  (define n (* IR IR))
;;  (sqrt (/ (+ (- -2 n) (* (- 2 (* 2 n)) q))
;;	   (+ (- -2 n) (* (+ -1 n) q))))) ; jacal solved

;; Refractive index of 1 is unrealistic.  Change the dielectric
;; constant of vacuum to n_s, rederive, then divide answer by n_s.
(define (granular-IR IR q IR0)
  (define n (* IR IR))
  (define ns (* IR0 IR0))
  (sqrt (* ns (/ (+ n (* 2 ns) (* -2 (- ns n) q))
		 (+ n (* 2 ns) (* (- ns n) q))))))

;; Given the angle of impinging light,
;; returns the angle of the transmitted light
(define (Snell-law n1 n2 th-i)
  (asin (* (/ n1 n2) (sin th-i))))

;; Given the angle of impinging light,
;; returns the angle of the transmitted light; even for metal.
(define (Snells-law n1 n2 th-i)
  (define sin-th (real-sin th-i))
  (define nrat (/ n2 n1))
  (real-atan (/ sin-th
		(real-part (sqrt (- (* nrat nrat) (* sin-th sin-th)))))))

(define (find-angles th_0 layers)
  (do ((layers layers (cdr layers))
       (angles (list th_0)
	       (cons (Snells-law (caar layers) (caadr layers) (car angles))
		     angles)))
      ((null? (cdr layers)) (reverse angles))))
;;(trace-all "fresneleq.scm") (set! *qp-width* 333)
