
(require 'FreeSnell)
(require 'databases)
(require 'database-interpolate)
(require 'eps-graph)
(require 'printf)

(define (HDPE-BE w)
  (define x (min 1 (max 0 (/ (- w 2.5e-6) 10e-6))))
  (make-rectangular
   (+ (* 1.8 (- 1 x)) (* 1.54 x))
   (* 0.002 (/ (max 0 (- w 2.5e-6)) 12.5e-6))))

(define (HDPE-fun w)
  (define x (min 1 (max 0 (/ (- w 2.5e-6) 10e-6))))
  (make-rectangular
   (+ (* 1.8 (- 1 x)) (* 1.54 x))
   (* 0.002 (/ (max 0 (- w 2.5e-6)) 12.5e-6))))

;;; No longer used.
(define (make-hdpe.nk)
  (close-database nk)
  (call-with-output-file "hdpe.nk"
    (lambda (port)
      (define (doline lst)
	(let ((nk (HDPE-fun (/ .01 (car lst)))))
	  (fprintf port "%07.2f		%g		%g\n"
		   (car lst)
		   (real-part nk)
		   (case (length lst)
		     ((1) (imag-part nk))
		     ((2) (cadr lst))))))
      (fprintf port "     /cm	n		k	; High Density Polyethylene\n")
      (for-each doline
		'(
		  (0500)
		  (0718)
		  (0723.30 0.05)
		  (0734    0.025)
		  (0749.51 0.04)
		  (0755)
		  (1000)
		  (1282)
		  (1299    0.0022)
		  (1333    0.0013)
		  (1370    0.006)
		  (1374    0.004)
		  (1461)
		  (1466.02 0.044)
		  (1490    0.028)
		  (1492.23 0.046)
		  (1497)
		  (2000)
		  (2500)
		  (2750    0.001)
		  (2815    0.0007)
		  (2855.34 0.06)
		  (2865    0.005)
		  (2916.50 0.06)
		  (2921)
		  (3000)
		  (3500)
		  (3550)
		  (3600    0.0003)
		  (4000    0.0003)
		  ))))

  (system "nk --del hdpe")
  (system "nk --add hdpe.nk")
  (set! nk (open-database (or (getenv "NK_DATABASE_PATH") "nk.rwb") 'rwb-isam))
  (set! HDPE (interpolate-from-table (open-table nk 'HDPE) 2)))

(defvar nk (open-database (or (getenv "NK_DATABASE_PATH") "nk.rwb") 'rwb-isam))
(defvar HDPE (interpolate-from-table (open-table nk 'HDPE) 2))

(define (hdpe-const)
  (define HDPE 1.54+0.002i)
  (plot-response
   (title "50.um Polyethylene Film;  n = 1.54+0.002i" "hdpe-const")
   (output-format 'png 600 184)
   (font 'Times 13 "-15.3" "")
   (samples 1200)
   (wavelengths 2.5e-6 25e-6)
   (smooth 0.05e-6)
   (incident 0 'T 'R)
   (range 0 1)
   (optical-stack
    (nominal 11e-6)
    (substrate 1)
    (layer HDPE 50e-6)
    (substrate 1))))

(define (hdpe-L100)
  (plot-response
   (title "100.um Polyethylene Film; variable n" "hdpe-L100")
   (output-format 'png 300 200)
   (font 'Times 11 "-15.3" "")
   (samples 600)
   (wavelengths 2e-6 20e-6 0e-6 20e-6)
   (smooth 0.05e-6)
   (incident 0 'T 'R)
   (range 0 1)
   ;;(IR HDPE-BE'real 'imag)
   (optical-stack
    (nominal 11e-6)
    (substrate 1)
    (layer HDPE-BE 100e-6)
    (substrate 1))
   (optical-stack
    (nominal 11e-6)
    (substrate 1)
    (layer* HDPE-BE 100e-6)
    (substrate 1))))

(define (hdpe-L33)
  (plot-response
   (title "33.um Polyethylene Film; variable n" "hdpe-L33")
   (output-format 'png 300 200)
   (font 'Times 11 "-15.3" "")
   (samples 600)
   (wavelengths 2e-6 20e-6 0e-6 20e-6)
   (smooth 0.05e-6)
   (incident 0 'T 'R)
   (range 0 1)
   ;;(IR HDPE-BE'real 'imag)
   (optical-stack
    (nominal 11e-6)
    (substrate 1)
    (layer HDPE-BE 33e-6)
    (substrate 1))
   (optical-stack
    (nominal 11e-6)
    (substrate 1)
    (layer* HDPE-BE 33e-6)
    (substrate 1))))

(define (polyeth-35)
  (define thk 35e-6)
  (plot-response
   (title (sprintf #f "35.um Polyethylene")
	  (sprintf #f "polyeth-35"))
   (output-format 'png 490 360)
   (font 14 "-450" "")
   (samples 1000)
   (range 0 1)
   (wavenumbers 4000 400)
   (incident 0 'T)
   (optical-stack
    (substrate 1)
    (layer HDPE thk)
    (substrate 1))
   (optical-stack
    (substrate 1)
    (layer* HDPE thk)
    (substrate 1))))

;;; Trombe, F.
;;; Devices for Lowering the Temperature of a Body by Heat Radiation Therefrom,
;;; US Patent 3,310,102 issued Mar. 21, 1967.
;;; http://patimg1.uspto.gov/.piw?Docid=03310102&idkey=NONE
(define (hdpe-50)
  (plot-response
   (title "50.um Polyethylene Film" "hdpe-50")
   (output-format 'png 800 245)
   (font 'Times 10 "-15.3" "")
   (samples 1200)
   (wavelengths 0.5e-6 15e-6)
   (smooth 0.05e-6)
   (incident 0 'T)
   (range 0 1)
   (optical-stack
    (nominal 11e-6)
    (substrate 1)
    (layer HDPE 50e-6)
    (substrate 1))))

;;; Martin, M.,
;;; Radiative Cooling,
;;; In Passive Cooling, edited by Jeffrey Cook.
;;; MIT Press, 1989, pp. 138-196.
;;; ISBN: 0262531712
;;;
;;; Citing:
;;; Conley, R. T. 1972
;;; "Infrared Spectroscopy", Second Edition.
;;; Boston, MA: Allyn and Bacon, Inc., p.266
(define (hdpe-12.7)
  (define str "12.7")
  (define thk 12.7e-6)
  (define smp 400)
  (plot-response
   (title (sprintf #f "%s.um High Density Polyethylene Film" str)
	  (sprintf #f "hdpe-%s" str))
   (output-format 'png 428 168)
   (font 'Times 13 "-15.3" "")
   ;;(IR HDPE 'real 'imag)
   (samples smp)
   ;;(smooth 0.025e-6)
   (range 0 1)
   (wavelengths 2e-6 15e-6)
   (incident 0 'T)
   (optical-stack
    (nominal 11e-6)
    (substrate 1)
    (layer* HDPE thk)
    (substrate 1))))

;;; C. G. Granqvist and A. Hjortsberg.
;;; Radiative cooling to low temperatures: General considerations and
;;; application to selectively emitting SiO films
;;; Journal of Applied Physics Vol <B>52</B>(6) pp. 4205-4220. June 1981.
;;; http://scitation.aip.org/getabs/servlet/GetabsServlet?prog=normal&amp;id=JAPIAU000052000006004205000001&amp;idtype=cvips&amp;gifs=yes
(define (hdpe-14)
  (define str "14")
  (define thk 14e-6)
  (define smp 400)
  (plot-response
   (title (sprintf #f "%s.um Polyethylene Film" str)
	  (sprintf #f "hdpe-%s" str))
   (output-format 'png 342 310)
   (font 'Times 10 "5.3" "")
   (samples smp)
   ;;(smooth 0.05e-6)
   (range 0 1)
   (wavelengths 4.5e-6 26e-6 3e-6 27e-6)
   (incident 0 'T)
   (optical-stack
    (nominal 11e-6)
    (substrate 1)
    (layer HDPE thk)
    (substrate 1))
   (optical-stack
    (nominal 11e-6)
    (substrate 1)
    ;;(layer* HDPE thk)
    ;;(layer* 1 1e-3)
    (layer HDPE thk)
    ;;(layer* 1 1e-3)
    ;;(layer* HDPE thk)
    (substrate 1))))

;;; http://www.mtecpas.com/Pub2/Pgtitle.html
;;; "A Practical Guide to FTIR Photoacoustic Spectroscopy" from MTEC
;;; Photoacoustics.
(define (hdpe-ftir)
  (plot-response
   (title (sprintf #f "Polyethylene FTIR")
	  (sprintf #f "hdpe-ftir"))
   (output-format 'png 425 265)
   (font 14 "-450" "")
   (samples 1000)
   (range 0 1)
   (wavenumbers 4000 450)
   (smooth 10)
   (incident 0 'T)
   (optical-stack
    (nominal 11e-6)
    (substrate 1)
    (layer HDPE 32e-6)
    (substrate 1))))
(define (hdpe-pas)
  (plot-response
   (title (sprintf #f "Polyethylene PAS")
	  (sprintf #f "hdpe-pas"))
   (output-format 'png 425 265)
   (font 14 "-450" "")
   (samples 1000)
   (range 0 1)
   (wavenumbers 4000 450)
   (incident 0 '(- 1 A (/ R 2)))
   (optical-stack
    (nominal 11e-6)
    (substrate 1)
    (layer* HDPE 32e-6)
    (substrate 1))))

;;; Mastai, Y., Diamant, Y., Aruna, S.T., and Zaban, A.
;;; TiO2 Nanocrystalline Pigmented Polyethylene Foils for Radiative Cooling Applications: Synthesis and Characterization,
;;; Langmuir,17, 22, 7118, 7123, 2001, 10.1021/la010370g.
;;; http://pubs.acs.org/cgi-bin/abstract.cgi/langd5/2001/17/i22/abs/la010370g.html
;;; Figure 5
(define (hdpe-33)
  (plot-response
   (title "33.um High Density Polyethylene foil" "hdpe-33")
   (output-format 'png 300 200)
   (font 'Times 11 "-15.3" "")
   (samples 1200)
   (wavelengths 2e-6 20e-6 0e-6 20e-6)
   (smooth 0.06e-6)
   (incident 0 'T 'R)
   (range 0 1)
   (optical-stack
    (nominal 11e-6)
    (substrate 1)
    (layer hdpe 33e-6)
    (substrate 1))))

(define (hdpe-33ni)
  (plot-response
   (title "33.um High Density Polyethylene foil" "hdpe-33ni")
   (output-format 'png 300 200)
   (font 'Times 11 "-15.3" "")
   (samples 1200)
   (wavelengths 2e-6 20e-6 0e-6 20e-6)
   ;;(smooth 0.01e-6)
   (incident 0 'T 'R)
   (range 0 1)
   (optical-stack
    (nominal 11e-6)
    (substrate 1)
    (layer* hdpe 33e-6)
    (substrate 1))))

;;; P. T. Tsilingiris,
;;; Comparative evaluation of the infrared transmission of polymer films
;;; Energy Conversion and Management, Volume 44, Issue 18, November
;;; 2003, Pages 2839-2856
;;; http://www.sciencedirect.com/science?_ob=GatewayURL&_method=citationSearch&_uoikey=B6V2P-489YNWC-4&_origin=SDEMFRHTML&_version=1&md5=e3a64a1c728927336cf00e02450a8edb
(define (hdpe-130)
  (plot-response
   (title (sprintf #f "130.um Polyethylene Film")
	  (sprintf #f "hdpe-130"))
   (output-format 'png 325 317)
   (font 'Times 12 "-15.3" "")
   (samples 1000)
   (range 0 1)
   (wavelengths 2.5e-6 14.9e-6)
   ;;(smooth 0.05e-6)
   (incident 0 'T)
   (optical-stack
    (nominal 11e-6)
    (substrate 1)
    (layer* HDPE 130e-6)
    (substrate 1))))

;;; http://www.chem.arizona.edu/courses/chem245/polyeth.html
;;; The IR Card Spectrum</A> from The University of Arizona details
;;; three sets of peak resulting from polyethylene C-H and C-C
;;; absorption in the IR region.
(define (hdpe-card)
  (plot-response
   (title (sprintf #f "Polyethylene Card")
	  (sprintf #f "hdpe-card"))
   (output-format 'png 640 442)
   (font 14 "-12345.0" "")
   (samples 2000)
   (range 0 1)
   (wavenumbers 4000 400)
   ;;(smooth 10)
   (incident 0 'T)
   (optical-stack
    (nominal 11e-6)
    (substrate 1)
    (layer* HDPE 14e-6)
    (substrate 1))))

;;; "PTFE &amp; Polyethylene IR Sample Cards"
;;; http://www.internationalcrystal.net/polycard.htm
;;; http://www.internationalcrystal.net/Figure2a.gif
(define (hdpe-card-absorb)
  (plot-response
   (title (sprintf #f "Polyethylene Card Absorption")
	  (sprintf #f "hdpe-card-absorb"))
   (output-format 'png 365 215)
   (font 12 "-15" "")
   ;;(range 0 3.29)
   (range 0 4)
   (samples 600)
   (logarithmic (/ .01 5500) (/ .01 550))
   ;;(wavenumbers 4000 400)
   ;;(smooth 25)
   (incident 0 '(- (ln T)))
   (optical-stack
    (nominal 11e-6)
    (substrate 1)
    (layer* HDPE 18e-6)
    (substrate 1))))

;;; Organic Analysis - IR Theory from California Polytechnic State
;;; University
;;; http://chemweb.calpoly.edu/chem/124/124Experiments/OrgAnal/IRTheory.html
(define (hdpe-pe)
  (plot-response
   (title (sprintf #f "25.4.um HDPE")
	  (sprintf #f "hdpe-pe"))
   (output-format 'png 745 255)		; 680 455
   (font 14 "-945.0" "")
   (samples 800)
   (range 0 1)
   ;;(range -.0728 .8278)
   ;;(wavenumber 4200 600)
   (marker (/ .01 4000) (/ .01 2000) (/ .01 1000))
   (logarithmic (/ .01 4250) (/ .01 660)) ; 760
   ;;(smooth .004)
   (incident 0 'T)
   (optical-stack
    (nominal 11e-6)
    (substrate 1)
    (layer HDPE 25.4e-6)
    (substrate 1))))

;;; http://www.bgtu.net/eng/ik/
;;; IR-Spectroscopy and FT-IR-Spectroscopy from the Belarussian State
;;; Technological University has "IR-Spectrums of polyethylene at
;;; various stages of ageing":
(define (hdpe-age-1)
  (plot-response
   (title "100.um HDPE" "hdpe-age-1")
   (output-format 'png 625 395)
   (font 14 "45.0" "")
   (samples 1200)
   (range 0 1)
   (wavenumbers 4000 200)
   (smooth 5)
   (incident 0 'T)
   (optical-stack
    (nominal 11e-6)
    (substrate 1)
    (layer HDPE 100e-6)
    (substrate 1))))

(define (polyethylene)
  (hdpe-const)
  (hdpe-card)
  (hdpe-card-absorb)
  (hdpe-pe)
  (hdpe-age-1)
  (hdpe-L100)
  (hdpe-ftir)
  (hdpe-pas)
  (hdpe-L33)
  (polyeth-35)
  (hdpe-50)
  (hdpe-12.7)
  (hdpe-14)
  (hdpe-33)
  (hdpe-33ni)
  (hdpe-130))

;;(polyethylene)
