/*
	This file is part of jrisk.
	
	jrisk is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.
	
	jrisk is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
	
	You should have received a copy of the GNU General Public License
	along with jrisk; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
 * Created on 15-feb-2005 by Alessio Treglia
 * Copyright  2005 Alessio Treglia
 */
import java.util.List;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Collections;

/**
 *
 * <p>Riproduce il comportamento di un lancio simultaneo di pi dadi.
 * Memorizza i riferimenti agli oggetti Dado (e dunque i risultati
 * dei lanci) in una collezione dinamica di tipo List.</p>
 * 
 * <p>La serie viene poi ordinata in modo non-crescente, cos da facilitare
 * l'operazione di confronto con precedenza ai dadi di valore maggiore.</p>
 * 
 * @author Alessio Treglia
 * @version 1.0.0
 * @see Battaglia
 * */
public class SerieDiDadi {
    /** Dadi lanciati */
    private List dadi;
    
    /**
     * Costruttore no-arg. Alloca memoria per il lancio
     * dei dadi.
     *
     */
    public SerieDiDadi() {
        dadi = new ArrayList();
    }
    
    /**
     * Costruisce l'oggetto ed effettua i primi
     * n lanci.
     * 
     * @param n numero di dadi da lanciare
     */
    public SerieDiDadi(int n) {
        dadi = new ArrayList();						// Instanzia una collezione di oggetti (Dado)
        lanciaDadi(n);								// Lancia i dadi
    }
    
    /**
     * Lancia gli n dadi desiderati e li ordina
     * in modo non-crescente. Prima di ogni lancio
     * svuota la collezione di dadi e la re-inizializza
     * a seconda del parametro di dadi da lanciare. Infine,
     * ordina la collezione di dadi in modo non-crescente.
     * 
     * @param n numero di dadi da lanciare
     */
    public void lanciaDadi(int n) {
        Dado dadoLanciato;
        
        dadi.clear();
        
        for(int i=0; i<n; i++) {
            dadoLanciato = new Dado();
            dadi.add(dadoLanciato);
        }
        /*
         * Richiede di ordinare la collezione secondo l'ordine
         * inverso di quello naturale
         */
        Collections.sort(dadi, Collections.reverseOrder());
    }
    
    /**
     * @return il numero dei dadi lanciati
     */
    public int getNumeroDadi() {
        return this.dadi.size();					// Restituisce la dimensione della collezione di dadi
    }
    /**
     * @return una List contenente i dadi lanciati
     */
    public List getDadi() {
        return dadi;
    }
    /**
     * Restituisce l'elemento della lista di posizione i
     * 
     * @param i indice dell'elemento da restituire
     * @return l'elemento di posizione i, <code>null</code> in caso di errore
     */
    public Dado getDado(int i) {
        if(i<dadi.size())
            return (Dado) dadi.get(i);
        return null;
    }

    /**
     * Restituisce una stringa contenente i risultati dei dadi
     * della serie.
     * 
     * @return una visualizzazione testuale della serie di dadi
     * lanciati
     */
    public String toString() {
        Iterator it = dadi.iterator();
        StringBuffer buffer = new StringBuffer();
        Dado temp;
        while(it.hasNext()) {
            temp = (Dado) it.next();
            buffer.append(temp + "\t");
        }
        return buffer.toString();
    }
}
