/*
	This file is part of jrisk.

	jrisk is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.
	
	jrisk is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
	
	You should have received a copy of the GNU General Public License
	along with jrisk; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
 * Created on 15-feb-2005 by Alessio Treglia
 * Copyright  2005 Alessio Treglia
 */

import java.io.IOException;

/**
 * Modella un comando che consente al giocatore di attaccare
 * un avversario, scegliendo un proprio territorio come base
 * di partenza e uno dell'avversario come punto di attacco.
 * 
 * @author Alessio Treglia
 * @version 1.0.0
 */
public class ComandoAttacca extends Comando {
    /** Contiene il nome del comando */
    private final String nomeComando = "attacca";
    /** Territorio dal quale far partire l'attacco */
    private String partenza;
    /** Territorio da attaccare */
    private String destinazione;
    /**
     * Costruisce il comando
     * 
     * @param da territorio di partenza
     * @param a territorio di destinazione
     */
    public ComandoAttacca(String da, String a) {
        this.partenza = da;
        this.destinazione = a;
        
    }

    /* (non-Javadoc)
     * @see Comando#getNomeComando()
     */
    public String getNomeComando() { return this.nomeComando; }

    /* (non-Javadoc)
     * @see Comando#esegui(Giocatore, java.util.Map)
     */
    public boolean esegui(Giocatore giocatore, Tabellone planisfero)
            throws IOException {
        Giocatore g1, g2;
        Territorio da, a;
        Battaglia battaglia;
        int esitoScontro;
        int nDadiAttacco, nDadiDifesa;
        int nArmate;
        
        /*
         * Controllo che il nome degli stati sia giusto
         */
        if(!ElencoTerritoriContinenti.territorioValido(partenza) ||
                !ElencoTerritoriContinenti.territorioValido(destinazione))
            return new ComandoNonValido().esegui(giocatore, planisfero);
        
        g1 = (Giocatore) planisfero.getGiocatoreAt(partenza);
        g2 = (Giocatore) planisfero.getGiocatoreAt(destinazione);
        
        /*
         * Si accerta che il giocatore attacchi da un proprio
         * territorio ad un altro avversario
         */
        nDadiAttacco = 0;
                
        if(giocatore == g1 && giocatore != g2) {
            da = g1.getTerritorio(partenza);
            a = g2.getTerritorio(destinazione);
            /*
             * si accerta che i territori siano confinanti
             */
            if(!da.confinaCon(a)) {
                g1.schermo.stampaln("Possono essere attaccati solo i territori confinanti");
                return false;
            }
            
            /*
             * Si accerta che il territorio da cui parte l'attacco
             * possiede almeno due armate. Calcola il numero di armate
             * dell'attacco 
             */
            if(da.puoAttaccare()) {
                g1.schermo.stampa("Con quante armate attacchi? (1 - 3) ");
                nDadiAttacco = g1.parser.getInt();
                /* passa il turno */
                if(nDadiAttacco == 0)
                    return false;
                if(!da.puoAttaccare(nDadiAttacco))
                    nDadiAttacco = da.maxDadiAttacco();
            } else {
                g1.schermo.stampaln("Non puoi attaccare da " + partenza);
                return false;
            }
            
            /*
             * Calcola il numero di dadi per la difesa.
             */
            g2.schermo.stampaln("L'avversario ti attacca con " + nDadiAttacco + " armate.");
            g2.schermo.stampa("Con quante armate ti difendi? (1 - 3) ");
            nDadiDifesa = g2.parser.getInt();
            if((nDadiDifesa<1) || (nDadiDifesa > a.maxDadiDifesa()))
                nDadiDifesa = a.maxDadiDifesa();
            g1.schermo.stampaln("Il difensore difende con " + nDadiDifesa + " dado/i.");
            /*
             * Inizializza le variabili di battaglia
             */
            battaglia = new Battaglia(g1,g2,da.getNome(),a.getNome());
            /*
             * Combattimento vero e proprio
             */
            esitoScontro = battaglia.scontro(nDadiAttacco, nDadiDifesa);
            /*
             * Stampa i lanci dei dadi
             */
            g1.schermo.stampaln(battaglia.getLancioAttacco().toString());
            g2.schermo.stampaln(battaglia.getLancioDifesa().toString());
            /*
             * Se l'attaccante ha vinto il territorio
             */
            if(esitoScontro > 0) {
                /* Stampa un messaggio di vittoria */
                g1.schermo.stampaln(g1.getColore() + " ha conquistato " + destinazione);
                /* Elimina il territorio dalla collezione dei territori del difensore
                 * e aggiungilo alla collezione di territori dell'attaccante.
                 */
                g1.aggiungiTerritorio(g2.rimuoviTerritorio(a.getNome()));
                /* Aggiorna il tabellone di gioco */
                aggiornaPlanisfero(planisfero,a,g1);
                /* Fai lo spostamento minimo di attacco */
                spostamentoPostAttacco(esitoScontro,da,a);                
                /* Fai lo spostamento opzionale */
                g1.schermo.stampaln("Vuoi spostare altre armate sul territorio conquistato? ");
                String risp = g1.parser.getParola();
                if(risp.equals("si")) {
                    new ComandoSposta(partenza,destinazione).esegui(g1,planisfero);
                }
                /* Se l'avversario ha perso, dammi tutte le sue carte */
                if(g2.eliminato())
                    g1.aggiungiCarte(g2.getCarte());
            }
            if(esitoScontro < 0) {
                /* Stampa un messaggio di vittoria */
                g2.schermo.stampaln(g2.getColore() + " ha conquistato " + partenza);
                /* Aggiungi il territorio conquistato alla collezione dei territori
                 * del difensore, togliendolo dalla colleziona dei territori dell'
                 * attaccante
                 */
                g2.aggiungiTerritorio(g1.rimuoviTerritorio(da.getNome()));
                /* aggiorna il tabellone */
                aggiornaPlanisfero(planisfero,da,g2);
                /* Fai lo spostamento minimo */
                spostamentoPostAttacco(1,a,da);
                /* Fai lo spostamento opzionale */
                g1.schermo.stampaln("Vuoi spostare altre armate sul territorio conquistato? ");
                String risp = g2.parser.getParola();
                if(risp.equals("si")) {
                    return new ComandoSposta(destinazione,partenza).esegui(g2,planisfero);
                }
                /* Se l'avversario ha perso, dammi tutte le sue carte */
                if(g1.eliminato())
                    g2.aggiungiCarte(g1.getCarte());
            }
        } else {
            g1.schermo.stampa("Non puoi attaccare un tuo territorio!\n");
        }
        return false;
    }
    /**
     * <p>Esegue lo spostamento delle armate dopo l'attacco.
     * L'attaccante deve spostare sul territorio conquistato
     * un numero di armate che deve essere almeno pari al numero
     * di armate con le quali ha sferrato l'attacco (da 1 a 3,
     * quindi). Nel caso sia il difensore a conquistare un
     * territorio il numero di armate da spostare  a scelta
     * del giocatore.</p>
     * 
     * <p>NOTA: su ogni territorio deve comunque rimanere sempre il
     * minimo numero di armate stabilito dalle regole.</p>
     * 
     * @param esito numero di armate da spostare
     * @param source <code>Territorio</code> dal quale spostare le armate
     * @param dest <code>Territorio</code> destinazione
     */
    private void spostamentoPostAttacco(int esito, Territorio source, Territorio dest) {
        source.muovi(esito,dest);
    }
    /**
     * Aggiorna il tabellone di gioco dopo l'attacco.
     * 
     * @param p riferimento all'oggetto <code>Tabellone</code>
     * @param t <code>Territorio</code> da aggiornare
     * @param g nuovo occupante del territorio
     */
    private void aggiornaPlanisfero(Tabellone p, Territorio t, Giocatore g) {
        p.aggiornaTerritorio(t,g);
    }
}
