/*
	This file is part of jrisk.
	
	jrisk is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.
	
	jrisk is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
	
	You should have received a copy of the GNU General Public License
	along with jrisk; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
 * Created on 19-mar-2005
 * Copyright  2005 Alessio Treglia
 */


import java.io.*;

class Lettore extends FilterReader {

    /* Il prossimo carattere da scandire, oppure 0. */
    private char next;

    public static Lettore in;

    /* Inizializzazione di Lettore.in */
    static {
        in = new Lettore();
    }

    public Lettore() {
        this(new InputStreamReader(System.in));
    }

    public Lettore(String s) {
        this(new StringReader(termina(s)));
    }

    /* Termina s con un terminatore di linea,
     * se s non  gi terminata in questo modo. */
    private static String termina(String s)  {
        if (s!=null)  {
            int lunghezza = s.length();
            /* se s non termina con un terminatore di linea */
            if (lunghezza>0 && s.charAt(lunghezza-1)!='\n')
                /* estende s con un terminatore di linea */
                s += '\n';
        }
        return s;
    }

    public Lettore(Reader rdr) {
        super(rdr);
        this.next = (char) 0;
    }

    public String leggiToken() {
        StringBuffer token = new StringBuffer();
        char ch;
        skipWhite();
        ch = get();
        while (!isWhite(ch)) {
            token.append(ch);
            ch = get();
        }
        return token.toString();
    }

    public String leggiString() {
        return leggiToken();
    }

    public char leggiChar() {
        char ch;
        while (eoln())
            skipEOLN();
        return get();
    }

    public long leggiLong() {
        /* assume che la prossima stringa sia l'intero da leggere */
        String token = leggiToken();
        long numero;
        try {
            numero = Long.parseLong(token);
        } catch (NumberFormatException e) {
            numero = 0;
        }
        return numero;
    }

    public int leggiInt() {
        return (int) leggiLong();
    }

    public short leggiShort() {
        return (short) leggiLong();
    }

    public float leggiFloat() {
        /* assume che la prossima stringa sia il float da leggere */
        String token = leggiToken();
        float numero;
        try {
            numero = (new Float(token)).floatValue();
        } catch (NumberFormatException e) {
            numero = 0;
        }
        return numero;
    }

    public double leggiDouble() {
        /* assume che la prossima stringa sia il double da leggere */
        String token = leggiToken();
        double numero;
        try {
            numero = (new Double(token)).doubleValue();
        } catch (NumberFormatException e) {
            numero = 0;
        }
        return numero;
    }

    public boolean leggiBoolean() {
        /* assume che la prossima stringa sia il boolean da leggere */
        String token = leggiToken();
        boolean b;
        try {
            b = (new Boolean(token)).booleanValue();
        } catch (NumberFormatException e) {
            b = false;
        }
        return b;
    }

    public String leggiLinea() {
        StringBuffer line = new StringBuffer();
        char ch;
        while (!eoln()) {
            ch = get();
            line.append(ch);
        }
        skipEOLN();
        return line.toString();
    }

    public void readln() {
        char ch;
        while (!eoln()) {
            ch = get();
        }
        skipEOLN();
    }

    public boolean eof() {
        char ch = peek();
        return (ch==(char)-1);
    }

    public boolean eoln() {
        char ch = peek();
        return (ch=='\n') || (ch=='\r') || eof();
    }

    private char peek() {
        if (next==0)
            next = scan();
        return next;
    }

    private char get() {
        char ch = peek();
        next = 0;  // lo considera comunque consumato
        return ch;
    }

    private char scan() {
        char ch;
        try {
            /* attenzione: c' una variabile in di classe in Lettore 
             * e una variabile in d'istanza in FilterReader */ 
            ch = (char) super.in.read();
        } catch (IOException e) {
            /* assumiamo di aver raggiunto l'EOF */
            ch = (char) -1;
        }
        return ch;
    }

    private void skipEOLN() {
        char ch = peek();
        if (ch=='\r') {
            get();
            ch = peek();
        }
        if (ch=='\n') {
            get();
        }
    }

    private void skipWhite() {
        char ch = peek();
        while (isWhite(ch)) {
            get();
            ch = peek();
        }
    }

    private static boolean isWhite(char ch) {
        return Character.isWhitespace(ch);
    }

    private static boolean isDigit(char ch) {
        return Character.isDigit(ch);
    }

}