/*
 * @(#)FirstPassAnalysis.java      1.5        13 September 1999
 *
 * This work is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * Copyright (c) 1999 Ericsson Telecom. All rights reserved.
 * Copyright (c) 2002 Per Cederberg. All rights reserved.
 */

package net.percederberg.mib;

import net.percederberg.mib.asn1.analysis.DepthFirstAdapter;
import net.percederberg.mib.asn1.node.Node;
import net.percederberg.mib.symbol.Symbol;
import net.percederberg.mib.symbol.TypeSymbol;
import net.percederberg.mib.symbol.ValueSymbol;
import net.percederberg.mib.type.IntegerType;
import net.percederberg.mib.type.ObjectIdentifierType;
import net.percederberg.mib.type.StringType;
import net.percederberg.mib.type.Constraint;
import net.percederberg.mib.type.SizeConstraint;
import net.percederberg.mib.type.ValueConstraint;
import net.percederberg.mib.type.ValueRangeConstraint;

/**
 * The first pass analysis of the parse tree. Creates symbols for
 * all identifiers in the MIB and adds them to the symbol table.
 * Also checks for unsupported syntactic constructions, unsupported
 * types and unknown import symbols.<p>
 *
 * During the first pass analysis the symbols will not be analyzed
 * for their type, though, as such an analysis requires full
 * knowledge of all type and value names used. That type of
 * analysis is performed in the second pass.<p>
 *
 * This class is used internally during MIB creation.
 *
 * @version  1.5
 * @author   Per Cederberg, per@percederberg.net
 */
class FirstPassAnalysis extends DepthFirstAdapter {

   /**
    * The MIB to work on.
    */
   private MIB     mib;

   /**
    * Creates a new first pass analyzer.
    *
    * @param   mib       the MIB to add symbols to
    */
   public FirstPassAnalysis(MIB mib) {
      this.mib = mib;
   }

   /**
    * Adds an imported type to the MIB. An warning will be
    * generated if the symbol is not one of the recognized type
    * symbols.
    *
    * @param  typeName    the name of an imported type
    */
   private void addImportedType(String typeName) {
      Symbol     sym = null;
      Constraint c;

      if (typeName.equals("DisplayString")) {
         sym = new TypeSymbol(typeName);
         sym.setType(new StringType());
      } else if (typeName.equals("IpAddress")) {
         sym = new TypeSymbol(typeName);
         c = new ValueConstraint(new Integer(4));
         c = new SizeConstraint((ValueConstraint)c);
         sym.setType(new StringType((SizeConstraint)c));
      } else if (typeName.equals("Counter")) {
         sym = new TypeSymbol(typeName);
         c = new ValueRangeConstraint(new Long(0), new Long(4294967295l));
         sym.setType(new IntegerType((ValueRangeConstraint)c));
      }
      if (sym != null) {
         mib.addSymbol(sym);
      } else {
         mib.addWarning("imported type unknown: " + typeName);
      }
   }

   /**
    * Adds an imported value to the MIB. An warning will be
    * generated if the symbol is not one of the recognized value
    * symbols.
    *
    * @param  valueName    the name of an imported value
    */
   private void addImportedValue(String valueName) {
      Symbol    sym = null;

      if (valueName.equals("enterprises")) {
         ObjectIdentifierType type = ObjectIdentifierType.getInstance();
         sym = new ValueSymbol("iso", type, 1);
         sym = new ValueSymbol("org", type, sym, 3);
         sym = new ValueSymbol("dod", type, sym, 6);
         sym = new ValueSymbol("internet", type, sym, 1);
         sym = new ValueSymbol("private", type, sym, 4);
         sym = new ValueSymbol("enterprises", type, sym, 1);
      }
      if (sym != null) {
         mib.addSymbol(sym);
      } else {
         mib.addWarning("imported value unknown: " + valueName);
      }
   }

   /**
    * Sets the output value to the name of the module reference.
    *
    * @param node    the parse tree node
    */
   protected void outModuleDefinition(Node node) {
      String name = (String)getOut(node.childOfType(Node.MODULEIDENTIFIER));
      mib.setName(name);
   }

   /**
    * Generates an warning if called.
    *
    * @param node    the parse tree node
    */
   protected void inTagDefault(Node node) {
      mib.addWarning("unsupported construct: ... TAGS",
                     node.firstLine(),
                     node.lastLine());
   }

   /**
    * Sets the output value to the name of the module reference.
    *
    * @param node    the parse tree node
    */
   protected void outModuleIdentifier(Node node) {
      setOut(node, getOut(node.childOfType(Node.MODULEREFERENCE)));
   }

   /**
    * Generates an warning if called.
    *
    * @param node    the parse tree node
    */
   protected void inExports(Node node) {
      mib.addWarning("unsupported construct: EXPORTS ...",
                     node.firstLine(),
                     node.lastLine());
   }

   /**
    * Analyzes the symbol if inside imports.
    *
    * @param node    the parse tree node
    */
   protected void outSymbol(Node node) {
      String id;

      if (!node.isAncestor(Node.IMPORTS) ||
         node.childrenOfType(Node.DEFINEDMACRONAME) > 0) {

         return;
      }

      if (node.childrenOfType(Node.IDENTIFIER) > 0) {
         id = (String)getOut(node.childOfType(Node.IDENTIFIER));
         addImportedValue(id);
      } else {
         id = (String)getOut(node.childOfType(Node.TYPEREFERENCE));
         addImportedType(id);
      }
   }

   /**
    * Generates an warning if called.
    *
    * @param node    the parse tree node
    */
   protected void inMacroDefinition(Node node) {
      mib.addWarning("unsupported construct: MACRO ...",
                     node.firstLine(),
                     node.lastLine());
   }

   /**
    * Creates and adds a type symbol.
    *
    * @param node    the parse tree node
    */
   protected void outTypeAssignment(Node node) {
      String id = (String)getOut(node.childOfType(Node.TYPEREFERENCE));
      mib.addSymbol(new TypeSymbol(id));
   }

   /**
    * Generates an warning if called.
    *
    * @param node    the parse tree node
    */
   protected void inBitStringType(Node node) {
      mib.addWarning("unsupported type: BIT STRING",
                     node.firstLine(),
                     node.lastLine());
   }

   /**
    * Generates an warning if called.
    *
    * @param node    the parse tree node
    */
   protected void inChoiceType(Node node) {
      mib.addWarning("unsupported type: CHOICE ...",
                     node.firstLine(),
                     node.lastLine());
   }

   /**
    * Generates an warning if called.
    *
    * @param node    the parse tree node
    */
   protected void inEnumeratedType(Node node) {
      mib.addWarning("unsupported type: ENUMERATED ...",
                     node.firstLine(),
                     node.lastLine());
   }

   /**
    * Generates an warning if called.
    *
    * @param node    the parse tree node
    */
   protected void inSelectionType(Node node) {
      mib.addWarning("unsupported type: name < ...",
                     node.firstLine(),
                     node.lastLine());
   }

   /**
    * Generates an warning if called.
    *
    * @param node    the parse tree node
    */
   protected void inTaggedType(Node node) {
      mib.addWarning("unsupported type: [X] ...",
                     node.firstLine(),
                     node.lastLine());
   }

   /**
    * Generates an warning if called.
    *
    * @param node    the parse tree node
    */
   protected void inAnyType(Node node) {
      mib.addWarning("unsupported type: ANY",
                     node.firstLine(),
                     node.lastLine());
   }

   /**
    * Generates an warning if called.
    *
    * @param node    the parse tree node
    */
   protected void inAlphabetConstraint(Node node) {
      mib.addWarning("unsupported constraint: FROM ...",
                     node.firstLine(),
                     node.lastLine());
   }

   /**
    * Creates and adds a value symbol.
    *
    * @param node    the parse tree node
    */
   protected void outValueAssignment(Node node) {
      String id = (String)getOut(node.childOfType(Node.IDENTIFIER));
      mib.addSymbol(new ValueSymbol(id));
   }

   /**
    * Sets the output value to the identifier string.
    *
    * @param node    the parse tree node
    */
   protected void outIdentifier(Node node) {
      setOut(node, getOut(node.childOfType(Node.TLCASEFIRST_IDENT)));
   }

   /**
    * Sets the output value to the identifier string.
    *
    * @param node    the parse tree node
    */
   protected void outModuleReference(Node node) {
      setOut(node, getOut(node.childOfType(Node.TUCASEFIRST_IDENT)));
   }

   /**
    * Sets the output value to the identifier string.
    *
    * @param node    the parse tree node
    */
   protected void outTypeReference(Node node) {
      setOut(node, getOut(node.childOfType(Node.TUCASEFIRST_IDENT)));
   }

   /**
    * Generates an warning if called.
    *
    * @param node    the parse tree node
    */
   protected void inTBOOLEAN(Node node) {
      mib.addWarning("unsupported type: BOOLEAN",
                     node.firstLine(),
                     node.lastLine());
   }

   /**
    * Generates an warning if called.
    *
    * @param node    the parse tree node
    */
   protected void inTREAL(Node node) {
      mib.addWarning("unsupported type: REAL",
                     node.firstLine(),
                     node.lastLine());
   }

   /**
    * Generates an warning if called.
    *
    * @param node    the parse tree node
    */
   protected void inTNULL(Node node) {
      mib.addWarning("unsupported type and value: NULL",
                     node.firstLine(),
                     node.lastLine());
   }

   /**
    * Sets the output value to the token image.
    *
    * @param node    the parse tree node
    */
   protected void outTUCASEFIRST_IDENT(Node node) {
      setOut(node, node.toString());
   }

   /**
    * Sets the output value to the token image.
    *
    * @param node    the parse tree node
    */
   protected void outTLCASEFIRST_IDENT(Node node) {
      setOut(node, node.toString());
   }
}

