/*
 * @(#)MIB.java               1.2             13 September 1999
 *
 * This work is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * Copyright (c) 1999 Ericsson Telecom. All rights reserved.
 * Copyright (c) 2002 Per Cederberg. All rights reserved.
 */

package net.percederberg.mib;

import java.io.File;
import java.io.FileInputStream;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

import net.percederberg.mib.asn1.node.Node;
import net.percederberg.mib.asn1.parser.AsnParser;
import net.percederberg.mib.symbol.Symbol;

/**
 * A class representing a MIB file. It also serves as a symbol table
 * for the symbols in the MIB file. Each MIB object represents a
 * distinct ASN.1 module, and several objects are thus required
 * to represent multiple modules. An ASN.1 symbol can be uniqely
 * identified by it's MIB (module) and symbol name.<p>
 *
 * The MIB class also handles the errors
 *
 * @version  1.2
 * @author   Per Cederberg, per@percederberg.net
 */
public class MIB extends Object {

   /**
    * The MIB filename.
    */
   private File         file = null;

   /**
    * The MIB name.
    */
   private String       name = null;

   /**
    * The hash table for all symbols.
    */
   private Hashtable    table = null;

   /**
    * A vector with all symbols in order.
    */
   private Vector       list = null;

   /**
    * A vector with all error messages.
    */
   private Vector       errors = null;

   /**
    * A vector with all warning messages.
    */
   private Vector       warnings = null;

   /**
    * A vector with all other messages.
    */
   private Vector       messages = null;

   /**
    * Creates a MIB object by reading and parsing the given file.
    *
    * @param   filename     the name of a readable MIB file
    *
    * @exception Exception if the file couldn't be parsed correctly
    */
   public MIB(String filename) throws Exception {
      this(new File(filename));
   }

   /**
    * Creates a MIB object by reading and parsing the given file.
    *
    * @param   file         a readable MIB file
    *
    * @exception Exception if the file couldn't be parsed correctly
    */
   public MIB(File file) throws Exception {
      // Initialize
      this.file = file;
      this.name = "";
      this.table = new Hashtable();
      this.list = new Vector();

      // Parse MIB file
      Node parseTree = AsnParser.parse(new FileInputStream(file));
      parseTree.apply(new FirstPassAnalysis(this));
      if (this.errors() == 0) {
         parseTree.apply(new SecondPassAnalysis(this));
      }
   }

   /**
    * Checks if this MIB is equal to some other object. The comparison
    * is based on name comparison.
    *
    * @param obj       the object to compare with
    *
    * @return true if the objects are equal, or
    *         false otherwise
    */
   public boolean equals(Object obj) {
      if (obj instanceof MIB) {
         return this.name.equals(obj.toString());
      } else {
         return this.name.equals(obj);
      }
   }

   /**
    * Returns the MIB name.
    *
    * @return the MIB name
    */
   public String toString() {
      return name;
   }

   /**
    * Returns the MIB filename.
    *
    * @return the MIB filename
    */
   public String getFilename() {
      return this.file.toString();
   }

   /**
    * Finds a symbol in the symbol table.
    *
    * @param   name     the symbol name
    *
    * @return  the symbol found, or null if not found
    */
   public Symbol findSymbol(String name) {
      return (Symbol)table.get(name);
   }

   /**
    * Returns an enumeration of all the symbols in the table. The enumeration
    * will return all the symbols in the order in which they were found
    * in the MIB file.
    *
    * @return an enumeration of all symbols
    */
   public Enumeration allSymbols() {
      return list.elements();
   }

   /**
    * Returns a concatenated string with all the messages, ordered by
    * category.
    *
    * @return a string with all parse messages
    */
   public String allMessages() {
      return vectorToString(errors) + vectorToString(warnings) +
             vectorToString(messages);
   }

   /**
    * Returns the number of found errors in this MIB.
    *
    * @return the number of errors found while parsing
    */
   public int errors() {
      if (errors == null) {
         return 0;
      } else {
         return errors.size();
      }
   }

   /**
    * Returns the number of found warnings in this MIB.
    *
    * @return the number of warnings found while parsing
    */
   public int warnings() {
      if (warnings == null) {
         return 0;
      } else {
         return warnings.size();
      }
   }

   /**
    * Clears the MIB by recursively clearing all the symbols in the tables.
    * Thereafter resets all the symbol tables to null. This method should
    * be called before disposing the MIB for consistency with other objects
    * in the system.
    */
   public void clear() {
      name = null;
      table = null;
      for (int i = 0; i < list.size(); i++) {
         Symbol sym = (Symbol)list.elementAt(i);
      }
      list = null;
      errors = null;
      warnings = null;
      messages = null;
   }

   /**
    * Sets the MIB name.
    *
    * @param name    the new MIB name
    */
   void setName(String name) {
      this.name = name;
   }

   /**
    * Adds a symbol to the symbol table.
    *
    * @param   sym      the symbol to add
    */
   void addSymbol(Symbol sym) {
      table.put(sym.getName(), sym);
      list.addElement(sym);
   }

   /**
    * Adds an error message without line information.
    *
    * @param   message      the error message to add
    */
   void addError(String message) {
      if (errors == null) {
         errors = new Vector();
      }
      errors.add(new Message(Message.ERROR, message));
   }

   /**
    * Adds an error message with line information.
    *
    * @param   message      the error message to add
    * @param   firstLine    the first line affected
    * @param   lastLine     the last line affected
    */
   void addError(String message, int firstLine, int lastLine) {
      if (errors == null) {
         errors = new Vector();
      }
      errors.add(new Message(Message.ERROR, message, firstLine, lastLine));
   }

   /**
    * Adds a warning message without line information.
    *
    * @param   message      the warning message to add
    */
   void addWarning(String message) {
      if (warnings == null) {
         warnings = new Vector();
      }
      warnings.add(new Message(Message.WARNING, message));
   }

   /**
    * Adds a warning message with line information.
    *
    * @param   message      the warning message to add
    * @param   firstLine    the first line affected
    * @param   lastLine     the last line affected
    */
   void addWarning(String message, int firstLine, int lastLine) {
      if (warnings == null) {
         warnings = new Vector();
      }
      warnings.add(new Message(Message.WARNING, message, firstLine, lastLine));
   }

   /**
    * Adds a message without line information.
    *
    * @param   message      the message to add
    */
   void addMessage(String message) {
      if (messages == null) {
         messages = new Vector();
      }
      messages.add(new Message(Message.OTHER, message));
   }

   /**
    * Adds a message with line information.
    *
    * @param   message      the message to add
    * @param   firstLine    the first line affected
    * @param   lastLine     the last line affected
    */
   void addMessage(String message, int firstLine, int lastLine) {
      if (messages == null) {
         messages = new Vector();
      }
      messages.add(new Message(Message.OTHER, message, firstLine, lastLine));
   }

   /**
    * Returns a string with the vector elements concatenated with
    * newline characters.
    *
    * @param v     a vector
    *
    * @return a string with the contents of the vector
    */
   private String vectorToString(Vector v) {
      Enumeration   e;
      String        result = "";

      if (v == null) {
         // Do nothing
         return "";
      }
      e = v.elements();
      while (e.hasMoreElements()) {
         result = result + e.nextElement() + "\n";
      }
      return result;
   }
}


/**
 * A class that handles a single error or warning message.
 *
 * @version  1.0
 * @author   Per Cederberg, per@percederberg.net
 */
class Message extends Object {

   /**
    * The error message type.
    */
   public static final int ERROR    = 0;

   /**
    * The warning message type.
    */
   public static final int WARNING  = 1;

   /**
    * The other message type.
    */
   public static final int OTHER    = 2;

   /**
    * The message type string.
    */
   private static final String[] MESSAGE_DESCRIPTIONS =
      { "ERROR", "Warning", "Message" };

   /**
    * The message type.
    */
   protected int     type;

   /**
    * The message text.
    */
   protected String  message;

   /**
    * The first line that the message applies to.
    */
   protected int     firstLine;

   /**
    * The last line that the message applies to.
    */
   protected int     lastLine;

   /**
    * Creates a new message with the OTHER type and no line numbers.
    *
    * @param   message     the message text
    */
   public Message(String message) {
      this(OTHER, message);
   }

   /**
    * Creates a new message with no line numbers.
    *
    * @param   type        the message type
    * @param   message     the message text
    */
   public Message(int type, String message) {
      this(type, message, -1, -1);
   }

   /**
    * Creates a new message with the given parameters.
    *
    * @param   type        the message type
    * @param   message     the message text
    * @param   firstLine   the first line to refer to
    * @param   lastLine    the last line to refer to
    */
   public Message(int type, String message, int firstLine, int lastLine) {
      this.type = type;
      this.message = message;
      this.firstLine = firstLine;
      if (lastLine <= 0) {
         this.lastLine = firstLine;
      } else {
         this.lastLine = lastLine;
      }
   }

   /**
    * Returns a string description of this message.
    *
    * @return a string description of the message
    */
   public String toString() {
      return MESSAGE_DESCRIPTIONS[type] + linesToString() + ": " +
             message;
   }

   /**
    * Returns a string description of the line numbers.
    *
    * @return a line number description string
    */
   private String linesToString() {
      if (firstLine > 0 && firstLine != lastLine) {
         return " (lines " + firstLine + "-" + lastLine + ")";
      } else if (firstLine > 0) {
         return " (line " + firstLine + ")";
      } else {
         return "";
      }
   }
}

