/*
 * @(#)MibValidator.java
 *
 * This work is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * As a special exception, the copyright holders of this library 
 * give you permission to link this library with independent modules
 * to produce an executable, regardless of the license terms of 
 * these independent modules, and to copy and distribute the 
 * resulting executable under terms of your choice, provided that 
 * you also meet, for each linked independent module, the terms and 
 * conditions of the license of that module. An independent module 
 * is a module which is not derived from or based on this library. 
 * If you modify this library, you may extend this exception to your 
 * version of the library, but you are not obligated to do so. If 
 * you do not wish to do so, delete this exception statement from 
 * your version. 
 *
 * Copyright (c) 2003 Per Cederberg. All rights reserved.
 */

package net.percederberg.mib.example;

import java.io.File;
import java.util.Vector;

import net.percederberg.mib.Mib;

/**
 * A program that parses and validates a MIB file. This program has
 * been created to simplify testing MIB files with the parser. If the
 * MIB file uses constructs or syntax that are not supported, an error
 * message will be printed to the standard output. The program will
 * also return the number of validation failures as its exit code.
 *
 * @author   Per Cederberg, per@percederberg.net
 * @version  1.1
 */
public class MibValidator extends Object {

    /**
     * The application main entry point.
     *
     * @param args    the command-line parameters
     */
    public static void main(String[] args) {
        Vector  files = new Vector();
        Mib     mib;
        int     errors = 0;
        int     i;

        // Check command-line arguments
        if (args.length < 1) {
            printHelp("No dir(s) or file(s) specified");
            System.exit(1);
        }
   
        // Find all files
        for (i = 0; i < args.length; i++) {
            try {
                addFile(new File(args[i]), files);
            } catch (Exception e) {
                printHelp(e.getMessage());
                System.exit(1);
            }
        }

        // Parser MIB files
        for (i = 0; i < files.size(); i++) {
            try {
                System.out.print("Reading " + files.get(i) + "... ");
                mib = new Mib((File) files.get(i));
                if (mib.getErrorCount() + mib.getWarningCount() == 0) {
                    System.out.println("[OK]");
                } else {
                    errors++;
                    System.out.println("[FAILED]");
                }
                System.out.print(mib.getAllMessages());
            } catch (Exception e) {
                System.out.println("[FAILED]");
                errors++;
                e.printStackTrace(System.out);
            }
        }

        // Returns error count
        System.exit(errors);
    }

    /**
     * Prints command-line help information.
     *
     * @param error    an optional error message, or null
     */
    private static void printHelp(String error) {
        System.err.println("Syntax: java " + MibValidator.class +
                           "<dir(s) or file(s)>");
        System.err.println();
        if (error != null) {
            System.err.print("Error: ");
            System.err.println(error);
            System.err.println();
        }
    }

    /**
     * Adds a directory or file to a vector. If the file specified is
     * a directory, only the files in the directory will be added.
     * 
     * @param file      the file to add
     * @param res       the vector to add the file(s) to
     */
    private static void addFile(File file, Vector res) {
        File[]  files;

        if (!file.exists()) {
            printHelp("File not found: " + file);
            System.exit(1);
        } else if (file.isDirectory()) {
            files = file.listFiles();
            for (int i = 0; i < files.length; i++) {
                if (files[i].isDirectory() 
                 || files[i].getName().endsWith(".mib")) {

                    addFile(files[i], res);
                }
            }
        } else {
            res.add(file);
        } 
    }
}
