/**
 * Ce programme montre comment il est possible d'utiliser et d'afficher des
 * images.
 *
 * De nombreux formats d'images sont supportées (png, jpg, bmp).
 *
 * Pour cela, il suffit d'utiliser les fonctions suivantes :
 * 
 *------------------------------------------------------------------------------
 * MLV_load_image : Charge en mémoire une image contenue dans un fichier au 
 *                      format PNG, BMP, JPG, etc ...
 *
 * MLV_Image* MLV_load_image(
 *    const char*    file_image    Chemin d'accès du fichier contenant l'image.
 * );
 *
 *------------------------------------------------------------------------------
 * MLV_free_image : Libère la mémoire utilisée par l'image
 *
 * void MLV_free_image(
 *    MLV_Image *    image    image à suprimer de la mémoire
 * );
 *
 *------------------------------------------------------------------------------
 * MLV_resize_image_with_proportions : redimensionne l'image de telle sorte que 
 *                                     les proportions d'origines soient 
 *     respectées et que la hauteur et la largeur de la nouvelle image soient à 
 *     la fois maximales et inférieures ou égales à la hauteur et la largeur 
 *     données en paramètres. 
 *     L'utilisateur peut entrer un entier négatif pour les champs width et 
 *     height. Ces champs ne sont alors pas pris en compte dans le 
 *     redimensionnement de l'image.
 *                                    
 * void MLV_resize_image_with_proportions(
 *    MLV_Image*    image,    image à redimentionner
 *    int           width,    Nouvelle largeur de l'image
 *    int           height    Nouvelle hauteur de l'image
 * );
 *
 *------------------------------------------------------------------------------
 *
 * Il existe de nombreuses fonctions permettant de manipuler les images.
 * Vous trouverez les prototypes de ces fonctions dans le fichier
 * MLV_image.h.
 *
 * Vous pouvez aussi accéder à ces prototypes et à la documentation de ces
 * fonctions en tapant sur le terminal la commande :
 *     man MLV_image.h
 *
 */

#include <MLV/MLV_all.h>


//
// Attention ! 
// Pour pouvoir compiler ce programme sous windows et sous macintosh,
// il faut, pour la déclaration du main, respecter strictement la syntaxe
// suivante :
//
int main( int argc, char *argv[] ){
	int width = 640, height = 480;
	MLV_Image *image_mlv, *image_bdx;
	int width_mlv, height_mlv;
	int width_bdx, height_bdx;

	//
	// On créé et affiche la fenêtre
	//
	MLV_create_window( "advanced - 6 - image", "image", width, height );

	//
	// On charge en mémoire deux fichiers images.
	//
	image_mlv = MLV_load_image( "./data/images/upe_mlv.png" );
	image_bdx = MLV_load_image( "./data/images/ub1.jpg" );

	//
	// On redimensionne les images de sorte à ce qu'elles gardent leurs
	// proportions d'origine, mais que leur largeur soient identiques
	//
	MLV_resize_image_with_proportions( image_mlv, width/3, -1);
	MLV_resize_image_with_proportions( image_bdx, width/3, -1);

	//
	// On récupére la taille de l'image de l'université de Marne-la-vallée
	// afin de pouvoir positionner le logo de l'université de Bordeaux 
	// juste à côté.
	//
    MLV_get_image_size( image_mlv, &width_mlv, &height_mlv );
    MLV_get_image_size( image_bdx, &width_bdx, &height_bdx );

	//
	// On affiche les logos de l'université de Marne-la-vallée et de 
	// de l'université de Bordeaux 1
	//
	MLV_draw_image( image_mlv, (width-width_mlv-height_bdx)/3, (height-height_mlv)/2 );
	MLV_draw_image( image_bdx, width_mlv+2*(width-width_mlv-height_bdx)/3, (height-height_bdx)/2 );

	//
	// Met à jour l'affichage
	//
	MLV_actualise_window();

	//
	// Attend 2 secondes
	//
	MLV_wait_seconds( 2 );

	//
	// Libère l'image qui a été crée
	//
	MLV_free_image( image_mlv );
	MLV_free_image( image_bdx );

	//
	// Ferme la fenêtre
	//
	MLV_free_window();
	return 0;
}
