# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008, 2009 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import gtk
import gobject
import pango
import gtksourceview2
import os.path

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief  Source view
# @author Eduardo Aguiar
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class SourceView (gtksourceview2.View):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Build widget
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, *args):

    # internal data
    self.__can_undo = False
    self.__can_redo = False
    self.__can_paste = False
    self.__language_manager = gtksourceview2.language_manager_get_default ()
    self.__scheme_manager = gtksourceview2.style_scheme_manager_get_default ()

    # widget data
    self.language = None
    self.scheme = None

    # instanciate a gtksourceview2.Buffer if none was passed
    if not args:
      args = [gtksourceview2.Buffer ()]

    # build widget
    gtksourceview2.View.__init__ (self, *args)
    self.set_auto_indent (True)
    self.set_indent_width (4)
    self.set_insert_spaces_instead_of_tabs (True)
    self.set_show_line_numbers (True)
    self.set_highlight_current_line (True)

    font = pango.FontDescription ("Monospace 10")
    self.modify_font (font)

    # connect handled signals
    self.connect_after ('undo', self.on_undo)
    self.connect_after ('redo', self.on_redo)
    buffer = self.get_buffer ()
    buffer.connect ("modified-changed", self.on_modified_changed)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Function: set source code
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def set_source_code (self, code):
    buffer = self.get_buffer ()
    buffer.begin_not_undoable_action () 
    buffer.set_text (code)
    buffer.end_not_undoable_action () 
    buffer.set_modified (False)

    self.grab_focus ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Function: get source code
  # @return unicode string
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_source_code (self):
    buffer = self.get_buffer ()
    start, end = buffer.get_bounds ()
    return unicode (buffer.get_text (start, end), 'utf-8')

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Function: set language
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def set_language (self, id):
    self.language = self.__language_manager.get_language (id)

    buffer = self.get_buffer ()
    buffer.set_language (self.language)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Function: set scheme
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def set_scheme (self, id):
    self.scheme = self.__scheme_manager.get_scheme (id)

    buffer = self.get_buffer ()
    buffer.set_style_scheme (self.scheme)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Function: set modified
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def set_modified (self, modified):
    buffer = self.get_buffer ()
    buffer.set_modified (modified)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Function: get modified
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_modified (self):
    buffer = self.get_buffer ()
    return buffer.get_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Function: scroll to line
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def scroll_to_line (self, linenumber):
    buffer = self.get_buffer ()
    iter = buffer.get_iter_at_line (linenumber - 1)
    if iter:
      buffer.place_cursor (iter)
      mark = buffer.get_insert ()
      self.scroll_to_mark (mark, 0.3)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Function: cut selected text to clipboard
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def cut_clipboard (self):
    clipboard = gtk.clipboard_get ()
    editable = self.get_editable ()
    buffer = self.get_buffer ()
    buffer.cut_clipboard (clipboard, editable)

    if not self.__can_paste:
        self.__can_paste = True
        rc = self.emit ('on_can_paste')
        return rc

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Function: copy selected text to clipboard
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def copy_clipboard (self):
    clipboard = gtk.clipboard_get ()
    buffer = self.get_buffer ()
    buffer.copy_clipboard (clipboard)

    if not self.__can_paste:
        self.__can_paste = True
        rc = self.emit ('on_can_paste')
        return rc

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Function: paste text from clipboard
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def paste_clipboard (self):
    clipboard = gtk.clipboard_get ()
    editable = self.get_editable ()
    buffer = self.get_buffer ()
    buffer.paste_clipboard (clipboard, None, editable)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Function: undo
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def undo (self):
    buffer = self.get_buffer ()
    buffer.undo ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Function: redo
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def redo (self):
    buffer = self.get_buffer ()
    buffer.redo ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Function: select all
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def select_all (self):
    buffer = self.get_buffer ()
    start, end = buffer.get_bounds ()
    buffer.select_range (start, end)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Function: delete selection
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def delete_selection (self):
    editable = self.get_editable ()
    buffer = self.get_buffer ()
    buffer.delete_selection (True, editable)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Event: on_undo
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_undo (self, widget, *args):
    buffer = widget.get_buffer ()
    if not buffer.can_undo ():   # last undo
      self.emit ('on_undo_modified', False)
      self.__can_undo = False
    if not self.__can_redo:
      self.emit ('on_redo_modified', True)
      self.__can_redo = True

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Event: on_redo
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_redo (self, widget, *args):
    buffer = widget.get_buffer ()
    if not buffer.can_redo ():   # last redo
      self.emit ('on_redo_modified', False)
      self.__can_redo = False
    if not self.__can_undo:
      self.emit ('on_undo_modified', True)
      self.__can_undo = True

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Event: on_modified_changed
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_modified_changed (self, buffer, *args):
    rc = self.emit ('on_modified_changed', buffer.get_modified ())
    return rc

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief SourceView signals    
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
gobject.type_register (SourceView)

gobject.signal_new ("on_undo_modified", SourceView,
        gobject.SIGNAL_RUN_LAST, gobject.TYPE_BOOLEAN, (bool, ))

gobject.signal_new ("on_redo_modified", SourceView,
        gobject.SIGNAL_RUN_LAST, gobject.TYPE_BOOLEAN, (bool, ))

gobject.signal_new ("on_can_paste", SourceView,
        gobject.SIGNAL_RUN_LAST, gobject.TYPE_BOOLEAN, ())

gobject.signal_new ("on_modified_changed", SourceView,
        gobject.SIGNAL_RUN_LAST, gobject.TYPE_BOOLEAN, (bool, ))

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Extension properties dialog
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class PropertiesDialog (gtk.Dialog):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Build widget
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def __init__ (self, extension):
    gtk.Dialog.__init__ (self,
       'Extension "%s" properties' % extension.id, None, gtk.DIALOG_MODAL,
       (gtk.STOCK_OK, gtk.RESPONSE_OK, gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL))
    self.set_position (gtk.WIN_POS_CENTER)
    self.set_default_size (580, 480)
    self.set_type_hint (gtk.gdk.WINDOW_TYPE_HINT_DIALOG)
    self.set_border_width (10)

    table = gtk.Table (6, 3)
    table.set_row_spacings (10)
    table.set_col_spacings (5)
    table.show ()
    self.vbox.pack_start (table)

    label = gtk.Label ()
    label.set_markup ('<b>ID</b>')
    label.set_alignment (1.0, -1)
    label.show ()
    table.attach (label, 0, 1, 0, 1, gtk.FILL, 0)

    self.extension_id_entry = gtk.Entry ()
    self.extension_id_entry.set_text (extension.id)
    self.extension_id_entry.show ()
    table.attach (self.extension_id_entry, 1, 3, 0, 1, yoptions=0)

    label = gtk.Label ()
    label.set_markup ('<b>Name</b>')
    label.set_alignment (1.0, -1)
    label.show ()
    table.attach (label, 0, 1, 1, 2, gtk.FILL, 0)

    self.extension_name_entry = gtk.Entry ()
    self.extension_name_entry.set_text (extension.name)
    self.extension_name_entry.show ()
    table.attach (self.extension_name_entry, 1, 3, 1, 2, yoptions=0)

    label = gtk.Label ()
    label.set_markup ('<b>Version</b>')
    label.set_alignment (1.0, -1)
    label.show ()
    table.attach (label, 0, 1, 2, 3, gtk.FILL, 0)

    self.extension_version_entry = gtk.Entry ()
    self.extension_version_entry.set_text (extension.version)
    self.extension_version_entry.show ()
    table.attach (self.extension_version_entry, 1, 3, 2, 3, yoptions=0)

    label = gtk.Label ()
    label.set_markup ('<b>Author</b>')
    label.set_alignment (1.0, -1)
    label.show ()
    table.attach (label, 0, 1, 3, 4, gtk.FILL, 0)

    self.extension_author_entry = gtk.Entry ()
    self.extension_author_entry.set_text (extension.author)
    self.extension_author_entry.show ()
    table.attach (self.extension_author_entry, 1, 3, 3, 4, yoptions=0)

    label = gtk.Label ()
    label.set_markup ('<b>Description</b>')
    label.set_alignment (1.0, -1)
    label.show ()
    table.attach (label, 0, 1, 4, 5, gtk.FILL, 0)

    self.extension_description_entry = gtk.Entry ()
    self.extension_description_entry.set_text (extension.description)
    self.extension_description_entry.show ()
    table.attach (self.extension_description_entry, 1, 3, 4, 5, yoptions=0)

    label = gtk.Label ()
    label.set_markup ('<b>Icon</b>')
    label.set_alignment (1.0, -1)
    label.show ()
    table.attach (label, 0, 1, 5, 6, gtk.FILL, 0)

    placeholder = gtk.Label (' ')
    placeholder.show ()
    table.attach (placeholder, 2, 3, 5, 6, yoptions=0)

    button = gtk.Button ()
    button.connect ('clicked', self.on_icon_edit)
    button.show ()

    self.icon_image = gtk.Image ()
    self.icon_data = ''

    if extension.icon_data:
      self.set_icon_data (extension.icon_data)
    else:
      self.icon_image.set_from_stock (gtk.STOCK_MISSING_IMAGE, gtk.ICON_SIZE_BUTTON)

    self.icon_image.show ()
    button.add (self.icon_image)

    table.attach (button, 1, 2, 5, 6, 0, 0)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief set icon data
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def set_icon_data (self, data):
    loader = gtk.gdk.PixbufLoader ()
    loader.set_size (64, 64)
    loader.write (data.decode ('base64'))
    loader.close ()

    self.icon_data = data
    self.icon_image.set_from_pixbuf (loader.get_pixbuf ())

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle icon edition
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_icon_edit (self, widget, *args):

    # choose file
    fs = gtk.FileChooserDialog ('Choose icon file')
    fs.add_button (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL)
    fs.add_button (gtk.STOCK_OK, gtk.RESPONSE_OK)

    filter = gtk.FileFilter ()
    filter.add_pattern ('*.png')
    filter.add_pattern ('*.jpg')
    filter.add_pattern ('*.gif')
    filter.add_pattern ('*.svg')
    fs.set_filter (filter)

    rc = fs.run ()
    filename = fs.get_filename ()
    fs.destroy ()

    if rc != gtk.RESPONSE_OK:
      return

    # read file
    fp = open (filename)
    data = fp.read ()
    fp.close ()

    # set new icon
    icon_data = data.encode ('base64')
    self.set_icon_data (icon_data)

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Widget
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Widget (gtk.VBox):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Initialize widget
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def __init__ (self, mediator):
    gtk.VBox.__init__ (self)
    self.mediator = mediator.copy ()
    self.set_border_width (5)
    self.set_spacing (5)
    self.show ()

    # menubar
    menubar = gtk.MenuBar ()
    menubar.show ()
    self.pack_start (menubar, False, False)

    item = gtk.MenuItem ('_File')
    item.show ()
    menubar.append (item)

    menu = gtk.Menu ()
    menu.show ()
    item.set_submenu (menu)

    item = gtk.ImageMenuItem (gtk.STOCK_NEW)
    item.connect ("activate", self.on_extension_new)
    item.show ()
    menu.append (item)

    item = gtk.ImageMenuItem (gtk.STOCK_OPEN)
    item.connect ("activate", self.on_extension_open)
    item.show ()
    menu.append (item)

    self.save_menuitem = gtk.ImageMenuItem (gtk.STOCK_SAVE)
    self.save_menuitem.set_sensitive (False)
    self.save_menuitem.connect ("activate", self.on_extension_save)
    self.save_menuitem.show ()
    menu.append (self.save_menuitem)

    self.save_as_menuitem = gtk.ImageMenuItem (gtk.STOCK_SAVE_AS)
    self.save_as_menuitem.set_sensitive (False)
    self.save_as_menuitem.connect ("activate", self.on_extension_save_as)
    self.save_as_menuitem.show ()
    menu.append (self.save_as_menuitem)

    self.close_menuitem = gtk.ImageMenuItem (gtk.STOCK_CLOSE)
    self.close_menuitem.set_sensitive (False)
    self.close_menuitem.connect ("activate", self.on_extension_close)
    self.close_menuitem.show ()
    menu.append (self.close_menuitem)

    item = gtk.SeparatorMenuItem ()
    item.show ()
    menu.append (item)

    item = gtk.ImageMenuItem (gtk.STOCK_QUIT)
    item.connect ("activate", self.on_file_close)
    item.show ()
    menu.append (item)

    item = gtk.MenuItem ('_Edit')
    item.show ()
    menubar.append (item)

    menu = gtk.Menu ()
    menu.show ()
    item.set_submenu (menu)

    self.properties_menuitem = gtk.ImageMenuItem (gtk.STOCK_PROPERTIES)
    self.properties_menuitem.set_sensitive (False)
    self.properties_menuitem.connect ("activate", self.on_extension_properties)
    self.properties_menuitem.show ()
    menu.append (self.properties_menuitem)

    # toolbar
    self.tooltips = gtk.Tooltips ()

    toolbar = gtk.Toolbar ()
    toolbar.set_style (gtk.TOOLBAR_ICONS)
    toolbar.set_tooltips (True)
    toolbar.show ()
    self.pack_start (toolbar, False, False)

    item = gtk.ToolButton (gtk.STOCK_NEW)
    item.connect ("clicked", self.on_extension_new)
    item.show ()
    item.set_tooltip (self.tooltips, "New extension")
    toolbar.insert (item, -1)

    item = gtk.ToolButton (gtk.STOCK_OPEN)
    item.connect ("clicked", self.on_extension_open)
    item.show ()
    item.set_tooltip (self.tooltips, "Open extension")
    toolbar.insert (item, -1)

    self.save_toolitem = gtk.ToolButton (gtk.STOCK_SAVE)
    self.save_toolitem.set_sensitive (False)
    self.save_toolitem.connect ("clicked", self.on_extension_save)
    self.save_toolitem.show ()
    self.save_toolitem.set_tooltip (self.tooltips, "Save extension")
    toolbar.insert (self.save_toolitem, -1)

    self.save_as_toolitem = gtk.ToolButton (gtk.STOCK_SAVE_AS)
    self.save_as_toolitem.set_sensitive (False)
    self.save_as_toolitem.connect ("clicked", self.on_extension_save_as)
    self.save_as_toolitem.show ()
    self.save_as_toolitem.set_tooltip (self.tooltips, "Save extension as")
    toolbar.insert (self.save_as_toolitem, -1)

    toolitem = gtk.SeparatorToolItem ()
    toolitem.show ()
    toolbar.insert (toolitem, -1)

    self.properties_toolitem = gtk.ToolButton (gtk.STOCK_PROPERTIES)
    self.properties_toolitem.set_sensitive (False)
    self.properties_toolitem.connect ("clicked", self.on_extension_properties)
    self.properties_toolitem.show ()
    self.properties_toolitem.set_tooltip (self.tooltips, "Set properties")
    toolbar.insert (self.properties_toolitem, -1)

    # notebook
    self.notebook = gtk.Notebook ()
    self.notebook.set_scrollable (True)
    self.notebook.connect_after ('switch-page', self.on_notebook_switch_page)
    self.notebook.show ()
    self.pack_start (self.notebook)

    # buttons
    hbox = gtk.HBox ()
    hbox.show ()
    self.pack_start (hbox, False)

    button = gtk.Button (stock=gtk.STOCK_CLOSE)
    button.connect ('clicked', self.on_file_close)
    button.show ()
    hbox.pack_end (button, False)

    # internal data
    self.next_untitled_id = 1

    # flags
    self.is_modified = False		# categories are modified
    self.is_selecting = False		# selecting new category
    self.category = None		# current selected category

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief return extension page
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def get_extension_page (self, extension):
    for i in range (self.notebook.get_n_pages ()):
      widget = self.notebook.get_nth_page (i)

      if widget.extension.path == extension.path:
        return i

    return -1

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief get current extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def get_extension (self):
    extension = None
    page_num = self.notebook.get_current_page ()

    if page_num != -1:
      widget = self.notebook.get_nth_page (page_num)
      extension = widget.extension

    return extension

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief add extension to notebook
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def add_extension (self, extension):

    # if extension is already being edited, select it and return
    page_num = self.get_extension_page (extension)
    if page_num != -1:
      self.notebook.set_current_page (page_num)
      return

    # create edition widget
    vbox = gtk.VBox ()
    vbox.set_spacing (5)
    vbox.show ()
    vbox.extension = extension

    sw = gtk.ScrolledWindow ()
    sw.set_policy (gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    sw.show ()
    vbox.pack_start (sw)

    vbox.sourceview = SourceView ()
    vbox.sourceview.set_language ('python')
    vbox.sourceview.set_source_code (extension.code)
    vbox.sourceview.connect ('on_modified_changed', self.on_sourceview_modified_changed)

    #self.sourceview.connect ('on_undo_modified', self.on_undo_modified)
    #self.sourceview.connect ('on_redo_modified', self.on_redo_modified)
    #self.sourceview.connect ('on_can_paste', self.on_can_paste)
    vbox.sourceview.show ()
    sw.add (vbox.sourceview)

    # create tab widget
    hbox = gtk.HBox ()
    hbox.set_spacing (3)
    hbox.show ()

    if extension.icon_data:
      pixbuf = self.mediator.call ('ui.render-icon', extension.icon_data, 24, 24)
      hbox.image = gtk.image_new_from_pixbuf (pixbuf)

    else:
      hbox.image = gtk.image_new_from_stock (gtk.STOCK_NEW, gtk.ICON_SIZE_MENU)

    hbox.image.show ()
    hbox.pack_start (hbox.image, False)

    hbox.label = gtk.Label (extension.id)
    hbox.label.show ()
    hbox.pack_start (hbox.label, False)

    image = gtk.image_new_from_stock (gtk.STOCK_CLOSE, gtk.ICON_SIZE_MENU)
    image.show ()

    button = gtk.Button ()
    button.set_image (image)
    button.set_relief (gtk.RELIEF_NONE)
    button.connect ('clicked', self.on_notebook_tab_close, extension)
    button.show ()
    hbox.pack_start (button, False)

    # create page into notebook
    page_num = self.notebook.append_page (vbox, hbox)
    self.notebook.set_current_page (page_num)
    self.notebook.set_tab_label_packing (vbox, False, False, gtk.PACK_START)
    self.notebook.set_tab_reorderable (vbox, True)

    # update window options
    self.update_options ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief remove extension from notebook
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def remove_extension (self, extension):

    # get extension widget
    page_num = self.get_extension_page (extension)
    if page_num == -1:
      return

    page_widget = self.notebook.get_nth_page (page_num)

    # check if extension is modified
    if extension and extension.is_modified:
      dialog = gtk.MessageDialog (None,
                  gtk.DIALOG_MODAL,
                  gtk.MESSAGE_QUESTION,
                  gtk.BUTTONS_YES_NO,
                  "Save extension '%s' before closing?" % extension.id)
      dialog.add_button (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL)
      rc = dialog.run ()
      dialog.destroy ()

      if rc == gtk.RESPONSE_CANCEL:
        return

      elif rc == gtk.RESPONSE_YES:
        self.save_extension (extension)

    # close tab
    self.notebook.remove_page (page_num)
    self.update_options ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief save extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def save_extension (self, extension, save_as_dialog=False):

    # show 'save as' dialog if necessary
    if extension.is_new or save_as_dialog:
      fs = gtk.FileChooserDialog ('Save extension "%s" as' % extension.id,
           action=gtk.FILE_CHOOSER_ACTION_SAVE)
      fs.add_button (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL)
      fs.add_button (gtk.STOCK_OK, gtk.RESPONSE_OK)
      fs.set_do_overwrite_confirmation (True)

      filter = gtk.FileFilter ()
      filter.set_name ('extension')
      filter.add_pattern ('*.xml')
      fs.set_filter (filter)

      rc = fs.run ()
      path = fs.get_filename ()
      fs.destroy ()

      if rc == gtk.RESPONSE_OK:
        root, ext = os.path.splitext (path)
        if ext != '.xml':
          path += '.xml'
      else:
        return

      extension.path = path

    # get widget
    page_num = self.get_extension_page (extension)
    widget = self.notebook.get_nth_page (page_num)
    widget.sourceview.set_modified (False)
    code = widget.sourceview.get_source_code () 

    # save extension
    extension.code = code.strip ()
    extension.is_modified = False
    extension.is_new = False
    self.mediator.call ('extension.save', extension, extension.path)

    # update options
    self.update_options ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief update current tab label
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def update_tab_label (self):
    extension = self.get_extension ()
    page_num = self.notebook.get_current_page ()

    # update text
    if page_num != -1:
      widget = self.notebook.get_nth_page (page_num)
      tab_widget = self.notebook.get_tab_label (widget)

      if extension.is_modified:
        text = '*%s' % extension.id
      else:
        text = extension.id

      tab_widget.label.set_text (text)

    # update icon
    if extension.icon_data:
      pixbuf = self.mediator.call ('ui.render-icon', extension.icon_data, 24, 24)
      tab_widget.image.set_from_pixbuf (pixbuf)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief update window options (toolbuttons and menuitems)
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def update_options (self):
    extension = self.get_extension ()

    if extension:
      is_opened = True
      is_modified = extension.is_modified
    else:
      is_opened = False
      is_modified = False

    # set sensitive
    self.save_menuitem.set_sensitive (is_modified)
    self.save_toolitem.set_sensitive (is_modified)
    self.save_as_menuitem.set_sensitive (is_opened)
    self.save_as_toolitem.set_sensitive (is_opened)
    self.close_menuitem.set_sensitive (is_opened)
    self.properties_menuitem.set_sensitive (is_opened)
    self.properties_toolitem.set_sensitive (is_opened)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle close button
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_file_close (self, widget, *args):
    self.mediator.call ('ui.working-area.close', self.working_area.id)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief create extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_extension_new (self, widget, *args):
    extension = self.mediator.call ('extension.new')
    extension.id = 'untitled-%02d' % self.next_untitled_id
    extension.name = 'untitled extension %02d' % self.next_untitled_id
    extension.path = 'untitled-%02d.xml' % self.next_untitled_id
    extension.is_modified = False
    extension.is_new = True

    self.next_untitled_id += 1
    self.add_extension (extension)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief open extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_extension_open (self, widget, *args):

    # build filechooser
    fs = gtk.FileChooserDialog ('Open extensions')
    fs.set_select_multiple (True)
    fs.add_button (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL)
    fs.add_button (gtk.STOCK_OK, gtk.RESPONSE_OK)

    filter = gtk.FileFilter ()
    filter.set_name ('extension')
    filter.add_pattern ('*.xml')
    fs.set_filter (filter)

    # run dialog
    rc = fs.run ()
    pathlist = fs.get_filenames ()
    fs.destroy ()

    if rc != gtk.RESPONSE_OK:
      return

    # open extensions
    for path in pathlist:
      extension = self.mediator.call ('extension.open', path)
      extension.path = path
      extension.is_modified = False
      extension.is_new = False
      self.add_extension (extension)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief save extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_extension_save (self, widget, *args):
    extension = self.get_extension ()
    self.save_extension (extension)
    self.update_tab_label ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief save as extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_extension_save_as (self, widget, *args):
    extension = self.get_extension ()
    self.save_extension (extension, save_as_dialog=True)
    self.update_tab_label ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief close extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_extension_close (self, widget, *args):
    extension = self.get_extension ()
    self.remove_extension (extension)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief close notebook tab
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_notebook_tab_close (self, button, extension, *args):
    self.remove_extension (extension)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief edit extension properties
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_extension_properties (self, widget, *args):
    extension = self.get_extension ()

    dialog = PropertiesDialog (extension)
    rc = dialog.run ()

    if rc == gtk.RESPONSE_OK:
      extension.id = dialog.extension_id_entry.get_text ()
      extension.name = dialog.extension_name_entry.get_text ()
      extension.version = dialog.extension_version_entry.get_text ()
      extension.author = dialog.extension_author_entry.get_text ()
      extension.description = dialog.extension_description_entry.get_text ()
      extension.icon_data = dialog.icon_data
      extension.is_modified = True

      self.update_tab_label ()
      self.update_options ()

    dialog.destroy ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief sourceview modified changed
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_sourceview_modified_changed (self, sourceview, is_modified, *args):
    if is_modified:
      extension = self.get_extension ()
      extension.is_modified = True
      self.update_tab_label ()
      self.update_options ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief notebook page switched
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_notebook_switch_page (self, widget, page, page_num, *args):
    self.update_options ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief show save/ignore/cancel dialog if there are modified items
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_widget_stopped (self):

    # if there is at least one extension modified, show save/ignore/cancel dialog
    model_is_modified = False
    model_save_modified = False

    for i in range (self.notebook.get_n_pages ()):
      widget = self.notebook.get_nth_page (i)

      if widget.extension.is_modified:
        model_is_modified = True

    # if model is modified, show save/ignore/cancel dialog
    if model_is_modified:
      dialog = gtk.MessageDialog (None,
                  gtk.DIALOG_MODAL,
                  gtk.MESSAGE_QUESTION,
                  gtk.BUTTONS_YES_NO,
                  "Save changes before closing?")
      dialog.add_button (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL)
      rc = dialog.run ()
      dialog.destroy ()

      if rc == gtk.RESPONSE_CANCEL:
        return True

      elif rc == gtk.RESPONSE_YES:
        model_save_modified = True

    # save modified extensions, if necessary
    if model_save_modified:
      for i in range (self.notebook.get_n_pages ()):
        widget = self.notebook.get_nth_page (i)

        if widget.extension.is_modified:
          self.save_extension (widget.extension)

    # clean-up code
    self.mediator.clear ()

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief icon data
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
ICON_DATA = '''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'''

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Extension class
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Extension (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, mediator):
    self.id = 'extension-builder'
    self.name = 'Extension Builder'
    self.author = 'Eduardo Aguiar'
    self.version = '0.1.0'
    self.description = 'IDE for extensions'
    self.mediator = mediator.copy ()
    self.icon_data = ICON_DATA

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Start extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def start (self):
    self.mediator.call ('toolbox.add', self.id, self.icon_data, 'Extension\nBuilder', self.on_activate)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Stop extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def stop (self):
    self.mediator.call ('ui.working-area.del', self.id)
    self.mediator.call ('toolbox.remove', self.id)
    self.mediator.clear ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief event: on_activate
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_activate (self, item_id):
    working_area = self.mediator.call ('ui.working-area.get', self.id)

    if not working_area:
      widget = Widget (self.mediator)

      working_area = self.mediator.call ('ui.working-area.new', self.id)
      working_area.set_default_size (600, 400)
      working_area.set_title (self.name)
      working_area.set_icon (self.icon_data)
      working_area.set_widget (widget)

    working_area.show ()
