// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <mobius/io/file_descriptor_reader.h>
#include <mobius/exception_posix.inc>
#include <unistd.h>
//#include <fcntl.h>

namespace mobius
{
namespace io
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief create object from an UNIX file descriptor
//! \param fd file descriptor
//! \param is_owner whether object is the owner of the file descriptor
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
file_descriptor_reader::file_descriptor_reader (int fd, bool is_owner)
  : fd_ (fd), is_owner_ (is_owner)
{
  // evaluate file size
  off_t size = lseek (fd_, 0, 2);
  if (size == -1)
    throw std::runtime_error (MOBIUS_EXCEPTION_POSIX);
  _set_size (size);

  // set position to the start of the file
  lseek (fd_, 0, 0);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief destructor. Close file descriptor, if object is its owner
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
file_descriptor_reader::~file_descriptor_reader ()
{
  if (is_owner_)
    close (fd_);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief set read position
//! \param offset offset in bytes
//! \param w either beginning, current or end
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
file_descriptor_reader::seek (offset_type offset, whence w)
{
  int iw = 0;

  if (w == whence::beginning)
    iw = SEEK_SET;

  else if (w == whence::current)
    iw = SEEK_CUR;

  else if (w == whence::end)
    iw = SEEK_END;

  off_t rc = lseek (fd_, offset, iw);
  if (rc == -1)
    throw std::runtime_error (MOBIUS_EXCEPTION_POSIX);

  _set_eof (static_cast <size_type> (rc) < get_size ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get read position
//! \return read position in bytes from the beginning of data
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
file_descriptor_reader::size_type
file_descriptor_reader::tell () const
{
  off_t offset = lseek (fd_, 0, SEEK_CUR);
  if (offset == -1)
    throw std::runtime_error (MOBIUS_EXCEPTION_POSIX);

  return offset;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief read bytes from reader
//! \param size size in bytes
//! \return bytearray containing data
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const mobius::bytearray
file_descriptor_reader::read (size_type size)
{
  mobius::bytearray buffer (size);
  ssize_t rc = ::read (fd_, buffer.data (), size);
  size_type count = 0;

  if (rc < 0)
    {
      if (errno == EAGAIN)
        ;  // 0 bytes returned from a NONBLOCK reading
      else
        throw std::runtime_error (MOBIUS_EXCEPTION_POSIX);
    }

  else if (rc == 0)
    _set_eof (true);

  else
    count = rc;

  buffer.resize (count);
  return buffer;
}

} // namespace mobius
} // namespace io