// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <mobius/io/file_reader.h>
#include <mobius/exception_posix.inc>

namespace mobius
{
namespace io
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief constructor
//! \param path local path to file
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
file_reader::file_reader (const std::string& path)
{
  fp_ = fopen (path.c_str (), "rb");
  if (!fp_)
    throw std::runtime_error (MOBIUS_EXCEPTION_POSIX);

  // evaluate file size
  if (fseek (fp_, 0, SEEK_END) == -1)
    throw std::runtime_error (MOBIUS_EXCEPTION_POSIX);

  _set_size (ftell (fp_));

  // set position to the start of the file
  ::rewind (fp_);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief destructor
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
file_reader::~file_reader ()
{
  fclose (fp_);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief set read position
//! \param offset offset in bytes
//! \param w either beginning, current or end
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
file_reader::seek (offset_type offset, whence w)
{
  int iw = 0;

  if (w == whence::beginning)
    iw = SEEK_SET;

  else if (w == whence::current)
    iw = SEEK_CUR;

  else if (w == whence::end)
    iw = SEEK_END;

  if (fseek (fp_, offset, iw) == -1)
    throw std::runtime_error (MOBIUS_EXCEPTION_POSIX);

  _set_eof (static_cast <size_type> (ftell (fp_)) < get_size ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get read position
//! \return read position in bytes from the beginning of data
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
file_reader::size_type
file_reader::tell () const
{
  long rc = ftell (fp_);
  if (rc == -1)
    throw std::runtime_error (MOBIUS_EXCEPTION_POSIX);

  return rc;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief read bytes from reader
//! \param size size in bytes
//! \return bytearray containing data
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const mobius::bytearray
file_reader::read (size_type size)
{
  mobius::bytearray buffer (size);
  size_t count = fread (buffer.data (), 1, size, fp_);

  if (count == 0)
    {
      if (ferror (fp_))
        throw std::runtime_error (MOBIUS_EXCEPTION_POSIX);

      else if (feof (fp_))
        _set_eof (true);
    }

  buffer.resize (count);
  return buffer;
}

} // namespace mobius
} // namespace io