# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import pymobius.registry.user_accounts
import pymobius.registry.cached_credentials
import pymobius.registry.main
from pymobius.registry import *
import pymobius.operating_system
import pymobius.app.chromium
import mobius
import binascii
import re
import struct
import traceback

ANT_ID = 'turing'
ANT_NAME = 'Turing'
ANT_VERSION = '1.0'

REGEX_WORDS = re.compile ("(\w[\w']*\w|\w)")

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Generic dataholder class
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class dataholder (object):
  pass

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Ant: Turing
# @author Eduardo Aguiar
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Ant (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, item):
    self.name = 'Turing Cryptographic Agent'
    self.version = '1.0'
    self.__item = item

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Run ant
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def run (self):
    if not self.__item.datasource:
      return

    self.__retrieve_dpapi_system_master_keys ()
    self.__retrieve_passwords ()
    self.__retrieve_password_hashes ()
    self.__retrieve_dpapi_credhist_hashes ()
    self.__test_passwords ()
    self.__create_passwords_from_hashes ()
    self.__retrieve_dpapi_user_master_keys ()
    self.__retrieve_win_credentials_passwords ()
    self.__retrieve_chromium_passwords ()
    self.__retrieve_ie_passwords ()
    self.__test_passwords ()
    self.__save_data ()

    self.user_mk = dict (self.__dpapi_user_master_keys)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save data into model
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __save_data (self):
    c = self.__item.case
    transaction = c.new_transaction ()

    # remove old data
    self.__item.remove_passwords ()
    self.__item.remove_password_hashes ()

    # save passwords
    for tmp_p in self.__passwords:
      p = self.__item.new_password (tmp_p.type, tmp_p.value, tmp_p.description)
      p.set_attribute ('item', self.__item.name)

      for name, value in tmp_p.metadata:
        p.set_attribute (name, value)

    # save password hashes
    for tmp_h in self.__password_hashes:
      h = self.__item.new_password_hash (tmp_h.type, tmp_h.value, tmp_h.description)

      if tmp_h.password != None:
        h.password = tmp_h.password

      for name, value in tmp_h.metadata:
        h.set_attribute (name, value)

    self.__item.set_ant (ANT_ID, ANT_NAME, ANT_VERSION)
    transaction.commit ()
  
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve DPAPI system master keys
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_dpapi_system_master_keys (self):
    self.__dpapi_system = []
    self.__dpapi_sys_master_keys = {}

    try:
      ant = pymobius.registry.main.Ant (self.__item)

      for registry in ant.get_data ():
        self.__retrieve_dpapi_system_from_registry (registry)
    except Exception, e:
      mobius.core.log ('WRN %s' % str (e))

    self.__retrieve_dpapi_system_master_keys_files ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve DPAPI_SYSTEM values from registry
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_dpapi_system_from_registry (self, registry):
    key = registry.get_key_by_path ('\\HKLM\\SECURITY\\Policy\\Secrets\\DPAPI_SYSTEM')
    if not key:
      return

    data = key.get_data_by_path ('CurrVal\\(default)')
    if data and data.size == 44:
      revision, local_credential, user_credential = struct.unpack ('<I20s20s', data.data)
      self.__dpapi_system.append (local_credential)
      self.__dpapi_system.append (user_credential)
      
    data = key.get_data_by_path ('OldVal\\(default)')
    if data and data.size == 44:
      revision, local_credential, user_credential = struct.unpack ('<I20s20s', data.data)
      self.__dpapi_system.append (local_credential)
      self.__dpapi_system.append (user_credential)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve and decrypt DPAPI system master keys
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_dpapi_system_master_keys_files (self):
    count = 0

    for opsys in pymobius.operating_system.scan (self.__item):
      for root in opsys.get_root_folders ():

        # Retrieve artifacts
        folder = root.get_child_by_path ('windows/system32/microsoft/protect/s-1-5-18')
        if folder:
          count += self.__retrieve_dpapi_system_master_keys_from_folder (folder)

          folder = folder.get_child_by_name ('User')
          if folder:
            count += self.__retrieve_dpapi_system_master_keys_from_folder (folder)

    mobius.core.log ('INF %d/%d system master keys decrypted' % (len (self.__dpapi_sys_master_keys), count))

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve and decrypt DPAPI system master keys from folder
  # @param folder Folder
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_dpapi_system_master_keys_from_folder (self, folder):
    count = 0

    for child in folder.get_children ():
      if child.is_file () and child.name.count ('-') == 4:
        self.__retrieve_dpapi_system_master_key (child)
        count += 1

    return count

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve and decrypt DPAPI system master key from file
  # @param f File
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_dpapi_system_master_key (self, f):

    # parse master key file
    reader = f.new_reader ()
    if not reader:
      return

    mkf = mobius.turing.dpapi.master_key_file (reader)
    if not mkf:
      return

    mk = mkf.master_key

    # try to decrypt using DPAPI_SYSTEM keys
    for key in self.__dpapi_system:
      mk.decrypt_with_key (key)

      if mk.is_decrypted ():
        self.__dpapi_sys_master_keys[mkf.guid] = mk.plain_text
        f.set_handled ()
        return

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve passwords
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_passwords (self):
    self.__passwords = []

    self.__retrieve_passwords_from_registry ()
    self.__retrieve_wifi_passwords_from_files ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve passwords from registry
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_passwords_from_registry (self):
    try:
      ant = pymobius.registry.main.Ant (self.__item)

      for registry in ant.get_data ():
        self.__retrieve_passwords_from_registry_lsa (registry)
        self.__retrieve_passwords_from_registry_outlook (registry)
        self.__retrieve_dpapi_system_from_registry (registry)
    except Exception, e:
      mobius.core.log ('WRN %s' % str(e))

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve passwords from LSA secrets
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_passwords_from_registry_lsa (self, registry):

    for key in registry.get_key_by_mask ('\\HKLM\\SECURITY\\Policy\\Secrets\\*'):
          
      # @see https://msdn.microsoft.com/en-us/library/windows/desktop/aa378826(v=vs.85).aspx
      if key.name == 'DefaultPassword':
        description = 'Automatic logon'
        ptype = 'os.default'
      
      elif key.name == '0083343a-f925-4ed7-b1d6-d95d17a0b57b-RemoteDesktopHelpAssistantAccount':
        description = 'User HelpAssistant logon'
        ptype = 'os.user'

      elif key.name == 'aspnet_WP_PASSWORD':
        description = 'User ASPNET logon'
        ptype = 'os.user'

      elif key.name == 'SCM:{3D14228D-FBE1-11D0-995D-00C04FD919C1}':
        description = 'IIS IWAM'
        ptype = 'app.iis'

      # @see http://nvidia.custhelp.com/app/answers/detail/a_id/3067/~/what-is-nvidia-%E2%80%99updatususer%E2%80%99%3F
      elif key.name == '_SC_nvUpdatusService':
        description = 'User UpdatusUser logon'
        ptype = 'os.user'

      elif key.name.startswith ('_SC_DB2'):
        description = 'DB2'
        ptype = 'app.db2'

      elif key.name.startswith ('_SC_postgresql-') or key.name.startswith ('_SC_pgsql-'):
        description = 'PostgreSQL'
        ptype = 'app.postgresql'

      else:
        description = None
        ptype = None

      # add current and old passwords
      if description:
        currval = get_data_as_string (key.get_data_by_path ('Currval\\(default)'))
        
        if currval:
          p = dataholder ()
          p.type = ptype
          p.value = currval
          p.description = description + ' password'

          p.metadata = []
          p.metadata.append (('source', 'LSA key %s currval' % key.name))

          self.__passwords.append (p)

        oldval = get_data_as_string (key.get_data_by_path ('Oldval\\(default)'))
        
        if oldval:
          p = dataholder ()
          p.type = ptype
          p.value = oldval
          p.description = description + ' old password'

          p.metadata = []
          p.metadata.append (('source', 'LSA key %s oldval' % key.name))

          self.__passwords.append (p)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve passwords for Outlook Express and Outlook 98-2000
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_passwords_from_registry_outlook (self, registry):

    for user_key in registry.get_key_by_mask ('\\HKU\\*'):
      sid = user_key.name
      pssp_key = user_key.get_key_by_path ('Software\\Microsoft\\Protected Storage System Provider\\' + sid)
 
      for subkey in user_key.get_key_by_mask ('Software\\Microsoft\\Internet Account Manager\\Accounts\*'):
        account_name = get_data_as_string (subkey.get_data_by_name ('SMTP Display Name'))
        email_address = get_data_as_string (subkey.get_data_by_name ('SMTP Email Address'))

        # POP3 password
        pop3_value_name = subkey.get_data_by_name ('POP3 Password2')
        pop3_password = self.__get_pssp_password (pssp_key, pop3_value_name)

        if pop3_password:
          p = dataholder ()
          p.type = 'email.pop3'
          p.value = pop3_password
          p.description = "E-mail " + email_address + " POP3 password"

          p.metadata = []
          p.metadata.append (('source', 'PSSP POP3 Password2 value'))
          p.metadata.append (('email', email_address))
          p.metadata.append (('user_sid', sid))
          self.__passwords.append (p)

        # SMTP password
        smtp_value_name = subkey.get_data_by_name ('SMTP Password2')
        smtp_password = self.__get_pssp_password (pssp_key, smtp_value_name)

        if smtp_password:
          p = dataholder ()
          p.type = 'email.smtp'
          p.value = smtp_password
          p.description = "E-mail " + email_address + " SMTP password"

          p.metadata = []
          p.metadata.append (('source', 'PSSP SMTP Password2 value'))
          p.metadata.append (('email', email_address))
          p.metadata.append (('user_sid', sid))
          self.__passwords.append (p)

        # HTTP password
        http_value_name = subkey.get_data_by_name ('HTTPMail Password2')
        http_password = self.__get_pssp_password (pssp_key, http_value_name)

        if http_password:
          p = dataholder ()
          p.type = 'email.http'
          p.value = http_password
          p.description = "E-mail " + email_address + " HTTP password"

          p.metadata = []
          p.metadata.append (('source', 'PSSP HTTPMail Password2 value'))
          p.metadata.append (('email', email_address))
          p.metadata.append (('user_sid', sid))
          self.__passwords.append (p)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get Protected Storage password
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def __get_pssp_password (self, pssp_key, idx_data):
    password = None
    
    if pssp_key and idx_data:
      value_name = unicode (idx_data.data[2:], 'utf-16-le').rstrip ('\0').encode ('utf-8')
      data = pssp_key.get_data_by_path ('Data\\220d5cc1-853a-11d0-84bc-00c04fd43f8f\\417e2d75-84bd-11d0-84bb-00c04fd43f8f\\' + value_name)

      if data:
        password = data.data.rstrip ('\0')

    return password

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve wifi passwords
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_wifi_passwords_from_files (self):
    for opsys in pymobius.operating_system.scan (self.__item):
      for root in opsys.get_root_folders ():
        folder = root.get_child_by_path ('ProgramData/Microsoft/Wlansvc/Profiles/Interfaces')
        if folder:
          for child in folder.get_children ():
            if child.is_folder () and child.name[0] == '{':
              self.__retrieve_wifi_passwords_from_interface_folder (child)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve wifi passwords from interface folder
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_wifi_passwords_from_interface_folder (self, folder):
    for child in folder.get_children ():
      if child.is_file () and child.name.lower ().endswith ('.xml'):
        try:
          self.__retrieve_wifi_passwords_from_xml (child)
        except Exception, e:
          print e

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve wifi passwords from interface folder
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_wifi_passwords_from_xml (self, f):
    if not f or f.is_reallocated:
      return

    reader = f.new_reader ()
    if not reader:
      return

    # parse data from file
    data = reader.read ()
    name = ''
    key_material = ''
    protected = False

    match = re.search ('<name>(.*?)</name>', data, re.IGNORECASE)
    if match:
      name = match.group (1)

    match = re.search ('<protected>(.*?)</protected>', data, re.IGNORECASE)
    if match:
      protected = match.group (1) == 'true'

    match = re.search ('<keyMaterial>(.*?)</keyMaterial>', data, re.IGNORECASE)
    if match:
      key_material = match.group (1).strip ()

    if not protected or name == 'WFD_GROUP_OWNER_PROFILE':
      return
    
    # decrypt password
    b = mobius.turing.dpapi.blob (binascii.unhexlify (key_material))
    mk = self.__dpapi_sys_master_keys.get (b.master_key_guid)

    if mk and b.decrypt (mk):
      value = b.plain_text.rstrip ('\x00')

      if '\x00' not in value:
        p = dataholder ()
        p.type = 'net.wifi'
        p.value = value
        p.description = "Wifi password. SSID: " + name

        p.metadata = []
        p.metadata.append (('Source', f.path.replace ('/', '\\')))
        p.metadata.append (('SSID', name))
        self.__passwords.append (p)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve password hashes
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_password_hashes (self):
    self.__password_hashes = []

    try:
      ant = pymobius.registry.main.Ant (self.__item)

      for reg in ant.get_data ():
        self.__retrieve_password_hashes_from_registry (reg)
    except Exception, e:
      traceback.print_exc ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve password hashes from registry
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_password_hashes_from_registry (self, reg):
    self.__retrieve_password_hashes_from_registry_user_accounts (reg)
    self.__retrieve_password_hashes_from_registry_cached_credential (reg)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve password hashes from user accounts
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_password_hashes_from_registry_user_accounts (self, registry):
    turing = mobius.turing.turing ()

    for account in pymobius.registry.user_accounts.get (registry):
      h_set = set ()

      for h in account.hashes:
        value = binascii.hexlify (h.value)
        k = (h.type, value, h.is_current)

        if k not in h_set:
          h_set.add (k)

          if h.is_current:
            description = 'User ' + account.username + ' password hash'
          else:
            description = 'User ' + account.username + ' old password hash'

          d = dataholder ()
          d.type = h.type
          d.value = value
          d.description = description

          status, d.password = turing.get_hash_password (d.type, d.value)
          if status != 1:
            d.password = None

          d.metadata = []
          d.metadata.append (('user_rid', str (account.rid)))
          d.metadata.append (('user_gid', str (account.gid)))
          d.metadata.append (('username', account.username))
          d.metadata.append (('fullname', account.fullname))
          d.metadata.append (('admin_comment', account.admin_comment))
          d.metadata.append (('user_comment', account.user_comment))
          d.metadata.append (('is_current', str (h.is_current)))
          self.__password_hashes.append (d)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve password hashes from cached credential
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_password_hashes_from_registry_cached_credential (self, registry):
    turing = mobius.turing.turing ()

    for name, credential in pymobius.registry.cached_credentials.get (registry):

      if credential.encryption_algorithm == 0:
        t = 'msdcc1'

      elif credential.encryption_algorithm == 10:
        t = 'msdcc2'

      else:
        mobius.core.log ('DEV Unknown credential encryption algorithm (%d)' % credential.encryption_algorithm)
        t = None

      if t:
        d = dataholder ()
        d.type = t
        d.value = binascii.hexlify (credential.mscachehash)
        d.description = 'User ' + credential.username + ' cached credential hash'

        status, d.password = turing.get_hash_password (d.type, d.value)
        if status != 1:
          d.password = None

        d.metadata = []
        d.metadata.append (('username', credential.username))

        if t == 'msdcc2':
          d.metadata.append (('iterations', '10240'))

        self.__password_hashes.append (d)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve password hashes from CREDHIST files
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_dpapi_credhist_hashes (self):
    self.__dpapi_sha1_hashes = set ()

    for opsys in pymobius.operating_system.scan (self.__item):
      for user_profile in opsys.get_profiles ():
        f = user_profile.get_entry_by_path ('%appdata%/Microsoft/Protect/CREDHIST')

        if f and not f.is_reallocated and f.is_file ():
          self.__retrieve_dpapi_credhist_from_file (f, user_profile.username)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve password hashes from CREDHIST file
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_dpapi_credhist_from_file (self, f, username):

    # parse CREDHIST file
    try:
      reader = f.new_reader ()

      if not reader:
        return

      credhist = mobius.turing.dpapi.credhist_file (reader)

    except Exception, e:
      mobius.core.log ('WRN %s' % str (e))
      return

    f.set_handled ()

    # create list of possible passwords
    case = self.__item.case
    keywords = set (p.value for p in self.__passwords)
    keywords.update (set (p.value for p in case.get_passwords ()))
    keywords.add ('')

    # test passwords
    for keyword in keywords:
      if credhist.decrypt_with_password (keyword):
        h = mobius.crypt.hash_digest ('sha1', keyword.encode ('utf-16'))
        self.__dpapi_sha1_hashes.add (h)

    # get hashes
    turing = mobius.turing.turing ()

    for idx, entry in enumerate (reversed (credhist.entries)):
      if entry.is_decrypted ():
        self.__dpapi_sha1_hashes.add (entry.hash_sha1)

        d = dataholder ()
        d.type = 'nt'
        d.value = binascii.hexlify (entry.hash_ntlm)
        d.description = 'User ' + username + ' old password hash #%d' % (idx + 1)

        status, d.password = turing.get_hash_password (d.type, d.value)
        if status != 1:
          d.password = None
        
        d.metadata = []
        d.metadata.append (('username', username))
        d.metadata.append (('source', f.path.replace ('/', '\\')))
        self.__password_hashes.append (d)
        
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Test passwords
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __test_passwords (self):

    # Test using mobius.turing
    turing = mobius.turing.turing ()
    hashes = [ h for h in self.__password_hashes if h.password == None ]

    for h in hashes:
      status, password = turing.get_hash_password (h.type, h.value)

      if status == 1:
        h.password = password

    # Generate wordlist composed of the following:
    # a. item passwords
    # b. case passwords
    # c. item's user names
    # d. case's user names
    hashes = [ h for h in self.__password_hashes if h.password == None ]
    if not hashes:
      return

    case = self.__item.case
    keywords = set (p.value for p in self.__passwords)
    keywords.update (set (p.value for p in case.get_passwords ()))
    
    for h in self.__password_hashes:
      metadata = dict (h.metadata)
      keywords.update (REGEX_WORDS.findall (metadata.get ('username', '')))
      keywords.update (REGEX_WORDS.findall (metadata.get ('fullname', '')))
      keywords.update (REGEX_WORDS.findall (metadata.get ('admin_comment', '')))
      keywords.update (REGEX_WORDS.findall (metadata.get ('user_comment', '')))

    for h in case.get_password_hashes ():
      keywords.update (REGEX_WORDS.findall (h.get_attribute ('username')))
      keywords.update (REGEX_WORDS.findall (h.get_attribute ('fullname')))
      keywords.update (REGEX_WORDS.findall (h.get_attribute ('admin_comment')))
      keywords.update (REGEX_WORDS.findall (h.get_attribute ('user_comment')))

    keywords = list (keywords)

    # Test keywords
    i = 0
    l = len (keywords)

    while i < l and hashes:
      keyword = keywords[i]
      nt_value = binascii.hexlify (mobius.turing.hash_nt (keyword))
      lm_value = binascii.hexlify (mobius.turing.hash_lm (keyword))
      h_tmp = []

      for h in hashes:
        attrs = dict (h.metadata)

        if h.type == 'nt':
          value = nt_value
          
        elif h.type == 'lm':
          value = lm_value

        elif h.type == 'msdcc1':
          value = binascii.hexlify (mobius.turing.hash_msdcc1 (keyword, attrs['username']))

        elif h.type == 'msdcc2':
          value = binascii.hexlify (mobius.turing.hash_msdcc2 (keyword, attrs['username'], int (attrs['iterations'])))

        if value == h.value:
          h.password = keyword

        else:
          h_tmp.append (h)

      if len (h_tmp) < len (hashes):
        hashes = h_tmp

      i = i + 1

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve DPAPI user master keys
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_dpapi_user_master_keys (self):
    self.__dpapi_user_master_keys = {}

    for opsys in pymobius.operating_system.scan (self.__item):
      for user_profile in opsys.get_profiles ():
        folder = user_profile.get_appdata_entry ('Microsoft/Protect')
        
        if folder and not folder.is_reallocated:
          for child in folder.get_children ():
            if child.is_folder () and child.name.startswith ('S-1-'):
              self.__retrieve_dpapi_user_master_keys_from_folder (child)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve DPAPI user master keys from SID folder
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_dpapi_user_master_keys_from_folder (self, folder):
    sid = folder.name
    count = 0
    count_decrypted = 0

    for child in folder.get_children ():
      if not child.is_reallocated and child.is_file () and child.name.count ('-') == 4:
        guid, mk = self.__retrieve_dpapi_user_master_keys_from_file (child, sid)

        if mk:
          self.__dpapi_user_master_keys[guid] = mk
          count_decrypted += 1

        count += 1

    mobius.core.log ('INF %d/%d user master keys decrypted. SID: %s' % (count_decrypted, count, sid))

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve DPAPI user master keys from MK file
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_dpapi_user_master_keys_from_file (self, f, sid):
    reader = f.new_reader ()

    if not reader:
      return None, None

    try:
      mkf = mobius.turing.dpapi.master_key_file (reader)
    except Exception, e:
      mobius.core.log ('WRN %s' % str (e))
      return None, None

    f.set_handled ()
    mk = mkf.master_key
      
    for h in self.__dpapi_sha1_hashes:
      if mk.decrypt_with_password_hash (sid, h):
        return mkf.guid, mk.plain_text

    keywords = set (p.value for p in self.__passwords)

    for keyword in keywords:
      if mk.decrypt_with_password (sid, keyword):
        return mkf.guid, mk.plain_text
    
    return None, None

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve Chromium browsers passwords
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_chromium_passwords (self):
    model = pymobius.app.chromium.model (self.__item)
    
    for profile in model.get_profiles ():
      try:
        self.__retrieve_chromium_passwords_from_profile (profile)
      except Exception, e:
        mobius.core.log ('WRN %s' % str (e))
        
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve Chromium passwords from profile
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_chromium_passwords_from_profile (self, profile):
    urls = set ()

    # get passwords from profile
    for p in profile.get_stored_passwords ():
      b = mobius.turing.dpapi.blob (p.password)
      mk = self.__dpapi_user_master_keys.get (b.master_key_guid)

      if mk and b.decrypt (mk):
        if b.plain_text and ord (b.plain_text[-1]) > 16:
          d = dataholder ()
          d.type = 'net.http'
          d.value = b.plain_text.rstrip ('\0')
          d.description = "Web password. URL: " + p.url

          d.metadata = []
          d.metadata.append (('URL', p.url))
          d.metadata.append (('User ID', p.username))
          d.metadata.append (('Profile path', profile.path))
          d.metadata.append (('Application', profile.app_name))

          self.__passwords.append (d)
          urls.add (p.url)

    # add URLs to Turing, as IE entropy hashes
    turing = mobius.turing.turing ()
    transaction = turing.new_transaction ()

    for url in urls:
      h = binascii.hexlify (mobius.turing.hash_ie_entropy (url))
      turing.set_hash ('ie.entropy', h, url)

    transaction.commit ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve Win Credentials passwords
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_win_credentials_passwords (self):
    for opsys in pymobius.operating_system.scan (self.__item):
      for user_profile in opsys.get_profiles ():
        try:
          folder = user_profile.get_entry_by_path ('%appdata%/Microsoft/Credentials')

          if folder and not folder.is_reallocated and folder.is_folder ():
            self.__retrieve_win_credentials_from_folder (folder)

          folder = user_profile.get_entry_by_path ('%localappdata%/Microsoft/Credentials')

          if folder and not folder.is_reallocated and folder.is_folder ():
            self.__retrieve_win_credentials_from_folder (folder)
        except Exception, e:
          mobius.core.log ('WRN %s' % str (e))

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve Win Credentials from folder
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_win_credentials_from_folder (self, folder):
    for child in folder.get_children ():
      if child.is_file () and not child.is_reallocated:
        self.__retrieve_win_credentials_from_file (child)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve Win Credentials from file
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_win_credentials_from_file (self, f):
    reader = f.new_reader ()

    if not reader:
      return

    decoder = mobius.decoder.data_decoder (reader)
    revision = decoder.get_uint32_le ()
    blob_size = decoder.get_uint64_le ()

    if blob_size > 0:
      blob_data = decoder.get_bytearray_by_size (blob_size)

      b = mobius.turing.dpapi.blob (blob_data)
      mk = self.__dpapi_user_master_keys.get (b.master_key_guid)

      if mk and b.decrypt (mk):
        p = self.__create_password_from_win_credential_data (b.plain_text)

        if p:
          p.metadata.append (('Credential file path', f.path.replace ('/', '\\')))
          self.__passwords.append (p)

    f.set_handled ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Create password from Win Credentials data
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __create_password_from_win_credential_data (self, data):
    decoder = mobius.decoder.data_decoder (data)

    header_size = decoder.get_uint32_le ()
    if header_size == 0:
      return None

    blob_size = decoder.get_uint32_le ()
    u1 = decoder.get_uint32_le ()
    u2 = decoder.get_uint32_le ()
    u3 = decoder.get_uint32_le ()
    last_update = decoder.get_nt_datetime ()
    u4 = decoder.get_uint32_le ()
    u5 = decoder.get_uint32_le ()
    flags = decoder.get_uint32_le ()
    u7 = decoder.get_uint32_le ()
    u8 = decoder.get_uint32_le ()

    domain_size = decoder.get_uint32_le ()
    domain = decoder.get_string_by_size (domain_size, 'utf-16le')
    
    s1_size = decoder.get_uint32_le ()
    s1 = decoder.get_string_by_size (s1_size, 'utf-16le')
    
    s2_size = decoder.get_uint32_le ()
    s2 = decoder.get_string_by_size (s2_size, 'utf-16le')
    
    s3_size = decoder.get_uint32_le ()
    s3 = decoder.get_string_by_size (s3_size, 'utf-16le')
    
    username_size = decoder.get_uint32_le ()
    username = decoder.get_string_by_size (username_size, 'utf-16le')
    
    password_size = decoder.get_uint32_le ()
    password_data = decoder.get_bytearray_by_size (password_size)
    p = None

    if domain.startswith ('WindowsLive:'):
      value = password_data.decode ('utf-16le')

      if value:
        p = dataholder ()
        p.type = 'net.account'
        p.value = password_data.decode ('utf-16le')
        p.description = 'Windows Live password. Account: %s' % username

        p.metadata = []
        p.metadata.append (('Domain', domain))
        p.metadata.append (('Account', username))

    else:
      mobius.core.log ('DEV Win Credential not handled: ' + domain)

      if password_data:
        mobius.core.log ('DEV Win Credential password: ' + pymobius.dump (password_data))

    return p

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve IE passwords
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_ie_passwords (self):
    try:
      ant = pymobius.registry.main.Ant (self.__item)

      for registry in ant.get_data ():
        self.__retrieve_ie_passwords_from_registry (registry)
    except Exception, e:
      mobius.core.log ('WRN %s' % str(e))

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve IE password from registry
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_ie_passwords_from_registry (self, registry):
    turing = mobius.turing.turing ()

    for value in registry.get_value_by_mask ('\\HKU\\*\\Software\\Microsoft\\Internet Explorer\\IntelliForms\\Storage2\\*'):
      status, url = turing.get_hash_password ('ie.entropy', value.name.lower ())

      if status == 1:    # found
        entropy = (url + '\0').encode ('utf-16le')
        blob = mobius.turing.dpapi.blob (value.data.data)
        mk = self.__dpapi_user_master_keys.get (blob.master_key_guid)

        if mk and blob.decrypt (mk, entropy):
          p = self.__retrieve_ie_passwords_from_data (blob.plain_text, url)
          if p:
            p.metadata.append (('Source', 'Registry value %s' % value.path))

      else:
        mobius.core.log ('INF IE entropy not found: %s' % value.name)
              
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve IE password from decrypted data
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_ie_passwords_from_data (self, data, url):
    decoder = mobius.decoder.data_decoder (data)

    header_size = decoder.get_uint32_le ()
    index_size = decoder.get_uint32_le ()
    data_size = decoder.get_uint32_le ()

    index_offset = header_size
    data_offset = header_size + index_size
    
    signature = decoder.get_string_by_size (4)
    if signature != 'WICK':
      return

    pos = decoder.get_uint32_le ()
    count = decoder.get_uint32_le ()
    values = []
    decoder.seek (pos + index_offset)
    
    for i in xrange (count):
      offset = decoder.get_uint32_le ()
      timestamp = decoder.get_nt_datetime ()
      size = decoder.get_uint32_le ()
      value = unicode (data[data_offset + offset: data_offset + offset + size * 2], 'utf-16le')
      values.append (value)

    username, password = values
    p = None

    if password:
      p = dataholder ()
      p.type = 'net.http'
      p.value = password
      p.description = "Web password. URL: " + url

      p.metadata = []
      p.metadata.append (('URL', url))
      p.metadata.append (('User ID', username))
      p.metadata.append (('Application', 'Internet Explorer'))
    
    return p
    
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Create passwords from revealed password hashes
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __create_passwords_from_hashes (self):
    hashes = [ h for h in self.__password_hashes if h.password != None ]

    turing = mobius.turing.turing ()
    transaction = turing.new_transaction ()

    for h in hashes:
      p = dataholder ()
      p.value = h.password
      p.type = 'os.user'
      p.description = h.description.replace (' hash', '')
      p.metadata = h.metadata[:]
      p.metadata.append (('hash_type', h.type))
      p.metadata.append (('hash_value', h.value))
      self.__passwords.append (p)

      turing.set_hash (h.type, h.value, h.password)
      
    transaction.commit ()
