// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "cipher_rc2.h"
#include <mobius/exception.inc>
#include <cstdint>
#include <stdexcept>

namespace mobius
{
namespace crypt
{
namespace
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief PITABLE
//! \see RFC 2268 - section 2
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static constexpr std::uint8_t PITABLE[] =
{
  0xd9, 0x78, 0xf9, 0xc4, 0x19, 0xdd, 0xb5, 0xed, 
  0x28, 0xe9, 0xfd, 0x79, 0x4a, 0xa0, 0xd8, 0x9d, 
  0xc6, 0x7e, 0x37, 0x83, 0x2b, 0x76, 0x53, 0x8e, 
  0x62, 0x4c, 0x64, 0x88, 0x44, 0x8b, 0xfb, 0xa2, 
  0x17, 0x9a, 0x59, 0xf5, 0x87, 0xb3, 0x4f, 0x13, 
  0x61, 0x45, 0x6d, 0x8d, 0x09, 0x81, 0x7d, 0x32, 
  0xbd, 0x8f, 0x40, 0xeb, 0x86, 0xb7, 0x7b, 0x0b, 
  0xf0, 0x95, 0x21, 0x22, 0x5c, 0x6b, 0x4e, 0x82, 
  0x54, 0xd6, 0x65, 0x93, 0xce, 0x60, 0xb2, 0x1c, 
  0x73, 0x56, 0xc0, 0x14, 0xa7, 0x8c, 0xf1, 0xdc, 
  0x12, 0x75, 0xca, 0x1f, 0x3b, 0xbe, 0xe4, 0xd1, 
  0x42, 0x3d, 0xd4, 0x30, 0xa3, 0x3c, 0xb6, 0x26, 
  0x6f, 0xbf, 0x0e, 0xda, 0x46, 0x69, 0x07, 0x57, 
  0x27, 0xf2, 0x1d, 0x9b, 0xbc, 0x94, 0x43, 0x03, 
  0xf8, 0x11, 0xc7, 0xf6, 0x90, 0xef, 0x3e, 0xe7, 
  0x06, 0xc3, 0xd5, 0x2f, 0xc8, 0x66, 0x1e, 0xd7, 
  0x08, 0xe8, 0xea, 0xde, 0x80, 0x52, 0xee, 0xf7, 
  0x84, 0xaa, 0x72, 0xac, 0x35, 0x4d, 0x6a, 0x2a, 
  0x96, 0x1a, 0xd2, 0x71, 0x5a, 0x15, 0x49, 0x74, 
  0x4b, 0x9f, 0xd0, 0x5e, 0x04, 0x18, 0xa4, 0xec, 
  0xc2, 0xe0, 0x41, 0x6e, 0x0f, 0x51, 0xcb, 0xcc, 
  0x24, 0x91, 0xaf, 0x50, 0xa1, 0xf4, 0x70, 0x39, 
  0x99, 0x7c, 0x3a, 0x85, 0x23, 0xb8, 0xb4, 0x7a, 
  0xfc, 0x02, 0x36, 0x5b, 0x25, 0x55, 0x97, 0x31, 
  0x2d, 0x5d, 0xfa, 0x98, 0xe3, 0x8a, 0x92, 0xae, 
  0x05, 0xdf, 0x29, 0x10, 0x67, 0x6c, 0xba, 0xc9, 
  0xd3, 0x00, 0xe6, 0xcf, 0xe1, 0x9e, 0xa8, 0x2c, 
  0x63, 0x16, 0x01, 0x3f, 0x58, 0xe2, 0x89, 0xa9, 
  0x0d, 0x38, 0x34, 0x1b, 0xab, 0x33, 0xff, 0xb0, 
  0xbb, 0x48, 0x0c, 0x5f, 0xb9, 0xb1, 0xcd, 0x2e, 
  0xc5, 0xf3, 0xdb, 0x47, 0xe5, 0xa5, 0x9c, 0x77, 
  0x0a, 0xa6, 0x20, 0x68, 0xfe, 0x7f, 0xc1, 0xad
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Left rotate value by n bits
//! \param v Value
//! \param n Number of bits
//! \return Value left rotated by n bits
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
inline constexpr std::uint16_t
left_rotate (std::uint16_t v, int n) noexcept
{
  return (v << n) | (v >> (16 - n));
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Right rotate value by n bits
//! \param v Value
//! \param n Number of bits
//! \return Value right rotated by n bits
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
inline constexpr std::uint16_t
right_rotate (std::uint16_t v, int n) noexcept
{
  return (v >> n) | (v << (16 - n));
}

} // namespace

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Constructor
//! \param key encryption/decryption password
//! \param mode cipher block mode
//! \param iv initialization vector
//! \see RFC 2268 - section 2
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
cipher_rc2::cipher_rc2 (
  const mobius::bytearray& key,
  const std::string& mode,
  const mobius::bytearray& iv
)
  : cipher_block (8, mode, iv)
{
  if (key.size () < 1 || key.size () > 127)
    throw std::out_of_range (MOBIUS_EXCEPTION_MSG ("Invalid key size"));

  std::uint8_t L[128];
  std::copy (key.begin (), key.end (), L);

  const int T8 = key.size ();

  for (auto i = key.size (); i < 128; i++)
    L[i] = PITABLE[(L[i-1] + L[i - T8]) & 0xff];
  
  L[128 - T8] = PITABLE[L[128 - T8]];
  
  for (int i = 127 - T8; i >= 0; i--)
    L[i] = PITABLE[L[i + 1] ^ L[i + T8]];
  
  for (int i = 0; i < 64; i++)
    k_[i] = std::uint16_t (L[i * 2]) | (std::uint16_t (L[(i * 2) + 1]) << 8);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Encrypt data block
//! \param data data block
//! \see RFC 2268, section 3
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
cipher_rc2::encrypt_block (mobius::bytearray& data) noexcept
{
  // Initialize R[0..3]
  std::uint16_t R[4];

  R[0] = std::uint16_t (data[0]) | (std::uint16_t (data[1]) << 8);
  R[1] = std::uint16_t (data[2]) | (std::uint16_t (data[3]) << 8);
  R[2] = std::uint16_t (data[4]) | (std::uint16_t (data[5]) << 8);
  R[3] = std::uint16_t (data[6]) | (std::uint16_t (data[7]) << 8);
  
  // five mixing
  R[0] = left_rotate (R[0] + (R[2] & R[3]) + (~R[3] & R[1]) + k_[0], 1);
  R[1] = left_rotate (R[1] + (R[3] & R[0]) + (~R[0] & R[2]) + k_[1], 2);
  R[2] = left_rotate (R[2] + (R[0] & R[1]) + (~R[1] & R[3]) + k_[2], 3);
  R[3] = left_rotate (R[3] + (R[1] & R[2]) + (~R[2] & R[0]) + k_[3], 5);

  R[0] = left_rotate (R[0] + (R[2] & R[3]) + (~R[3] & R[1]) + k_[4], 1);
  R[1] = left_rotate (R[1] + (R[3] & R[0]) + (~R[0] & R[2]) + k_[5], 2);
  R[2] = left_rotate (R[2] + (R[0] & R[1]) + (~R[1] & R[3]) + k_[6], 3);
  R[3] = left_rotate (R[3] + (R[1] & R[2]) + (~R[2] & R[0]) + k_[7], 5);

  R[0] = left_rotate (R[0] + (R[2] & R[3]) + (~R[3] & R[1]) + k_[8], 1);
  R[1] = left_rotate (R[1] + (R[3] & R[0]) + (~R[0] & R[2]) + k_[9], 2);
  R[2] = left_rotate (R[2] + (R[0] & R[1]) + (~R[1] & R[3]) + k_[10], 3);
  R[3] = left_rotate (R[3] + (R[1] & R[2]) + (~R[2] & R[0]) + k_[11], 5);

  R[0] = left_rotate (R[0] + (R[2] & R[3]) + (~R[3] & R[1]) + k_[12], 1);
  R[1] = left_rotate (R[1] + (R[3] & R[0]) + (~R[0] & R[2]) + k_[13], 2);
  R[2] = left_rotate (R[2] + (R[0] & R[1]) + (~R[1] & R[3]) + k_[14], 3);
  R[3] = left_rotate (R[3] + (R[1] & R[2]) + (~R[2] & R[0]) + k_[15], 5);

  R[0] = left_rotate (R[0] + (R[2] & R[3]) + (~R[3] & R[1]) + k_[16], 1);
  R[1] = left_rotate (R[1] + (R[3] & R[0]) + (~R[0] & R[2]) + k_[17], 2);
  R[2] = left_rotate (R[2] + (R[0] & R[1]) + (~R[1] & R[3]) + k_[18], 3);
  R[3] = left_rotate (R[3] + (R[1] & R[2]) + (~R[2] & R[0]) + k_[19], 5);

  // one mashing
  R[0] += k_[R[3] & 0x3f];
  R[1] += k_[R[0] & 0x3f];
  R[2] += k_[R[1] & 0x3f];
  R[3] += k_[R[2] & 0x3f];

  // six mixing
  R[0] = left_rotate (R[0] + (R[2] & R[3]) + (~R[3] & R[1]) + k_[20], 1);
  R[1] = left_rotate (R[1] + (R[3] & R[0]) + (~R[0] & R[2]) + k_[21], 2);
  R[2] = left_rotate (R[2] + (R[0] & R[1]) + (~R[1] & R[3]) + k_[22], 3);
  R[3] = left_rotate (R[3] + (R[1] & R[2]) + (~R[2] & R[0]) + k_[23], 5);

  R[0] = left_rotate (R[0] + (R[2] & R[3]) + (~R[3] & R[1]) + k_[24], 1);
  R[1] = left_rotate (R[1] + (R[3] & R[0]) + (~R[0] & R[2]) + k_[25], 2);
  R[2] = left_rotate (R[2] + (R[0] & R[1]) + (~R[1] & R[3]) + k_[26], 3);
  R[3] = left_rotate (R[3] + (R[1] & R[2]) + (~R[2] & R[0]) + k_[27], 5);

  R[0] = left_rotate (R[0] + (R[2] & R[3]) + (~R[3] & R[1]) + k_[28], 1);
  R[1] = left_rotate (R[1] + (R[3] & R[0]) + (~R[0] & R[2]) + k_[29], 2);
  R[2] = left_rotate (R[2] + (R[0] & R[1]) + (~R[1] & R[3]) + k_[30], 3);
  R[3] = left_rotate (R[3] + (R[1] & R[2]) + (~R[2] & R[0]) + k_[31], 5);

  R[0] = left_rotate (R[0] + (R[2] & R[3]) + (~R[3] & R[1]) + k_[32], 1);
  R[1] = left_rotate (R[1] + (R[3] & R[0]) + (~R[0] & R[2]) + k_[33], 2);
  R[2] = left_rotate (R[2] + (R[0] & R[1]) + (~R[1] & R[3]) + k_[34], 3);
  R[3] = left_rotate (R[3] + (R[1] & R[2]) + (~R[2] & R[0]) + k_[35], 5);

  R[0] = left_rotate (R[0] + (R[2] & R[3]) + (~R[3] & R[1]) + k_[36], 1);
  R[1] = left_rotate (R[1] + (R[3] & R[0]) + (~R[0] & R[2]) + k_[37], 2);
  R[2] = left_rotate (R[2] + (R[0] & R[1]) + (~R[1] & R[3]) + k_[38], 3);
  R[3] = left_rotate (R[3] + (R[1] & R[2]) + (~R[2] & R[0]) + k_[39], 5);

  R[0] = left_rotate (R[0] + (R[2] & R[3]) + (~R[3] & R[1]) + k_[40], 1);
  R[1] = left_rotate (R[1] + (R[3] & R[0]) + (~R[0] & R[2]) + k_[41], 2);
  R[2] = left_rotate (R[2] + (R[0] & R[1]) + (~R[1] & R[3]) + k_[42], 3);
  R[3] = left_rotate (R[3] + (R[1] & R[2]) + (~R[2] & R[0]) + k_[43], 5);

  // one mashing
  R[0] += k_[R[3] & 0x3f];
  R[1] += k_[R[0] & 0x3f];
  R[2] += k_[R[1] & 0x3f];
  R[3] += k_[R[2] & 0x3f];

  // five mixing
  R[0] = left_rotate (R[0] + (R[2] & R[3]) + (~R[3] & R[1]) + k_[44], 1);
  R[1] = left_rotate (R[1] + (R[3] & R[0]) + (~R[0] & R[2]) + k_[45], 2);
  R[2] = left_rotate (R[2] + (R[0] & R[1]) + (~R[1] & R[3]) + k_[46], 3);
  R[3] = left_rotate (R[3] + (R[1] & R[2]) + (~R[2] & R[0]) + k_[47], 5);

  R[0] = left_rotate (R[0] + (R[2] & R[3]) + (~R[3] & R[1]) + k_[48], 1);
  R[1] = left_rotate (R[1] + (R[3] & R[0]) + (~R[0] & R[2]) + k_[49], 2);
  R[2] = left_rotate (R[2] + (R[0] & R[1]) + (~R[1] & R[3]) + k_[50], 3);
  R[3] = left_rotate (R[3] + (R[1] & R[2]) + (~R[2] & R[0]) + k_[51], 5);

  R[0] = left_rotate (R[0] + (R[2] & R[3]) + (~R[3] & R[1]) + k_[52], 1);
  R[1] = left_rotate (R[1] + (R[3] & R[0]) + (~R[0] & R[2]) + k_[53], 2);
  R[2] = left_rotate (R[2] + (R[0] & R[1]) + (~R[1] & R[3]) + k_[54], 3);
  R[3] = left_rotate (R[3] + (R[1] & R[2]) + (~R[2] & R[0]) + k_[55], 5);

  R[0] = left_rotate (R[0] + (R[2] & R[3]) + (~R[3] & R[1]) + k_[56], 1);
  R[1] = left_rotate (R[1] + (R[3] & R[0]) + (~R[0] & R[2]) + k_[57], 2);
  R[2] = left_rotate (R[2] + (R[0] & R[1]) + (~R[1] & R[3]) + k_[58], 3);
  R[3] = left_rotate (R[3] + (R[1] & R[2]) + (~R[2] & R[0]) + k_[59], 5);

  R[0] = left_rotate (R[0] + (R[2] & R[3]) + (~R[3] & R[1]) + k_[60], 1);
  R[1] = left_rotate (R[1] + (R[3] & R[0]) + (~R[0] & R[2]) + k_[61], 2);
  R[2] = left_rotate (R[2] + (R[0] & R[1]) + (~R[1] & R[3]) + k_[62], 3);
  R[3] = left_rotate (R[3] + (R[1] & R[2]) + (~R[2] & R[0]) + k_[63], 5);
  
  // update data
  data[0] = R[0];
  data[1] = R[0] >> 8;
  data[2] = R[1];
  data[3] = R[1] >> 8;
  data[4] = R[2];
  data[5] = R[2] >> 8;
  data[6] = R[3];
  data[7] = R[3] >> 8;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Decrypt data block
//! \param data data block
//! \see RFC 2268, section 4
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
cipher_rc2::decrypt_block (mobius::bytearray& data) noexcept
{
  // Initialize R[0..3]
  std::uint16_t R[4];

  R[0] = std::uint16_t (data[0]) | (std::uint16_t (data[1]) << 8);
  R[1] = std::uint16_t (data[2]) | (std::uint16_t (data[3]) << 8);
  R[2] = std::uint16_t (data[4]) | (std::uint16_t (data[5]) << 8);
  R[3] = std::uint16_t (data[6]) | (std::uint16_t (data[7]) << 8);
  
  // five mixing
  R[3] = right_rotate (R[3], 5) - (R[1] & R[2]) - (~R[2] & R[0]) - k_[63];
  R[2] = right_rotate (R[2], 3) - (R[0] & R[1]) - (~R[1] & R[3]) - k_[62];
  R[1] = right_rotate (R[1], 2) - (R[3] & R[0]) - (~R[0] & R[2]) - k_[61];
  R[0] = right_rotate (R[0], 1) - (R[2] & R[3]) - (~R[3] & R[1]) - k_[60];

  R[3] = right_rotate (R[3], 5) - (R[1] & R[2]) - (~R[2] & R[0]) - k_[59];
  R[2] = right_rotate (R[2], 3) - (R[0] & R[1]) - (~R[1] & R[3]) - k_[58];
  R[1] = right_rotate (R[1], 2) - (R[3] & R[0]) - (~R[0] & R[2]) - k_[57];
  R[0] = right_rotate (R[0], 1) - (R[2] & R[3]) - (~R[3] & R[1]) - k_[56];

  R[3] = right_rotate (R[3], 5) - (R[1] & R[2]) - (~R[2] & R[0]) - k_[55];
  R[2] = right_rotate (R[2], 3) - (R[0] & R[1]) - (~R[1] & R[3]) - k_[54];
  R[1] = right_rotate (R[1], 2) - (R[3] & R[0]) - (~R[0] & R[2]) - k_[53];
  R[0] = right_rotate (R[0], 1) - (R[2] & R[3]) - (~R[3] & R[1]) - k_[52];

  R[3] = right_rotate (R[3], 5) - (R[1] & R[2]) - (~R[2] & R[0]) - k_[51];
  R[2] = right_rotate (R[2], 3) - (R[0] & R[1]) - (~R[1] & R[3]) - k_[50];
  R[1] = right_rotate (R[1], 2) - (R[3] & R[0]) - (~R[0] & R[2]) - k_[49];
  R[0] = right_rotate (R[0], 1) - (R[2] & R[3]) - (~R[3] & R[1]) - k_[48];

  R[3] = right_rotate (R[3], 5) - (R[1] & R[2]) - (~R[2] & R[0]) - k_[47];
  R[2] = right_rotate (R[2], 3) - (R[0] & R[1]) - (~R[1] & R[3]) - k_[46];
  R[1] = right_rotate (R[1], 2) - (R[3] & R[0]) - (~R[0] & R[2]) - k_[45];
  R[0] = right_rotate (R[0], 1) - (R[2] & R[3]) - (~R[3] & R[1]) - k_[44];

  // one mashing
  R[3] -= k_[R[2] & 0x3f];
  R[2] -= k_[R[1] & 0x3f];
  R[1] -= k_[R[0] & 0x3f];
  R[0] -= k_[R[3] & 0x3f];

  // six mixing
  R[3] = right_rotate (R[3], 5) - (R[1] & R[2]) - (~R[2] & R[0]) - k_[43];
  R[2] = right_rotate (R[2], 3) - (R[0] & R[1]) - (~R[1] & R[3]) - k_[42];
  R[1] = right_rotate (R[1], 2) - (R[3] & R[0]) - (~R[0] & R[2]) - k_[41];
  R[0] = right_rotate (R[0], 1) - (R[2] & R[3]) - (~R[3] & R[1]) - k_[40];

  R[3] = right_rotate (R[3], 5) - (R[1] & R[2]) - (~R[2] & R[0]) - k_[39];
  R[2] = right_rotate (R[2], 3) - (R[0] & R[1]) - (~R[1] & R[3]) - k_[38];
  R[1] = right_rotate (R[1], 2) - (R[3] & R[0]) - (~R[0] & R[2]) - k_[37];
  R[0] = right_rotate (R[0], 1) - (R[2] & R[3]) - (~R[3] & R[1]) - k_[36];

  R[3] = right_rotate (R[3], 5) - (R[1] & R[2]) - (~R[2] & R[0]) - k_[35];
  R[2] = right_rotate (R[2], 3) - (R[0] & R[1]) - (~R[1] & R[3]) - k_[34];
  R[1] = right_rotate (R[1], 2) - (R[3] & R[0]) - (~R[0] & R[2]) - k_[33];
  R[0] = right_rotate (R[0], 1) - (R[2] & R[3]) - (~R[3] & R[1]) - k_[32];

  R[3] = right_rotate (R[3], 5) - (R[1] & R[2]) - (~R[2] & R[0]) - k_[31];
  R[2] = right_rotate (R[2], 3) - (R[0] & R[1]) - (~R[1] & R[3]) - k_[30];
  R[1] = right_rotate (R[1], 2) - (R[3] & R[0]) - (~R[0] & R[2]) - k_[29];
  R[0] = right_rotate (R[0], 1) - (R[2] & R[3]) - (~R[3] & R[1]) - k_[28];

  R[3] = right_rotate (R[3], 5) - (R[1] & R[2]) - (~R[2] & R[0]) - k_[27];
  R[2] = right_rotate (R[2], 3) - (R[0] & R[1]) - (~R[1] & R[3]) - k_[26];
  R[1] = right_rotate (R[1], 2) - (R[3] & R[0]) - (~R[0] & R[2]) - k_[25];
  R[0] = right_rotate (R[0], 1) - (R[2] & R[3]) - (~R[3] & R[1]) - k_[24];

  R[3] = right_rotate (R[3], 5) - (R[1] & R[2]) - (~R[2] & R[0]) - k_[23];
  R[2] = right_rotate (R[2], 3) - (R[0] & R[1]) - (~R[1] & R[3]) - k_[22];
  R[1] = right_rotate (R[1], 2) - (R[3] & R[0]) - (~R[0] & R[2]) - k_[21];
  R[0] = right_rotate (R[0], 1) - (R[2] & R[3]) - (~R[3] & R[1]) - k_[20];

  // one mashing
  R[3] -= k_[R[2] & 0x3f];
  R[2] -= k_[R[1] & 0x3f];
  R[1] -= k_[R[0] & 0x3f];
  R[0] -= k_[R[3] & 0x3f];

  // five mixing
  R[3] = right_rotate (R[3], 5) - (R[1] & R[2]) - (~R[2] & R[0]) - k_[19];
  R[2] = right_rotate (R[2], 3) - (R[0] & R[1]) - (~R[1] & R[3]) - k_[18];
  R[1] = right_rotate (R[1], 2) - (R[3] & R[0]) - (~R[0] & R[2]) - k_[17];
  R[0] = right_rotate (R[0], 1) - (R[2] & R[3]) - (~R[3] & R[1]) - k_[16];

  R[3] = right_rotate (R[3], 5) - (R[1] & R[2]) - (~R[2] & R[0]) - k_[15];
  R[2] = right_rotate (R[2], 3) - (R[0] & R[1]) - (~R[1] & R[3]) - k_[14];
  R[1] = right_rotate (R[1], 2) - (R[3] & R[0]) - (~R[0] & R[2]) - k_[13];
  R[0] = right_rotate (R[0], 1) - (R[2] & R[3]) - (~R[3] & R[1]) - k_[12];

  R[3] = right_rotate (R[3], 5) - (R[1] & R[2]) - (~R[2] & R[0]) - k_[11];
  R[2] = right_rotate (R[2], 3) - (R[0] & R[1]) - (~R[1] & R[3]) - k_[10];
  R[1] = right_rotate (R[1], 2) - (R[3] & R[0]) - (~R[0] & R[2]) - k_[9];
  R[0] = right_rotate (R[0], 1) - (R[2] & R[3]) - (~R[3] & R[1]) - k_[8];

  R[3] = right_rotate (R[3], 5) - (R[1] & R[2]) - (~R[2] & R[0]) - k_[7];
  R[2] = right_rotate (R[2], 3) - (R[0] & R[1]) - (~R[1] & R[3]) - k_[6];
  R[1] = right_rotate (R[1], 2) - (R[3] & R[0]) - (~R[0] & R[2]) - k_[5];
  R[0] = right_rotate (R[0], 1) - (R[2] & R[3]) - (~R[3] & R[1]) - k_[4];

  R[3] = right_rotate (R[3], 5) - (R[1] & R[2]) - (~R[2] & R[0]) - k_[3];
  R[2] = right_rotate (R[2], 3) - (R[0] & R[1]) - (~R[1] & R[3]) - k_[2];
  R[1] = right_rotate (R[1], 2) - (R[3] & R[0]) - (~R[0] & R[2]) - k_[1];
  R[0] = right_rotate (R[0], 1) - (R[2] & R[3]) - (~R[3] & R[1]) - k_[0];
  
  // update data
  data[0] = R[0];
  data[1] = R[0] >> 8;
  data[2] = R[1];
  data[3] = R[1] >> 8;
  data[4] = R[2];
  data[5] = R[2] >> 8;
  data[6] = R[3];
  data[7] = R[3] >> 8;
}

} // namespace crypt
} // namespace mobius
