# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021,2022,2023,2024 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import traceback

import mobius


# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Entry
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Entry(object):
    pass


# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Piece
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Piece(object):
    pass


# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Decode file
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def decode(f):
    try:
        return decode_file(f)
    except Exception as e:
        mobius.core.logf('WRN %s %s' % (str(e), traceback.format_exc()))


# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Decode data from TempDL/PHash_XXX.dat
# @see ICH_loadPieces - helper_ICH.pas (line 528)
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def decode_file(f):
    # try to open file
    reader = f.new_reader()
    if not reader:
        return

    # decode header
    decoder = mobius.decoder.data_decoder(reader)

    signature = decoder.get_string_by_size(14)
    num64 = decoder.get_uint64_le()
    len_item = decoder.get_uint32_le()
    hash_sha1 = mobius.encoder.hexstring(decoder.get_bytearray_by_size(20))
    num32 = decoder.get_uint32_le()

    if num64 != 1 or num32 != 1:
        return

    if signature == '__ARESDBP102__':
        version = 2

    elif signature == '__ARESDBP103__':
        version = 3

    else:
        return

    # create data object
    entry = Entry()
    entry.signature = signature
    entry.version = version
    entry.hash_sha1 = hash_sha1
    entry.phash_path = f.path.replace('/', '\\')
    entry.phash_creation_time = f.creation_time
    entry.phash_last_modification_time = f.modification_time
    entry.pieces = []

    # decode pieces
    pos = decoder.tell()
    size = decoder.get_size()

    while pos < size:
        piece = Piece()

        if version == 2:
            piece.progress = 0
            piece.hash_sha1 = mobius.encoder.hexstring(decoder.get_bytearray_by_size(20))
            piece.done = False  # unknown
            pos += 20

        else:  # version 3
            piece.done = decoder.get_uint8() == 1
            piece.progress = decoder.get_uint64_le()
            piece.hash_sha1 = mobius.encoder.hexstring(decoder.get_bytearray_by_size(20))
            pos += 29

        entry.pieces.append(piece)

    return entry
