/* This file is part of the MPFRCPP Library.

  Copyright (c) 2006 -- 2007 Alexey V. Beshenov <bav.272304@gmail.com>.

  The MPFRCPP Library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public License as
  published by the Free Software Foundation; either version 2.1 of the
  License, or (at your option) any later version.

  The MPFRCPP Library is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with the MPFRCPP Library; see the file COPYING.LIB. If
  not, write to the Free Software Foundation, Inc., 51 Franklin Street,
  Fifth Floor, Boston, MA 02110-1301, USA. */

/**
 * @file assignment.hpp
 * @date 2007-03-30
 * Assignment operator overloads for mpfrcpp::Real.
 * INTERNAL HEADER, NOT TO BE USED DIRECTLY.
 */

#ifndef INCLUDED_BY_MPFRCPP
    #error THIS IS AN INTERNAL HEADER ONLY, NOT TO BE USED DIRECTLY
#endif

#ifndef MPFRCPP_REAL_ASSIGNMENT
#define MPFRCPP_REAL_ASSIGNMENT

namespace mpfrcpp {

    //------------------------------------------------------------

    Real& Real::operator= ( const Real& y ) throw() {
        if ( getPrecision() < y.getPrecision() )
            setPrecision( y.getPrecision() );
        mpfr_set ( getMpfrT(), y.getMpfrT(),
                   getParameters().getDefaultRoundMode().getMpfrRndT() );
        return *this;
    }

    //------------------------------------------------------------

    Real& Real::operator= ( const mpfr_t& x ) throw() {
        mpfr_set ( getMpfrT(), x,
                   getParameters().getDefaultRoundMode().getMpfrRndT() );
        return *this;
    }

    //------------------------------------------------------------

    Real& Real::operator= ( const unsigned long int y ) throw() {
        mpfr_set_ui ( getMpfrT(), y,
                      getParameters().getDefaultRoundMode().getMpfrRndT() );
        return *this;
    }

    Real& Real::operator= ( const unsigned int y ) throw() {
        mpfr_set_ui ( getMpfrT(), y,
                      getParameters().getDefaultRoundMode().getMpfrRndT() );
        return *this;
    }

    Real& Real::operator= ( const unsigned short int y ) throw() {
        mpfr_set_ui ( getMpfrT(), y,
                      getParameters().getDefaultRoundMode().getMpfrRndT() );
        return *this;
    }

    //------------------------------------------------------------

    Real& Real::operator= ( const long int y ) throw() {
        mpfr_set_si ( getMpfrT(), y,
                      getParameters().getDefaultRoundMode().getMpfrRndT() );
        return *this;
    }

    Real& Real::operator= ( const int y ) throw() {
        mpfr_set_si ( getMpfrT(), y,
                      getParameters().getDefaultRoundMode().getMpfrRndT() );
        return *this;
    }

    Real& Real::operator= ( const short int y ) throw() {
        mpfr_set_si ( getMpfrT(), y,
                      getParameters().getDefaultRoundMode().getMpfrRndT() );
        return *this;
    }

    //------------------------------------------------------------

    Real& Real::operator= ( const double& y ) throw() {
        mpfr_set_d ( getMpfrT(), y,
                     getParameters().getDefaultRoundMode().getMpfrRndT() );
        return *this;
    }

    //------------------------------------------------------------

    Real& Real::operator= ( const long double& y ) throw() {
        mpfr_set_ld ( getMpfrT(), y,
                      getParameters().getDefaultRoundMode().getMpfrRndT() );
        return *this;
    }

    //------------------------------------------------------------

    Real& Real::operator= ( const mpz_t& y ) throw() {
        mpfr_set_z ( getMpfrT(), y,
                     getParameters().getDefaultRoundMode().getMpfrRndT() );
        return *this;
    }

    Real& Real::operator= ( const mpq_t& y ) throw() {
        mpfr_set_q ( getMpfrT(), y,
                     getParameters().getDefaultRoundMode().getMpfrRndT() );
        return *this;
    }

    Real& Real::operator= ( const mpf_t& y ) throw() {
        if ( getPrecision() < mpf_get_prec( y ) )
            setPrecision( mpf_get_prec( y ) );
        mpfr_set_f ( getMpfrT(), y,
                     getParameters().getDefaultRoundMode().getMpfrRndT() );
        return *this;
    }

    //------------------------------------------------------------

#ifdef GMP_CPP_INTERFACE
    Real& Real::operator= ( const mpz_class& y ) throw() {
        mpfr_set_z ( getMpfrT(), y.get_mpz_t(),
                     getParameters().getDefaultRoundMode().getMpfrRndT() );
        return *this;
    }

    //------------------------------------------------------------

    Real& Real::operator= ( const mpq_class& y ) throw() {
        mpfr_set_q ( getMpfrT(), y.get_mpq_t(),
                     getParameters().getDefaultRoundMode().getMpfrRndT() );
        return *this;
    }

    //------------------------------------------------------------

    Real& Real::operator= ( const mpf_class& y ) throw() {
        if ( getPrecision() < Precision( y.get_prec() ) )
            setPrecision( Precision( y.get_prec() ) );
        mpfr_set_f ( getMpfrT(), y.get_mpf_t(),
                     getParameters().getDefaultRoundMode().getMpfrRndT() );
        return *this;
    }
#endif    // GMP_CPP_INTERFACE

    //------------------------------------------------------------

    Real& Real::operator= ( const std::string& s ) throw() {
        mpfr_set_str( getMpfrT(), s.c_str(), 0,
                      getParameters().getDefaultRoundMode().getMpfrRndT() );
        return *this;
    }

    //------------------------------------------------------------

    Real& Real::operator= ( const bool y ) throw() {
        mpfr_set_ui ( getMpfrT(), static_cast<int>( y ),
                      getParameters().getDefaultRoundMode().getMpfrRndT() );
        return *this;
    }

    //------------------------------------------------------------

} // namespace mpfrcpp

#endif    // MPFRCPP_REAL_ASSIGNMENT
