/* This file is part of the MPFRCPP Library.

   Copyright (c) 2006-2007 Alexey V. Beshenov <bav.272304@gmail.com>.

   The MPFRCPP Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License as
   published by the Free Software Foundation; either version 2.1 of the
   License, or (at your option) any later version.

   The MPFRCPP Library is distributed in the hope that it will be
   useful, but WITHOUT ANY WARRANTY; without even the implied warranty
   of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the MPFRCPP Library; see the file COPYING.LIB. If
   not, write to the Free Software Foundation, Inc., 51 Franklin Street,
   Fifth Floor, Boston, MA 02110-1301, USA. */

/**
 * @file gauss-legendre.cpp
 * @date 2007-07-04
 */

#include <list>
#include <mpfrcpp/mpfrcpp.hpp>
#include <mpfrcpp/extra/std_overloads.hpp>
#include <iostream>
#include <fstream>
#include <vector>
#include <algorithm>
#include <iterator>

//------------------------------------------------------------

/**
 * Integrate --- apply Gaussian quadrature with table of points and
 * weights at [begin, end) and weight function g.
 */

template<typename T, class Iterator, class F, class G>
T Integrate (const F& f = F(),
             const Iterator& begin = Iterator(), const Iterator& end = Iterator(),
             const G& g = G()) {
    T sum = 0;
    for (Iterator i = begin; i < end; i+=2)
        sum += f(*i) / g(*i) * (*(i+1));
    return sum;
}

//------------------------------------------------------------

template<typename T> class IdentityWeightF {
    public:
        T operator() (const T& x) const { return T(1); }
};

template<typename T> class Function {
    public:
        T operator() (const T& x) const { return std::cos(x)/(x*x+T(1)); }
};

//------------------------------------------------------------

int main () {

    mpfr::Library.setPrecision(335);
    std::vector<mpfr::Real> pointsWeights;
    std::ifstream ifile ("gauss-legendre-175.txt");
    if (!ifile) std::exit(1);

    std::copy (std::istream_iterator<mpfr::Real> (ifile),
               std::istream_iterator<mpfr::Real> (),
               std::insert_iterator<std::vector<mpfr::Real> >(pointsWeights, pointsWeights.end()));

/*
    std::copy (pointsWeights.begin(), pointsWeights.end(),
               std::ostream_iterator<mpfr::Real> (std::cout, "\n"));
*/

    std::cout << "\\int_{-1}^1 cos(x)/(x^2+1) = ";
    std::cout << Integrate<mpfr::Real, std::vector<mpfr::Real>::iterator, Function<mpfr::Real>, IdentityWeightF<mpfr::Real> > (Function<mpfr::Real>(), pointsWeights.begin(), pointsWeights.end(), IdentityWeightF<mpfr::Real>());
    std::cout << std::endl;

    return 0;

}
