
#   Copyright (C) 2002 Yannick Gingras <ygingras@ygingras.net>
#   Copyright (C) 2002 Vincent Barbin <vbarbin@openbeatbox.org>

#   This file is part of Open Beat Box.

#   Open Beat Box is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.

#   Open Beat Box is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.

#   You should have received a copy of the GNU General Public License
#   along with Open Beat Box; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


from qt import *
from guiPath import *
from OBBConstants import *
from PixmapSet import *
from OBBFuncts import *

class OBBWidget(QObject):
    def __init__(self, pixmapSet, parent=None, x=0, y=0):
        QObject.__init__(self, parent)
        self.pixmapSet = pixmapSet
        self.subWidgets = []
        self.revChildList = []
        self.curState = DISABLED
        self.pos = QPoint(x, y)
        self.updateMask()

    def addSubWidget(self, widget):
        # stacking order for repaints
        self.subWidgets.append(widget)
        self.updateMask()

        # inverse order for events
        self.revChildList = list(self.subWidgets)
        self.revChildList.reverse()

    def updateMask(self):
        mask = self.pixmapSet.getState(DISABLED).mask()
        self.region = QRegion(mask)
        self.region.translate(self.pos.x(), self.pos.y())

        self.rect = self.region.boundingRect()

        for widget in self.subWidgets:
            opaqueRegion = self.region
            widgetRegion = QRegion(widget.opaqueRegion())
            # map to local coords
            widgetRegion.translate(self.pos.x(), self.pos.y())
            newRegion = opaqueRegion.unite(widgetRegion)
            
            self.region = newRegion
        self.curBoundingRect = self.region.boundingRect()

    def boundingRect(self):
        return self.curBoundingRect
            
    def opaqueRegion(self):
        return self.region

    def size(self):
        return self.boundingRect().size()

    def topLeft(self):
        return self.pos

    def setState(self, state):
        if self.curState == state:
            return

        #self.__pixmapChanged = 1
        self.curState = state
        self.emit(PYSIGNAL("stateChanged()"), ())
        self.parent().repaintWidget(self, self.region)

    def toggleState(self):
        if self.curState == ACTIVATED:
            self.setState(DESACTIVATED)
        else:
            self.setState(ACTIVATED)


    def curPixmap(self, rect):
        """curPixmap : returns the curent state image including the one of
        every subWidget """

        #print "-" * 60
        #printRect(rect)
       
        statePix = self.pixmapSet.getState(self.curState)
        #size = self.region.boundingRect().size()
        size = rect.size()
        optimization = statePix.optimization()

        pixmap = QPixmap( size,
                          statePix.depth(),
                          optimization )

        curMask = statePix.mask()

        pixMask = QBitmap( size, 0, optimization )
        pixMask.fill(Qt.color0)

        bitBlt( pixMask,
                0,
                0,
                curMask,
                rect.topLeft().x(),
                rect.topLeft().y(),
                rect.width(),
                rect.height() )
        
        bitBlt( pixmap,
                0,
                0,
                statePix,
                rect.topLeft().x(),
                rect.topLeft().y(),
                rect.width(),
                rect.height() )
        
        self.blitChilds(pixmap, pixMask, rect)

        pixmap.setMask(pixMask)

        return pixmap

    def blitChilds(self, pixmap, mask, rect):
        """blitChilds : blit all subWidgets on pixmap and
        combine their mask to mask """
        for widget in self.subWidgets:
            if widget.intersects(rect):
                # get the subRect the widget is in
                # all this remapping will just blow my fuse
                #                             -- YGingras
                
                widgetPos  = widget.topLeft()
                widgetRect = widget.boundingRect()
                invalidRect = widgetRect.intersect(rect)
                mappedTopLeft = pointOffset( invalidRect.topLeft(),
                                             widgetPos )
                mappedInvRect = QRect( mappedTopLeft, invalidRect.size() )
                widgetPixmap = widget.curPixmap(mappedInvRect)

                # blit the image
                bitBlt( pixmap,
                        pointOffset(invalidRect.topLeft(), rect.topLeft()),
                        widgetPixmap )

                # blit the mask
                #  1st, check if we need the mask
                needMask = 0
                widgetPos = widget.topLeft()
                needMask |= widgetPos.x() < 0
                needMask |= widgetPos.y() < 0
                needMask |= widgetPixmap.width()+widgetPos.x() > \
                            self.rect.width()
                needMask |= widgetPixmap.height()+widgetPos.y() > \
                            self.rect.height()

                if needMask:
                    widgetMask = widgetPixmap.mask()
                    bitBlt( mask,
                        invalidRect.topLeft().x() - rect.topLeft().x(),
                        invalidRect.topLeft().y() - rect.topLeft().y(),
                        widgetMask,
                        0,
                        0,
                        widgetMask.rect().width(),
                        widgetMask.rect().height(),
                        Qt.OrROP )


    def conbineMask(self, child, childPixmap, mask):
        childMask = childPixmap.mask()
        childPos = child.topLeft()
        bitBlt( mask,
                childPos.x(),
                childPos.y(),                
                childMask,
                0,
                0,
                childMask.width(),
                childMask.height(),
                Qt.OrROP )
        
    def containsPoint(self, point):
        if self.region.contains(point):
            return 1
        
        for widget in self.subWidgets:
            # map to local coords
            mappedPoint = pointOffset(point, self.pos)
            if widget.containsPoint(mappedPoint):
                return 1
        return 0

    def intersects(self, rect):
        selfRect = self.boundingRect()
        return selfRect.intersects(rect)


    #######################
    # parent forwarding
    #######################

    def repaintWidget(self, widget, region):
        #self.__pixmapChanged = 1
        
        # remap region
        newRegion = QRegion(region)
        newRegion.translate(self.pos.x(), self.pos.y())
        
        # call parent
        self.parent().repaintWidget(self, newRegion)

    def captureMouse(self, func=None):
        """captureMouse : Ensure that func will be called when the
        mouse is released."""
        self.parent().captureMouse(func)

    def releaseMouse(self):
        self.parent().releaseMouse()

    def setDrag(self, isDraged=1):
        self.parent().setDrag(isDraged)

    def isDraged(self):
        return self.parent().isDraged()

    def drag(self, e):
        mappedEvent = QMouseEvent( e.type(),
                                   self.mapPointToParent(e.pos()),
                                   e.button(),
                                   e.state() )

        self.parent().drag(mappedEvent)

    def startDrag(self, e):
        mappedEvent = QMouseEvent( e.type(),
                                   self.mapPointToParent(e.pos()),
                                   e.button(),
                                   e.state() )

        self.parent().startDrag(mappedEvent)


    #######################
    # mapping
    #######################

    def mapPointToSelf(self, point):
        return pointOffset(point, self.pos)

    def mapPointToParent(self, point):
        return addOffset(point, self.pos)

    def mapMouseEvent(self, e):
        return QMouseEvent( e.type(),
                            self.mapPointToSelf(e.pos()),
                            e.button(),
                            e.state() )

    def mapWheelEvent(self, e):
        return QWheelEvent( self.mapPointToSelf(e.pos()),
                            e.delta(),
                            e.state(),
                            e.orientation() )

    
    #######################
    # events
    #######################
    
    def mousePressEvent(self, e):
        mappedEvent = self.mapMouseEvent(e)

        for widget in self.revChildList:
            if widget.containsPoint(mappedEvent.pos()):
                return widget.mousePressEvent(mappedEvent)
        
        self.emit(PYSIGNAL('beingPressed(e)'), (e,))

    def mouseReleaseEvent(self, e):
        mappedEvent = self.mapMouseEvent(e)

        for widget in self.revChildList:
            if widget.containsPoint(mappedEvent.pos()):
                return widget.mouseReleaseEvent(mappedEvent)
        
        self.emit(PYSIGNAL('beingReleased(e)'), (e,))

    
    def mouseDoubleClickEvent(self, e):
        mappedEvent = self.mapMouseEvent(e)

        for widget in self.revChildList:
            if widget.containsPoint(mappedEvent.pos()):
                return widget.mouseDoubleClickEvent(mappedEvent)
        
        self.emit(PYSIGNAL('beingDblClicked(e)'), (e,))


    def mouseMoveEvent(self, e):
        if self.isDraged():
            return self.parent().drag(e)
        
        mappedEvent = self.mapMouseEvent(e)

        for widget in self.revChildList:
            if widget.containsPoint(mappedEvent.pos()):
                return widget.mouseMoveEvent(mappedEvent)
        
        self.emit(PYSIGNAL('beingHovered(e)'), (e,))


    def keyPressEvent(self, e):
        pass

    def keyReleaseEvent(self, e):
        pass

    def wheelEvent(self, e):
        mappedEvent = self.mapWheelEvent(e)

        for widget in self.revChildList:
            if widget.containsPoint(mappedEvent.pos()):
                return widget.wheelEvent(mappedEvent)
        
        self.emit(PYSIGNAL('beingWheeled(e)'), (e,))

