/*
 * OPALE is a scientific library under LGPL. Its main goal is to
 * develop mathematical tools for any scientist.
 *
 * Copyright (C) 2002 Opale Group
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * You can visit the web site http://opale.tuxfamily.org to obtain more
 * informations about this program and/or to contact the authors by mail
 * developers@opale.tuxfamily.org.
 */




package opale.m2d;
import opale.tools.*;
import java.io.*;

/**
* Cette classe modlise un repre (O;I;J) quelconque dfinit soit dans un autre repre soit dans le repre absolu (orthonorm canonique) de l'espace affine R2.
* En outre on trouve dans cette classe des mthodes de calculs pour les normes et les distances dans le repre.
* @since Opale-2d 0.15, Elle tend la classe OpaleObject 
* @since Opale-2d 0.1
* @author O.C.
*/
public final class OIJ extends OpaleObject  implements  Cloneable
{
/**
* Repre orthonorm canonique de l espace affine R2
*/
static final public OIJ OIJ_ABSOLUTE = new OIJ();
static 
	{
	OIJ_ABSOLUTE.setName("OIJ_ABSOLUTE");
	}

private final static double precision = 1.e-15;

private String name;	// le nom du repere.
private OIJ rep;
private double oriX,oriY;	//les coordonnes du centre.
private double[] I, J;	//les vecteurs de base.
private double normSqI, normSqJ; // normes dans la base orthonorm canonique
private double IJ; // produit scalaire de I avec J
private double theta; // l'cart angulaire des deux vecteurs dans le repre canonique
//private Matrix2D PSM; // matrice du produit scalaire dans ce repre
private Matrix2D P; // matrice de passage du repere canonique vers ce repre



/**
* Constructeur par dfaut. Construit un repre orthonorm canonique.
*/ 
public OIJ()
	{
	name = new String("Sans nom");
	rep = OIJ.OIJ_ABSOLUTE;
	oriX = oriY = 0.;
	I = new double[2];
	J = new double[2];
	I[0]=1;
	I[1]=0;
	J[0]=0;
	J[1]=1;
	IJ = 0.;
	theta = Math.PI/2;
	normSqI = normSqJ = 1.;
	P = new Matrix2D();
	}
	
/**
* Constructeur qui prend en argument les caractristiques du repre  construire dans le repre absolu..
* @param double oriX, oriY les coordonnes du centre.
* @param double Ix, Iy les coordonnes du premier vecteur de base.
* @param double Jx, Jy les coordonnes du deuxime vecteur de base.
*/
public OIJ(double oriX, double oriY, double Ix, double Iy, double Jx, double Jy) throws OIJException
	{
	name = new String("Sans nom");
	this.rep = OIJ.OIJ_ABSOLUTE;
	this.oriX = oriX;
	this.oriY = oriY;
	if ( Math.abs(Ix*Jy - Iy*Jx) > precision )
		{
		I = new double[2];
		J = new double[2];
		I[0]=Ix;
		I[1]=Iy;
		J[0]=Jx;
		J[1]=Jy;
		normSqI = Ix*Ix+Iy*Iy;
		normSqJ = Jx*Jx+Jy*Jy;
		IJ = Ix*Jx+Iy*Jy;
		theta = Math.acos(IJ/Math.sqrt(normSqI*normSqJ));
		P = new Matrix2D();
		matPassageAbs(this,P);
		}
	else throw new OIJException("Erreur  la construction d'un repre !");
	}

/**
* Constructeur qui prend en argument les caractristiques du repre  construire.
* @param OIJ rep, le repre de base.
* @param double oriX, oriY les coordonnes du centre.
* @param double Ix, Iy les coordonnes du premier vecteur de base.
* @param double Jx, Jy les coordonnes du deuxime vecteur de base.
*/
public OIJ(OIJ rep,double oriX, double oriY, double Ix, double Iy, double Jx, double Jy) throws OIJException
	{
	name = new String("Sans nom");
	this.rep = rep;
	this.oriX = oriX;
	this.oriY = oriY;
	if ( Math.abs(Ix*Jy - Iy*Jx) > precision )
		{
		I = new double[2];
		J = new double[2];
		I[0]=Ix;
		I[1]=Iy;
		J[0]=Jx;
		J[1]=Jy;
		P = new Matrix2D();
		matPassageAbs(this,P);
		normSqI = rep.normSq(Ix,Iy);
		normSqJ = rep.normSq(Jx,Jy);
		IJ = Ix*Jx*rep.normSqI+Iy*Jy*rep.normSqJ+(Ix*Jy+Iy*Jx)*rep.IJ;
		theta = Math.acos(IJ/Math.sqrt(normSqI*normSqJ));
		}
	else throw new OIJException("Erreur  la construction d'un repre !");
	}

/**
* Constructeur de copie.
* @param OIJ rep, le repere  copier;
*/
public OIJ(OIJ rep)
	{
	this.name = rep.name;
	this.rep = rep.rep;
	this.oriX = rep.oriX;
	this.oriY = rep.oriY;
	I = new double[2];
	J = new double[2];
	I[0]=rep.I[0];
	I[1]=rep.I[1];
	J[0]=rep.J[0];
	J[1]=rep.J[1];
	P = new Matrix2D();
	matPassageAbs(this,P);
	normSqI = rep.normSqI;
	normSqJ = rep.normSqJ;
	IJ = rep.IJ;
	theta = rep.theta;
	}
	
/**
* Renvoi le nom donn au repre.
* @return String, le nom.
*/
public final String getName()
	{
	return name;
	}

/**
* Change le nom du repre.
* @param String name, le nom  donner.
*/
public final void setName(String name)
	{
	this.name = new String(name);
	}

/**
* Cette mthode teste si le repere est gal  un objet pass en argument.
* @param Object obj un objet  comparer avec le repere.
* @return <code>true</code> si l'objet  comparer est une instance de OIJ et est gal au repere courant; <code>false</code> sinon.
*/
public boolean equals(Object obj)
{
if (obj instanceof OIJ)
	{
	OIJ rep = (OIJ) obj;
	Matrix2D mv01 = new Matrix2D();
	Matrix2D mv02 = new Matrix2D();

	matPassageAbs(rep,mv01);
	matPassageAbs(this,mv02);

	return mv01.equals(mv02);
	}
return false;
}

/**
* Renvoie le carre de la norme du vecteur I.
* @return double le carre de la norme.
*/
public final double normSqI()
	{
	return normSqI;
	}

/**
* Renvoie le carre de la norme du vecteur J.
* @return double le carre de la norme.
*/
public final double normSqJ()
	{
	return normSqJ;
	}

/**
* Renvoie la valeur de l'ecart angulaire entre les vecteurs I et J.
* @return double l'angle
*/
public final double getTheta()
	{
	return theta;
	}

/**
* Renvoie le carre de la norme d'un vecteur dans le repre.
* @param double x, double y les coordonnes du vecteur.
* @return double le carre de la norme.
*/
public final double normSq(double x, double y)
	{
	return x*x*normSqI + 2*x*y*IJ + y*y*normSqJ;
	}

/**
* Renvoie la norme d'un vecteur dans le repre.
* @param double x, double y les coordonnes du vecteur.
* @return double la norme.
*/
public final double norm(double x, double y)
	{
	return Math.sqrt(normSq(x,y));
	}


/**
* Renvoie le carre de la distance entre deux points dans le repre.
* @param double x1, double y1 les coordonnes du premier point.
* @param double x2, double y2 les coordonnes du deuxime point.
* @return double le carre de la distance.
*/
public final double distSq(double x1, double y1, double x2, double y2)
	{
	return normSq(x2-x1,y2-y1);
	}


/**
* Renvoie la distance entre deux points dans le repre.
* @param double x1, double y1 les coordonnes du premier point.
* @param double x2, double y2 les coordonnes du deuxime point.
* @return double la distance.
*/
public final double dist(double x1, double y1, double x2, double y2)
	{
	return norm(x2-x1,y2-y1);
	}


/**
* Calcule la matrice de passage et le vecteur de passage du repere de dfinition au repere courant.
* @param double[][] mat, la matrice calcule.
* @param double[] vect, le vecteur calcul.
*/
public void matPassage(Matrix2D mv)
	{

	mv.set(0,0,I[0]);
	mv.set(1,0,I[1]);
	mv.set(0,1,J[0]);
	mv.set(1,1,J[1]);

	mv.setV(0,oriX);
	mv.setV(1,oriY);
	}


/**
* Calcule la matrice de passage et le vecteur de passage du repere spcifi en argument au repere courant.
* @param OIJ rep, le repere de dpart.
* @param double[][] mat, la matrice calcule.
* @param double[] vect, le vecteur calcul.
*/
public void matPassage(OIJ r,Matrix2D mv)
	{
	Matrix2D mv01 =new Matrix2D();
	Matrix2D mv02 =new Matrix2D();

	double[][] tmp = new double[2][2];

	matPassageAbs(r,mv01);
	matPassageAbs(this,mv02);
	double det=mv01.get(1,1)*mv01.get(0,0) - mv01.get(0,1)*mv01.get(1,0);
	tmp[0][0] = mv01.get(1,1)/det;
	tmp[0][1] = -mv01.get(0,1)/det;
	tmp[1][0] = -mv01.get(1,0)/det;
	tmp[1][1] = mv01.get(0,0)/det;

	mv.setV(0,tmp[0][0]*(mv02.getV(0)-mv01.getV(0))+tmp[0][1]*(mv02.getV(1)-mv01.getV(1)));
	mv.setV(1,tmp[1][0]*(mv02.getV(0)-mv01.getV(0))+tmp[1][1]*(mv02.getV(1)-mv01.getV(1)));

	mv.set(0,0,tmp[0][0]*mv02.get(0,0)+tmp[0][1]*mv02.get(1,0));
	mv.set(0,1,tmp[0][0]*mv02.get(0,1)+tmp[0][1]*mv02.get(1,1));
	mv.set(1,0,tmp[1][0]*mv02.get(0,0)+tmp[1][1]*mv02.get(1,0));
	mv.set(1,1,tmp[1][0]*mv02.get(0,1)+tmp[1][1]*mv02.get(1,1));

	}

//public static void matPassage(OIJ rep1, OIJ rep2,  double[][] mat, double[] vect);

private static void matPassageAbs(OIJ r, Matrix2D mv)
	{
	if (r != OIJ.OIJ_ABSOLUTE)
		{
		mv.set(0,0,r.I[0]);
		mv.set(1,0,r.I[1]);
		mv.set(0,1,r.J[0]);
		mv.set(1,1,r.J[1]);

		mv.setV(0,r.oriX);
		mv.setV(1,r.oriY);

		if (r.rep != OIJ.OIJ_ABSOLUTE)
			{
			Matrix2D mv2 =new Matrix2D();
			double[][] tmp= new double[2][2];	
			tmp[0][0] = mv.get(0,0);
			tmp[0][1] = mv.get(0,1);
			tmp[1][0] = mv.get(1,0);
			tmp[1][1] = mv.get(1,1);

			matPassageAbs(r.rep,mv2);
			
			mv.set(0,0,mv2.get(0,0)*tmp[0][0] + mv2.get(0,1)*tmp[1][0]);
			mv.set(0,1,mv2.get(0,0)*tmp[0][1] + mv2.get(0,1)*tmp[1][1]);
			mv.set(1,0,mv2.get(1,0)*tmp[0][0] + mv2.get(1,1)*tmp[1][0]);
			mv.set(1,1,mv2.get(1,0)*tmp[0][1] + mv2.get(1,1)*tmp[1][1]);

			tmp[0][0] = mv.getV(0);	

			mv.setV(0,mv2.get(0,0)*mv.getV(0) + mv2.get(0,1)*mv.getV(1) + mv2.getV(0));
			mv.setV(1,mv2.get(1,0)*tmp[0][0] + mv2.get(1,1)*mv.getV(1) + mv2.getV(1));
			}
		}
	else
		{
		mv.set(0,0,1);
		mv.set(1,0,0);
		mv.set(0,1,0);
		mv.set(1,1,1);	

		mv.setV(0,0);
		mv.setV(1,0);
		}
	}
public String toString()
	{
	return name;//+"\n"+I[0]+"\t"+J[0]+"\t"+oriX+"\n"+I[1]+"\t"+J[1]+"\t"+oriY+"\n";
	}
public String toLongString()
	{
	StringBuffer s = new StringBuffer(name+"\n");
	s.append(I[0]);
	s.append("\t");
	s.append(J[0]);
	s.append("\t");
	s.append(oriX+"\n");
	s.append(I[1]+"\t");
	s.append(J[1]+"\t");
	s.append(oriY+"\n");
	s.append("normSqI : ");
	s.append(normSqI+"\n");
	s.append("normSqJ : ");
	s.append(normSqJ+"\n");
	s.append("I.J : ");
	s.append(IJ+"\n");
	s.append("Theta : ");
	s.append(theta+"\n");
	return s.toString();
	}
	
/**
* Fixe les coordonnes de l'origine.
* @param double x, labscisse de l'origine.
* @param double y, l'ordonne de l'origine.
*/
public  final void setOrigin(double x, double y)
	{
	oriX = x;
	oriY = y;
	}
	
	
/**
* Renvoie l'abscisse de l'origine.
* @return double ,l'abscisse.
*/	
public final  double getOriginX()
	{
	return oriX;
	}

/**
* Renvoie l'ordonne de l'origine.
* @return double ,l'ordonne.
*/
public  final double getOriginY()
	{
	return oriY;
	}
	
/**
* Fixe les coordonnes du premier vecteur de base.
* @param double x, l'abscisse.
* @param double y, l'ordonne.
*/
public  final void setI(double x, double y) throws OIJException
	{
	if ( Math.abs(x*J[1] - y*J[0]) > precision )
		{
		I[0] = x;
		I[1] = y;
		normSqI = rep.normSq(x,y);
		IJ = x*J[0]*rep.normSqI+y*J[1]*rep.normSqJ+(x*J[1]+y*J[0])*rep.IJ;
		theta = Math.acos(IJ/Math.sqrt(normSqI*normSqJ));
		matPassageAbs(this,P);
		}
	else throw new OIJException("erreur");
	}
	
/**
* Fixe les coordonnes du premier vecteur de base.
* @param double x, l'abscisse.
* @param double y, l'ordonne.
*/
public  final void setJ(double x, double y) throws OIJException
	{
	if ( Math.abs(x*I[1] - y*I[0]) > precision )
		{
		J[0] = x;
		J[1] = y;
		normSqJ = rep.normSq(x,y);
		IJ = x*I[0]*rep.normSqI+y*I[1]*rep.normSqJ+(x*I[1]+y*I[0])*rep.IJ;
		theta = Math.acos(IJ/Math.sqrt(normSqI*normSqJ));
		matPassageAbs(this,P);
		}
	else throw new OIJException("erreur");
	}

/**
* Renvoie l'abscisse du premier  vecteur de base.
* @return double, l'abscisse.
*/
public final  double getIx()
	{
	return I[0];
	}

/**
* Renvoie l'ordonne du premier  vecteur de base.
* @return double, l'ordonne.
*/
public final  double getIy()
	{
	return I[1];
	}
	
/**
* Renvoie l'abscisse du deuxime vecteur de base.
* @return double, l'abscisse.
*/
public final  double getJx()
	{
	return J[0];
	}
		
		
/**
* Renvoie l'ordonne du deuxime vecteur de base.
* @return double, l'ordonne.
*/
public  final double getJy()
	{
	return J[1];
	}
	
/**
* Fixe le repre de base.
* @param mOIJ, le repre de base.
*/	
public final void setOIJ(OIJ rep)
	{
	this.rep = rep;
	normSqI = rep.normSq(I[0],I[1]);
	normSqJ = rep.normSq(J[0],J[1]);
	IJ = I[0]*J[0]*rep.normSqI+I[1]*J[1]*rep.normSqJ+(I[0]*J[1]+I[1]*J[0])*rep.IJ;
	theta = Math.acos(IJ/Math.sqrt(normSqI*normSqJ));
	matPassageAbs(this,P);
	}
	

/**
* Renvoie le repre de base.
* @return OIJ, le repre de base.
*/
public OIJ getOIJ()
	{
	return rep;
	}

/**
* Ecrit ses informations dans un flux.
* @param WFile f, le fichier  ecrire.
*/
public void write(PrintWriter f,OpaleSet p) throws InvalidFormatException 
	{
	f.println("{");
	if ((this == OIJ_ABSOLUTE) || (rep == OIJ_ABSOLUTE)) f.println("OIJ OIJ_ABSOLUTE");
	else {
		if (!p.contains(getOIJ())) throw new InvalidFormatException("Error writing the file "+f+" : the OIJ "+getOIJ()+" doesn't exist !!");
		f.println("OIJ "+getOIJ().getId());
		}
	f.println("O " +oriX+" " +oriY);
	f.println("I " +I[0]+" "+I[1]);
	f.println("J " +J[0]+" "+J[1]);
	f.println("}");
	}



/**
* This method read the data in a file at format '.opa' . 
* @param StreamReader f, stream to read.
* @param OpaleSet session, the group of object that are read in the stream.
*/
public void read( StreamReader f,OpaleSet session)  throws java.io.IOException,InvalidFormatException
	{
	double i0=1,i1=0,j0=0,j1=1;
	int type;
	String sval = f.next();	//on lit la premiere '{'
	if ( !sval.equals("{") ) throw new InvalidFormatException("Stream of data non valid !!");
	while ( !((sval = f.next()).equals("}")))
		{
		if (sval.equals("OIJ"))
			{
			sval = f.next();
			if (sval.equals("OIJ_ABSOLUTE"))
				rep = OIJ_ABSOLUTE;
			else {
			if (Debug.On) Debug.print("Dans OIJ : "+session.contains(sval));
			if (Debug.On) Debug.print("Dans OIJ : "+session.getObject(sval));
			if ( (!session.contains(sval)) || (session.getObject(sval)) == null) throw new InvalidFormatException("Data file "+f+" is wrong  : the identifiant "+sval+" doesn't exist !!");
			if (session.getObject(sval) instanceof OIJ) rep = (OIJ) session.getObject(sval);
			else throw new InvalidFormatException("Data file "+f+" is non valid  : the identifiant "+sval+" is not a OIJ !!");
			}
			}
		else if (sval.equals("O"))
			{
			oriX = f.nextDouble();
			oriY = f.nextDouble();
			}
		else if (sval.equals("I"))
			{
			i0 = f.nextDouble();
	
			i1 = f.nextDouble();
			}
		else if (sval.equals("J"))
			{
			j0 = f.nextDouble();
			j1 = f.nextDouble();
			}
		}
	
	try
		{
		setI(i0,i1);
		setJ(j0,j1);
		}
	catch(OIJException e)
		{
		throw new InvalidFormatException("Impossible to construct this object OIJ.");
		}
	}

/**
* Cre un nouvel objet de mme classe et de mme contenu.
* @return Object un clone de l'objet.
* @exception  OutOfMemoryError s'il n'y a pas assez de mmoire.
* @see        java.lang.Cloneable
* @since Opale-2d 0.11
*/
public Object clone()
	{
	//try
		//{
		OIJ p = (OIJ) super.clone();
		p.name = name;
		p.oriX = oriX;
		p.oriY = oriY;
		p.rep = rep;
		p.I[0] = I[0];
		p.I[1] = I[1];
		p.J[0] = J[0];
		p.J[1] = J[1];
		p.P = new Matrix2D();
		matPassageAbs(p,p.P);
		p.normSqI = normSqI;
		p.normSqJ = normSqJ;
		p.IJ = IJ;
		p.theta = theta;
	
		return p;
	/*	}
	catch (CloneNotSupportedException e) // ne devrait jamais arriver
		{
		throw new InternalError();
		}*/
   	}
	
/**
* Determine if the OIJ is direct.
* @return boolean true if direct.
* @date 03/2001
*/
public boolean isDirect()
	{
	return ( I[0]*J[1]-I[1]*J[0] >0);
	}
	
	
public static void main(String[] arg) throws OIJException
	{
	OIJ rep = new OIJ(0,0,-3,1,-2,1);
	Debug.print(""+rep.isDirect());
	}

}




