/*
 * OPALE is a scientific library under LGPL. Its main goal is to
 * develop mathematical tools for any scientist.
 *
 * Copyright (C) 2002 Opale Group
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * You can visit the web site http://opale.tuxfamily.org to obtain more
 * informations about this program and/or to contact the authors by mail
 * developers@opale.tuxfamily.org.
 */




package opale.m2d;
import opale.tools.*;
import java.io.*;

/**
* This class defines a simple 2d-point with 2 coordinates. This class inherits of the superclass Object2D and implements the interface Transformable that implies a point is an transformable object.
* @author O.C.
* @since Opale-2d 0.1
*/
public class Point2D extends Object2D implements Transformable, Cloneable
{
protected double x,y;

/**
* Default constructor ; Point is intialized with the origin
*/
public Point2D()
	{
	super();
	if (Debug.On) Debug.print("Dans Point2D()");
	}
	

/**
* Initializes the point at the origin of a specified OIJ
* @param OIJ rep.
*/
public Point2D(OIJ rep)
        {
        super(rep);
        }

/**
* Constructs a point with specified coordinates and a specified OIJ.
* @param OIJ rep
* @param double x, y the coordinates.
*/
public Point2D(OIJ rep,double x, double y)
        {
        super(rep);
        this.x = x;
        this.y = y;
        }

/**
* Copy constructor
* @param Point2D, the point to be copied.
*/
public Point2D(Point2D p)
        {
        super(p.getOIJ());
        this.x = p.x;
        this.y = p.y;
        }

/**
* Gets the x-coordinate of the point.
* @return double, the x-coordinate.
*/
public final double getX()
	{
	return x;
	}
/**
* Gets the y-coordinate of the point.
* @return double, the y-coordinate.
*/
public final double getY()
	{
	return y;
	}
	
/**
* Sets the coordinates of the point.
* @param double x, double y, the new coordinates
*/
public final void setLocation(double x, double y)
	{
	this.x = x;
	this.y = y;
	}

/**
* Sets the coordinates of the point with a specififed Point2D.
* @param  Point2D, an existing Point2D
*/
public final void setLocation(Point2D p)
	{
	Point2D q =(Point2D) p.clone();
	q.changeOIJ(getOIJ());
	
	this.x = q.x;
	this.y = q.y;
	}
	

// les mthodes de l'interface Transformable.

public void translate(double dx,double dy)
	{
	this.x+=dx;
	this.y+=dy;
	}
	

public void rotate(double theta)
	{
	final double costh=Math.cos(theta);
	final double sinth=Math.sin(theta);
	final OIJ rep = getOIJ();
	final double invsin = 1./Math.sin(rep.getTheta());
	final double tmp = Math.cos(rep.getTheta())*invsin*sinth;
	final double nI = Math.sqrt(rep.normSqI());
	final double nJ = Math.sqrt(rep.normSqJ());
	final double a00 =  costh - tmp;
	final double a01 =  -sinth*invsin/nI;
	final double a10 =   sinth*invsin/nJ;
	final double a11 =  costh + tmp;
	


	final double x=this.x;

	this.x=x*a00+y*a01 ;
	this.y=x*a10+this.y*a11 ;
	}

public void rotate(double oriX, double oriY,double theta)
	{
	this.translate(-oriX,-oriY);
	this.rotate(theta);
	this.translate(oriX,oriY);
	}



public void transform(AffineTransform2D t)
	{
	t.transform(this);
	}

   /*Cette mthode effectue une homothtie de centre A et de rapport k
   *@param A, le centre de l'homothetie
   *@param k, le rapport
   */
   public void homothetie(Point2D A, double k){
          //a verifier
          this.x=k*(this.x-A.x)+A.x ;
          this.y=k*(this.y-A.x)+A.y ;
   }

/**
* Changes the OIJ of this point
* @param OIJ rep, the new repear
*/
public void changeOIJ(OIJ rep)
	{
	Matrix2D mv = new Matrix2D();
	getOIJ().matPassage(rep,mv);
	
	double[] p = mv.compute(x,y);
	x = p[0];
	y = p[1];
	setOIJ(rep);
	}
	
	
/**
* Clones this object.
* @return Object the result clone
* @exception  OutOfMemoryError if no enough memory
* @see        java.lang.Cloneable
*/
public Object clone()
	{
	Point2D p = (Point2D) super.clone();
	p.x = x;
	p.y = y;
	return p;
   	}

/**
* Tests if the specififed object equals this point.
* @param Object obj, an existing object.
* @return <code>true</code> if the specififed object is an instance of Point2D and it equals this point ; false otherwise
* @since Opale 0.1
*/
public boolean equals(Object obj)
	{
	if (obj instanceof Point2D)
		{
        	Point2D p = (Point2D) obj;
		return (x == p.x) && (y == p.y) && (getOIJ().equals(p.getOIJ()));
        	}
    	return false;
    }


public String toString()
	{
	String s="("+this.x+"; "+this.y+")";
	return s;
	}
	

public void writeMore(PrintWriter f, OpaleSet p) throws InvalidFormatException
	{
	f.println("X " +x);
	f.println("Y " +y);
	}	
	
public int readKeyWord(String word, StreamReader f, OpaleSet p) throws java.io.IOException, InvalidFormatException
	{
	int type;
	if ( word.equals("X"))
		{
		x = f.nextDouble();
		return 0;
		}
	else if ( word.equals("Y"))
		{
		y = f.nextDouble();
		return 0;
		}
	return -1;
		
	}
	
/**
* Determine if (OA,OB) is direct.
* @param Point2D O,A,B, three points.
* @return boolean true if direct.
* @date 03/2001
*/
/*public boolean isDirect(Point2D O, Point2D A, Point2D B)
	{
	if ( (O.getOIJ().equals(A.getOIJ())) && (O.getOIJ().equals(B.getOIJ())))	// oprations couteuse !!
		return ( (A.x-O.x)*(B.x-O.x)+(A.y-O.y)*(B.y-O.y) >0);
	else
		System.exit(-1); 	// non cod encore !!

	return true;	// for the compiler
	}
	
public static void main(String[] arg)
	{
	}*/
		
}
