/*
 * OPALE is a scientific library under LGPL. Its main goal is to
 * develop mathematical tools for any scientist.
 *
 * Copyright (C) 2002 Opale Group
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * You can visit the web site http://opale.tuxfamily.org to obtain more
 * informations about this program and/or to contact the authors by mail
 * developers@opale.tuxfamily.org.
 */




package opale.matrix.solver;

import opale.tools.Debug;
import opale.mathtools.DVect;
import opale.matrix.*;

/**
* This class implements LU decomposition according to the book "Analyse matricielle applique  l'art de l'ingnieur", Lascaux & Thodor, Ed. Masson.
* @author O.C.
* @date 04/2001
*/


public class SolverLU implements Solver, Decomposition
{

private double[][] LU;
private int[] pivot;
private int n,signepermut;

public void decomp(Matrix A)
	{
	if (A.getRowDim() != A.getColumnDim() ) throw new IllegalArgumentException("LU decomposition implemented only for square matrix");
	n = A.getRowDim();
	int  jmax,tmp;
	double Amax,pivinv;
	
	pivot = new int[n];
	signepermut=1;
	
	if (Decomposition.REPLACE) LU=A.getInternalArray();
	else LU = A.getArrayCopy();
	
	for (int i=0;i<n;i++)
		pivot[i] = i;
		
	for (int k=0;k<n-1;k++)
		{
		Amax = Math.abs(LU[pivot[k]][k]);
		jmax = k;
		
		for (int i=k+1;i<n;i++)
			{
			if (Math.abs(LU[pivot[i]][k])>Amax)
				{
				Amax = Math.abs(LU[pivot[i]][k]);
				jmax = i;
				}
			}
			
		if (jmax != k)
			{
			tmp = pivot[k];
			pivot[k] = pivot[jmax];
			pivot[jmax]=tmp;
			signepermut=-signepermut;
			}
		pivinv = 1./LU[pivot[k]][k];
			
		for (int i=k+1;i<n;i++)
			{
			double c = LU[pivot[i]][k]*pivinv;
			LU[pivot[i]][k]=c;
			
			for(int j=k+1;j<n;j++)
				LU[pivot[i]][j] = LU[pivot[i]][j] - c*LU[pivot[k]][j];
			}
		}
	}

/** 
* Compute the determinant : product of U[i][i]
* @return     double, the determinant
*/
public double determinant() 
	{
	double det = (int) signepermut;
	for (int i = 0; i < n; i++) 
		det *= LU[pivot[i]][i];

	return det;
	}
			

/**
* Solve LUx=b.
* @param Matrix A, not used.
*/
public void solve(Matrix A,DVect b, DVect x)
	{
	if ((n != b.size()) || (n != x.size())) throw new IllegalArgumentException("matrix and vector dimension are not equals.");
	
	double[] B = b.toArray();
	double[] X = x.toArray();
	
	
	for (int i=0;i<n;i++)
		{
		double c=0.;
		for (int j=0;j<i;j++)
			c+=LU[pivot[i]][j]*X[j];
		X[i] = B[pivot[i]]-c;
		}
	
	for (int i=n-1;i>=0;i--)
		{
		double c=0.;
		for (int j=i+1;j<n;j++)
			c+=LU[pivot[i]][j]*X[j];
		X[i] = (X[i]-c)/LU[pivot[i]][i];
		}
		
	}
	

public static void main(String[] arg)
	{
	double[][] tab = { {4,1}, {1,4} };
	Matrix A=new Matrix(tab);
	
	DVect b=new DVect(2);
	DVect x=new DVect(2);

	System.out.println(A);
	
	b.set(0,1);
	b.set(1,0);
	
	
	SolverLU lu = new SolverLU();
	lu.decomp(A);
	System.out.println("det = " +lu.determinant());
	lu.solve(null,b,x);
		

	
	System.out.println(x);
	System.out.println(A);
	}
	
			
}
	
	

