/*
SDX: Documentary System in XML.
Copyright (C) 2000, 2001, 2002  Ministere de la culture et de la communication (France), AJLSM

Ministere de la culture et de la communication,
Mission de la recherche et de la technologie
3 rue de Valois, 75042 Paris Cedex 01 (France)
mrt@culture.fr, michel.bottin@culture.fr

AJLSM, 17, rue Vital Carles, 33000 Bordeaux (France)
sevigny@ajlsm.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.
59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
or connect to:
http://www.fsf.org/copyleft/gpl.html
*/
package fr.gouv.culture.sdx.utils.lucene;

import fr.gouv.culture.sdx.exception.SDXException;
import fr.gouv.culture.sdx.exception.SDXExceptionCode;
import fr.gouv.culture.sdx.utils.SdxObjectImpl;
import fr.gouv.culture.sdx.utils.Utilities;
import org.apache.lucene.analysis.standard.StandardAnalyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.*;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;

import java.io.File;
import java.io.IOException;

/**
 * Created by IntelliJ IDEA.
 * User: rpandey
 * Date: Dec 10, 2002
 * Time: 1:56:41 PM
 * To change this template use Options | File Templates.
 */
abstract public class LuceneDataStore extends SdxObjectImpl {

    /**TODO implement a fields list for keeping track of analyzers on field level, and something similar to meta analyzer,*/
    /** The field name used to store the entity's id. */
    public final static String ID_FIELD = "sdxdocid";
    /** The field used to search for all documents. */
    public final static String ALL_FIELD = "sdxall";
    /**Value for the all field*/
    public final static String ALL_VALUE = "1";
    /**A field name used to store a xml lang value if desired*/
    public final static String XML_LANG_FIELD = "xmllang";
    /** The Lucene file to check to see if a database already exists. */
    protected final static String LUCENE_CHECK_FILE = "segments";
    /**The location of the FSDirectory*/
    protected File fsdFile = null;
    /** The Lucene filesystem directory where the index is stored. */
    protected FSDirectory fsd = null;
    /** The Lucene searcher used for locating entities within the datastore. */
    protected Searcher searcher = null;
    /**The analyzer for the index behind the datastore*/
    protected org.apache.lucene.analysis.Analyzer analyzer = new StandardAnalyzer();


    public LuceneDataStore() {
    }

    protected LuceneDataStore(File dir) {
        this.fsdFile = dir;
    }

    /**Return's the absolute path for the
     *directory in which the database resides
     */
    public String getIndexPath() {
        if (this.fsdFile == null)
            return null;
        else
            return fsdFile.getAbsolutePath();
    }

    /**Initializes the LuceneDataStore*/
    protected void init(boolean create) throws SDXException {
        //acquiring a lucene FSDirectory for the class
        getFSDirectory(create);
        //ensuring we have an index to work with
        verifyIndex();
        //acquiring a searcher
        recycleSearcher();
    }

    /**Set's the class field 'fsd' a FSDirectory instance for the fsdFile object*/
    protected void getFSDirectory(boolean create) throws SDXException {
        if (this.fsdFile != null && this.fsd == null) {
            try {
                this.fsd = FSDirectory.getDirectory(this.fsdFile, create);
            } catch (IOException e) {
                //unable to get a FSDirectory object
                String[] args = new String[2];
                args[0] = getIndexPath();
                args[1] = e.getMessage();
                throw new SDXException(logger, SDXExceptionCode.ERROR_LUCENE_ACQUIRE_FSDIRECTORY, args, e);
            }
        }
    }

    /**Ensures we have an index to work with*/
    protected void verifyIndex() throws SDXException {
        try {
            if (!fsd.fileExists(LUCENE_CHECK_FILE)) {
                IndexWriter w = new IndexWriter(fsd, analyzer, true);
                w.close();
            }
        } catch (IOException e) {
            String[] args = new String[2];
            args[0] = getIndexPath();
            args[1] = e.getMessage();
            throw new SDXException(logger, SDXExceptionCode.ERROR_LUCENE_WRITE, args, e);
        }
    }

    /** Recycles the searcher*/
    protected synchronized void recycleSearcher() throws SDXException {
        try {
            if (this.searcher != null)
            //freeing resources
            //TODO?:but what if an external call has already closed the searcher or the searcher is in use?-rbp
                this.searcher.close();

            //getting the new searcher
            this.searcher = new IndexSearcher(fsd);

        } catch (IOException e) {
            String[] args = new String[1];
            args[0] = getIndexPath();
            throw new SDXException(logger, SDXExceptionCode.ERROR_LUCENE_SEARCHER_GET, args, e);
        }
    }

    /**Gets an IndexWriter for the current FSDirectory*/
    protected IndexWriter getWriter() throws IOException {
        return getWriter(this.fsd);
    }

    /**Get's an IndexWriter based upon the analyzer class field
     * and the provided Lucene Directory
     *
     * @param directory The lucene directory for which the writer is desired
     * @return  IndexWriter
     * @throws IOException
     */
    protected synchronized IndexWriter getWriter(Directory directory) throws IOException {
        IndexWriter w = new IndexWriter(directory, this.analyzer, false);
        return w;
    }

    /**Returns an index reader for the current FSDirectory.*/
    protected IndexReader getReader() throws SDXException {
        try {
            return IndexReader.open(this.fsd);
        } catch (IOException e) {
            String[] args = new String[2];
            args[0] = getIndexPath();
            args[1] = e.getMessage();
            throw new SDXException(logger, SDXExceptionCode.ERROR_LUCENE_READ, args, e);
        }
    }

    /**Gets the search class variable*/
    protected Searchable getSearcher() {
        return this.searcher;
    }


    /**Optimizeds the index for the current FSDirectory.*/
    protected synchronized void optimize() throws SDXException {
        IndexWriter w = null;
        try {
            w = getWriter();
            //optimizing
            w.optimize();
        } catch (IOException e) {
            String[] args = new String[1];
            args[0] = getIndexPath();
            throw new SDXException(logger, SDXExceptionCode.ERROR_LUCENE_OPTIMIZE, args, e);

        } finally {
            if (w != null) {
                try {
                    w.close();
                } catch (IOException e) {
                    String[] args = new String[1];
                    args[0] = getIndexPath();
                    throw new SDXException(logger, SDXExceptionCode.ERROR_LUCENE_OPTIMIZE, args, e);
                }
            }
            //resetting the searcher
            recycleSearcher();
        }
    }

    /**Writes a document to the index
     *
     * @param lDoc  The lucene document to add
     */
    protected synchronized void write(Document lDoc) throws SDXException {
        if (lDoc != null) {
            //we have a single document to add
            IndexWriter w = null;
            try {
                //initializing an index writer for our main index directory
                w = getWriter();
                //we add the single document
                w.addDocument(lDoc);
                //TODO : implement a kind of locking for the searcher ? -pb
            } catch (IOException e) {
                String[] args = new String[2];
                args[0] = getIndexPath();
                args[1] = e.getMessage();
                throw new SDXException(logger, SDXExceptionCode.ERROR_LUCENE_WRITE, args, e);
            } finally {
                try {
                    if (w != null) w.close();
                } catch (IOException e) {
                    String[] args = new String[2];
                    args[0] = getIndexPath();
                    args[1] = e.getMessage();
                    throw new SDXException(logger, SDXExceptionCode.ERROR_LUCENE_WRITE, args, e);
                }
            }
        }

    }

    protected synchronized void writeLuceneData(LuceneData ld) throws SDXException {
        if (ld != null) {
            Document lDoc = ld.getLuceneDocument();
            if (lDoc != null)
                this.write(lDoc);
        }
    }

    /**Deletes a document from the index
     * based upon its id field, marked protected
     * as it is specfic implementation
     *
     * @param docId The id of the document to be deleted
     */
    protected synchronized void delete(String docId) throws SDXException {
        if (Utilities.checkString(docId)) {
            String[] docIds = {docId};
            delete(docIds);
        }
    }

    /**Deletes documents from the index
     * based upon their id field, marked protected
     * as it is specfic implementation
     *
     * @param docIds The ids of the document to be deleted
     */
    protected synchronized void delete(String[] docIds) throws SDXException {
        if (docIds != null) {
            Term[] terms = new Term[docIds.length];
            for (int i = 0; i < docIds.length; i++) {
                String docId = docIds[i];
                if (Utilities.checkString(docId)) {
                    //the below ligne tightly couples us to LuceneDatastore
                    Term term = new Term(ID_FIELD, docId);
                    terms[i] = term;
                }
            }
            delete(terms);
        }
    }

    /**Deletes a document from the index
     * based upon a term field
     *
     * @param term The term of the document to be deleted
     */
    public void delete(Term term) throws SDXException {
        Term[] terms = {term};
        delete(terms);
    }

    /**List of terms to delete using a single index reader
     *
     * @param terms
     * @throws SDXException
     */
    public synchronized void delete(Term[] terms) throws SDXException {
        if (terms != null) {
            IndexReader r = null;
            try {
                //the index reader for deletion
                for (int i = 0; i < terms.length; i++) {
                    Term term = terms[i];
                    if (term != null) {
                        if (r == null) r = getReader();
                        //deleting the document
                        r.delete(term);
                    }
                }
            } catch (IOException e) {
                String[] args = new String[2];
                args[0] = getIndexPath();
                args[1] = e.getMessage();
                throw new SDXException(logger, SDXExceptionCode.ERROR_LUCENE_DELETE, args, e);
            } finally {
                try {
                    if (r != null) r.close();
                } catch (IOException e) {
                    String[] args = new String[2];
                    args[0] = getIndexPath();
                    args[1] = e.getMessage();
                    throw new SDXException(logger, SDXExceptionCode.ERROR_LUCENE_DELETE, args, e);
                }
            }

        }
    }


    /**Free's the resources associated with this data store
     * USE WITH CARE!
     *
     * @throws IOException  Lucene IOExceptions
     */
    protected void freeResources() throws IOException {
        if (this.searcher != null)
            this.searcher.close();
        if (this.fsd != null)
            this.fsd.close();
        this.fsd = null;
    }

    /**Ensures that resources are freed*/
    protected void finalize() throws Throwable {
        this.freeResources();
        super.finalize();
    }

    /**
     * Searches the database.
     *
     * @param   q   A Lucene query.
     */
    public synchronized Hits search(Query q) throws SDXException {
        return this.search(q, null);
    }

    /**
     * Searches the database.
     *
     * @param   q   A Lucene query.
     */
    public synchronized Hits search(Query q, Filter f) throws SDXException {
        try {
            if (q == null) throw new SDXException(null, SDXExceptionCode.ERROR_LUCENE_QUERY_NULL, null, null);

            //ensuring we have a searcher
            if (this.searcher == null) {
                recycleSearcher();
                if (this.searcher == null) {
                    String[] args = new String[2];
                    args[0] = getIndexPath();
                    throw new SDXException(logger, SDXExceptionCode.ERROR_LUCENE_SEARCHER_GET, args, null);
                }
            }
            if (f != null)
                return this.searcher.search(q, f);

            else
                return this.searcher.search(q);

        } catch (IOException e) {
            String[] args = new String[2];
            args[0] = getIndexPath();
            args[1] = e.getMessage();
            throw new SDXException(logger, SDXExceptionCode.ERROR_LUCENE_SEARCH, args, e);
        }
    }

    /**
     * Returns the number of entities within this database.
     */
    public long size() {
        Hits h = null;
        try {
            h = search(new TermQuery(new Term(ALL_FIELD, ALL_VALUE)));

            if (h != null)
                return h.length();
            else
                return 0;
        } catch (SDXException e) {
            return 0;
        }
    }

}










