/*
SDX: Documentary System in XML.
Copyright (C) 2000, 2001, 2002  Ministere de la culture et de la communication (France), AJLSM

Ministere de la culture et de la communication,
Mission de la recherche et de la technologie
3 rue de Valois, 75042 Paris Cedex 01 (France)
mrt@culture.fr, michel.bottin@culture.fr

AJLSM, 17, rue Vital Carles, 33000 Bordeaux (France)
sevigny@ajlsm.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.
59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
or connect to:
http://www.fsf.org/copyleft/gpl.html
*/
/*
 * Created by IntelliJ IDEA.
 * User: rpandey
 * Date: 7 nov. 2002
 * Time: 12:03:13
 * To change template for new class use
 * Code Style | Class Templates options (Tools | IDE Options).
 */
package fr.gouv.culture.sdx.repository;

import fr.gouv.culture.sdx.document.Document;
import fr.gouv.culture.sdx.exception.SDXException;
import fr.gouv.culture.sdx.exception.SDXExceptionCode;
import fr.gouv.culture.sdx.utils.Utilities;
import oracle.sql.BLOB;
import org.apache.avalon.excalibur.io.IOUtil;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

public class ORACLERepository extends JDBCRepository {

    /** Returns an SQL query that could create the table.
     * <p>
     * This query should not have any parameter.
     * @return The query.
     */

    protected String getTableCreationQuery() {
        // TODO: at some time, implement the proper handling of the original document (column data_t)
        //changed dataType of FIELD_DATA_TRANSFORMED from "LONGBLOB NOT NULL" TO  only "LONGBLOB"
        return "CREATE TABLE " + getTableName() + " ( " + FIELD_ID + " VARCHAR(255) NOT NULL, " + FIELD_DATA + " BLOB, PRIMARY KEY (" + FIELD_ID + "))";
    }

    /** Returns an SQL query that could add a document.
     * <p>
     * This query should have three parameters, the first for the id, the second
     * for the doctype and the third for the data.
     * @return The query.
     */
    protected String getDocumentAddQuery() {
        //TODOImplement: at some time, implement the proper handling of the original document (column data_t).
        return "INSERT INTO " + getTableName() + " (" + FIELD_ID + "," + FIELD_DATA + ") VALUES (?, EMPTY_BLOB())";
    }

    /** Adds a document to the repository.
     *
     * @param doc   A document.
     * @param c     A connection to the repository.
     * @throws fr.gouv.culture.sdx.exception.SDXException
     */
    public synchronized void add(Document doc, RepositoryConnection c) throws SDXException {
        //ensuring we have valid objects
        Utilities.checkDocument(logger, doc);
        this.checkConnection(c);

        Connection conn = ((JDBCRepositoryConnection) c).getConnection();
        String queryString = this.getDocumentAddQuery();
        PreparedStatement ps = null;
        PreparedStatement oraStmt = null;
        ResultSet oraRs = null;
        InputStream is = null;
        OutputStream os = null;
        try {
            ps = conn.prepareStatement(queryString);
            ps.setString(PARAM_INDEX_FIELD_ID, doc.getId());
            //TODO?: remove the transformed data field as we dont use it anymore
            ps.executeUpdate();
            conn.commit();
            //this is will give us the original file
            oraStmt = conn.prepareStatement("SELECT " + FIELD_DATA + " FROM " + getTableName() + " WHERE " + FIELD_ID + " = ? FOR UPDATE");
            oraStmt.clearParameters();
            oraStmt.setString(PARAM_INDEX_FIELD_ID, doc.getId());
            conn.setAutoCommit(false);
            oraRs = oraStmt.executeQuery();
            if (oraRs.next()) {
                oracle.sql.BLOB oraBlob = ((oracle.jdbc.OracleResultSet) oraRs).getBLOB(FIELD_DATA);
                os = oraBlob.getBinaryOutputStream();
                int chunk = oraBlob.getChunkSize();
                byte[] buffer = new byte[chunk];
                int length = -1;
                is = doc.openStream();
                while ((length = is.read(buffer)) != -1)
                    os.write(buffer, 0, length);
                os.flush();
            }
            conn.commit();
        } catch (SQLException e) {
            String[] args = new String[3];
            args[0] = doc.getId();
            args[1] = this.getId();
            args[2] = e.getMessage();
            throw new SDXException(logger, SDXExceptionCode.ERROR_ADD_DOC, args, e);
        } catch (IOException e) {
            String[] args = new String[3];
            args[0] = doc.getId();
            args[1] = this.getId();
            args[2] = e.getMessage();
            throw new SDXException(logger, SDXExceptionCode.ERROR_ADD_DOC, args, e);
        } finally {
            if (ps != null) {
                try {
                    ps.close();
                } catch (SQLException e) {
                    String[] args = new String[2];
                    args[0] = this.getId();
                    args[1] = e.getMessage();
                    throw new SDXException(logger, SDXExceptionCode.ERROR_CLOSE_SQL_PREPARED_STATEMENT, args, e);
                }
            }

            if (os != null) {
                try {
                    os.close();
                } catch (IOException e) {
                    throw new SDXException(logger, SDXExceptionCode.ERROR_CLOSE_STREAM, null, e);
                }
            }

            if (is != null) {
                try {
                    is.close();
                } catch (IOException e) {
                    throw new SDXException(logger, SDXExceptionCode.ERROR_CLOSE_STREAM, null, e);
                }
            }
            if (oraRs != null) {
                try {
                    oraRs.close();
                } catch (SQLException e) {
                    String[] args = new String[2];
                    args[0] = this.getId();
                    args[1] = e.getMessage();
                    throw new SDXException(logger, SDXExceptionCode.ERROR_CLOSE_RESULT_SET, args, e);
                }
            }
            if (oraStmt != null) {
                try {
                    oraStmt.close();
                } catch (SQLException e) {
                    String[] args = new String[2];
                    args[0] = this.getId();
                    args[1] = e.getMessage();
                    throw new SDXException(logger, SDXExceptionCode.ERROR_CLOSE_SQL_PREPARED_STATEMENT, args, e);
                }
            }
        }
    }

    /** Writes the content of a document to an output stream.
     *
     * @param doc A document.
     * @param os An output stream.
     * @param c A connection to the repository.
     * @throws SDXException
     */
    public void get(Document doc, OutputStream os, RepositoryConnection c) throws SDXException {
        //ensuring we have valid objects
        Utilities.checkDocument(logger, doc);
        Utilities.checkOutputStream(logger, os);
        this.checkConnection(c);

        Connection conn = ((JDBCRepositoryConnection) c).getConnection();
        String queryString = getDocumentGetQuery();
        PreparedStatement ps = null;
        ResultSet rs = null;
        try {
            ps = conn.prepareStatement(queryString);
            ps.setString(PARAM_INDEX_FIELD_ID, doc.getId());
            rs = ps.executeQuery();
            if (rs.next()) {
                oracle.sql.BLOB oraBlob = ((oracle.jdbc.OracleResultSet) rs).getBLOB(FIELD_DATA);
                IOUtil.copy(oraBlob.getBinaryStream(), os);
            }
        } catch (SQLException e) {
            String[] args = new String[3];
            args[0] = doc.getId();
            args[1] = this.getId();
            args[2] = e.getMessage();
            throw new SDXException(logger, SDXExceptionCode.ERROR_GET_DOC, args, e);
        } catch (IOException e) {
            String[] args = new String[3];
            args[0] = doc.getId();
            args[1] = this.getId();
            args[2] = e.getMessage();
            throw new SDXException(logger, SDXExceptionCode.ERROR_GET_DOC, args, e);
        } finally {
            if (ps != null) {
                try {
                    ps.close();
                } catch (SQLException e) {
                    String[] args = new String[2];
                    args[0] = this.getId();
                    args[1] = e.getMessage();
                    throw new SDXException(logger, SDXExceptionCode.ERROR_CLOSE_SQL_PREPARED_STATEMENT, args, e);
                }
            }

            if (rs != null) {
                try {
                    rs.close();
                } catch (SQLException e) {
                    String[] args = new String[2];
                    args[0] = this.getId();
                    args[1] = e.getMessage();
                    throw new SDXException(logger, SDXExceptionCode.ERROR_CLOSE_RESULT_SET, args, e);
                }

            }
        }

    }

    /** Opens an input stream to read the content of a document.
     *
     * @return          The stream.
     * @param encoding  An encoding (may be null).
     * <p>If <code> null</code> or invalid we use a default.
     * <p>TODOImplement use of encoding currently not implemented , will do soon.
     * @param doc       A document to read.
     * @param c         A connection to the repository.
     * @throws SDXException
     */
    public InputStream openStream(Document doc, String encoding, RepositoryConnection c) throws SDXException {
        //ensuring we have valid objects
        Utilities.checkDocument(logger, doc);
        this.checkConnection(c);
        //verifying the encoding, if not valid we use a default, see checkEncoding() for details
        encoding = checkEncoding(encoding);
        Connection conn = ((JDBCRepositoryConnection) c).getConnection();
        String queryString = getDocumentGetQuery();
        PreparedStatement ps = null;
        ResultSet rs = null;
        try {
            ps = conn.prepareStatement(queryString);
            ps.setString(1, doc.getId());
            rs = ps.executeQuery();
            InputStream is;
            if (rs.next()) {
                oracle.sql.BLOB oraBlob = ((oracle.jdbc.OracleResultSet) rs).getBLOB(FIELD_DATA);
                is = oraBlob.getBinaryStream();
                return is;
            } else {
                String[] args = new String[2];
                args[0] = doc.getId();
                args[1] = this.getId();
                throw new SDXException(logger, SDXExceptionCode.ERROR_GET_DOC, args, null);
            }
        } catch (SQLException e) {
            String[] args = new String[3];
            args[0] = doc.getId();
            args[1] = this.getId();
            args[2] = e.getMessage();
            throw new SDXException(logger, SDXExceptionCode.ERROR_GET_DOC, args, e);
        } finally {
            if (ps != null) {
                try {
                    if (ps != null) ps.close();
                } catch (SQLException e) {
                    String[] args = new String[2];
                    args[0] = this.getId();
                    args[1] = e.getMessage();
                    throw new SDXException(logger, SDXExceptionCode.ERROR_CLOSE_SQL_PREPARED_STATEMENT, args, e);
                }
            }

            if (rs != null) {
                try {
                    if (rs != null) rs.close();
                } catch (SQLException e) {
                    String[] args = new String[2];
                    args[0] = this.getId();
                    args[1] = e.getMessage();
                    throw new SDXException(logger, SDXExceptionCode.ERROR_CLOSE_RESULT_SET, args, e);
                }
            }

        }

    }

    /**
     * Creates the table.
     */
    protected void createTable(Connection conn) throws SDXException {
        PreparedStatement ps = null;
        try {
            ps = conn.prepareStatement(getTableCreationQuery());
            ps.executeUpdate();
        } catch (SQLException e) {
            String[] args = new String[2];
            args[0] = this.getTableName();
            args[1] = e.getMessage();
            /*we don't throw a specific oracle exception, as an exception is thrown when a table exists, we must
            attempt to create this table each init() call, as oracle doesnt give us useful metadata about a database*/
            if (e.getErrorCode() != 955)
                throw new SDXException(logger, SDXExceptionCode.ERROR_CREATE_TABLE, args, e);
        } finally {
            if (ps != null) {
                try {
                    ps.close();
                } catch (SQLException e1) {
                    String[] args1 = new String[2];
                    args1[0] = this.getId();
                    args1[1] = e1.getMessage();
                    throw new SDXException(logger, SDXExceptionCode.ERROR_CLOSE_SQL_PREPARED_STATEMENT, args1, e1);
                }
            }
        }
    }

}
