/*
SDX: Documentary System in XML.
Copyright (C) 2000, 2001, 2002  Ministere de la culture et de la communication (France), AJLSM

Ministere de la culture et de la communication,
Mission de la recherche et de la technologie
3 rue de Valois, 75042 Paris Cedex 01 (France)
mrt@culture.fr, michel.bottin@culture.fr

AJLSM, 17, rue Vital Carles, 33000 Bordeaux (France)
sevigny@ajlsm.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.
59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
or connect to:
http://www.fsf.org/copyleft/gpl.html
*/
package fr.gouv.culture.sdx.search.lucene.query;

import fr.gouv.culture.sdx.document.IndexableDocument;
import fr.gouv.culture.sdx.documentbase.LuceneDocumentBase;
import fr.gouv.culture.sdx.documentbase.LuceneIndexParameters;
import fr.gouv.culture.sdx.exception.SDXException;
import fr.gouv.culture.sdx.exception.SDXExceptionCode;
import fr.gouv.culture.sdx.framework.Framework;
import fr.gouv.culture.sdx.framework.FrameworkImpl;
import fr.gouv.culture.sdx.search.lucene.Field;
import fr.gouv.culture.sdx.search.lucene.FieldList;
import fr.gouv.culture.sdx.search.lucene.analysis.MetaAnalyzer;
import fr.gouv.culture.sdx.search.lucene.queryparser.QueryParser;
import fr.gouv.culture.sdx.utils.SdxObject;
import fr.gouv.culture.sdx.utils.Utilities;
import fr.gouv.culture.sdx.utils.constants.Node;
import fr.gouv.culture.sdx.utils.logging.LoggingUtils;
import fr.gouv.culture.sdx.utils.lucene.LuceneDataStore;
import fr.gouv.culture.sdx.utils.save.SaveParameters;

import org.apache.avalon.excalibur.io.FileUtil;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.Hits;
import org.apache.lucene.search.Searchable;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;
import org.apache.lucene.store.RAMDirectory;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.AttributesImpl;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.rmi.Naming;
import java.rmi.Remote;
import java.rmi.RemoteException;
import java.util.Date;
import java.util.Locale;

/**
 * Information and services related to a LuceneIndex.
 *
 * <p>
 * From here we can get an IndexReader, a Searcher, and know a little more
 * about defined fields.
 */
public class LuceneIndex extends LuceneDataStore implements SdxObject, Index {

	/** The type of index */
	private boolean _isSub = false;

    /**The application in which this index is used.*/
    private String appId = "";

    /**The document base in which this index is used.*/
    private String dbId = "";

    /**The container for available search fields. */
    private FieldList fields = null;

    /**The temporary RAM directory for a batch of documents*/
    private RAMDirectory tempBatch = null;
    /** The current directory name*/
    public String CURRENT_INDEX_DIR = "current";

    /**The MetaAnalyzer*/
    private MetaAnalyzer metaAnalyzer = null;

    /**The specified query parser*/
    private Class queryParserClass = null;

    /**Default query parser class name*/
    private static final String DEFAULT_QUERYPARSER = "fr.gouv.culture.sdx.search.lucene.queryparser.DefaultQueryParser";

    /**The RMI registry name for the remote index*/
    private String remoteIndexName = "";

    /**The remote host for the remote index*/
    private String rmiHost = "";

    /**The remote port for the remote index*/
    private int rmiPort;

    /**The creation date of this index*/
    protected Date creationTimestamp = null;

    /**The last modification date of this index*/
    protected Date lastModificationTimestamp = null;

    /** int representation of indexing action for adding a document. */
    public final static int ACTION_ADD_DOCUMENT = 1;

    /**Defaults for IndexWriter parameters; these are based on lucene defaults*/
    public static final int DEFAULT_MAX_FIELD_LENGTH = 10000;
    /**Defaults for IndexWriter parameters; these are based on lucene defaults*/
    public static final int DEFAULT_MAX_MERGE_DOCS = Integer.MAX_VALUE;
    /**Defaults for IndexWriter parameters; these are based on lucene defaults*/
    public static final int DEFAULT_MERGE_FACTOR = 10;

    /**Lucene IndexWriter parameter*/
    private int maxFieldLength = DEFAULT_MAX_FIELD_LENGTH; //Defaulted
    /**Lucene IndexWriter parameter*/
    private int maxMergeDocs = DEFAULT_MAX_MERGE_DOCS; //Defaulted
    /**Lucene IndexWriter parameter*/
    private int mergeFactor = DEFAULT_MERGE_FACTOR; //Defaulted

    /**String representation of the element name "queryParser". */
    private static final String ELEMENT_NAME_QUERYPARSER = "queryParser";

    /**String representation of the attribute named "class". */
    private static final String ATTRIBUTE_CLASS = "class";

    /**String representation of the attribute name "maxFieldLength". */
    private final String ATTRIBUTE_MAX_FIELD_LENGTH = "maxFieldLength";
		/**int indicating the number of actual documents labeled with the sdxall:1 field*/
    private int _docCount = -1;


    /**
     * Builds an index with fields definition and a path to the index files.
     *
     * @param   dir     A directory where the index is kept.
     * TODOJavadoc
     */
    public LuceneIndex(File dir, String host, Integer port, String appId, String dbId) throws SDXException {
        super(dir);

        this.appId = appId;
        this.dbId = dbId;


        if (Utilities.checkString(host))
            this.rmiHost = host;

        if (port != null)
            this.rmiPort = port.intValue();
        else
            this.rmiPort = FrameworkImpl.SDX_DEFAULT_RMI_PORT;

        if (Utilities.checkString(appId) && Utilities.checkString(dbId))
            this.remoteIndexName = Utilities.buildRmiName(this.rmiHost, this.rmiPort, appId, dbId);
    }

    public LuceneIndex(File dir, String host, Integer port, String appId, String dbId, boolean useCompoundFiles) throws SDXException {
        super(dir);

        this.appId = appId;
        this.dbId = dbId;
        this.useCompoundFiles = useCompoundFiles;

        if (Utilities.checkString(host))
            this.rmiHost = host;

        if (port != null)
            this.rmiPort = port.intValue();
        else
            this.rmiPort = FrameworkImpl.SDX_DEFAULT_RMI_PORT;

        if (Utilities.checkString(appId) && Utilities.checkString(dbId))
            this.remoteIndexName = Utilities.buildRmiName(this.rmiHost, this.rmiPort, appId, dbId);
    }


    /**
     * Sub index constructor
     */
    public LuceneIndex(LuceneIndex parentIndex, File dir)
    {
    	super(dir);
    	this._isSub = true;
    	this.appId = parentIndex.appId;
    	this.dbId = parentIndex.dbId;
    	this.rmiHost = parentIndex.rmiHost;
    	this.rmiPort = parentIndex.rmiPort;
    	this.useCompoundFiles = parentIndex.useCompoundFiles;
        this.remoteIndexName = Utilities.buildRmiName(this.rmiHost, this.rmiPort, this.appId, this.dbId);
    }

    /**
     * Sub index constructor
     */
    public LuceneIndex(LuceneIndex parentIndex)
    {
    	super(new File(parentIndex.fsdFile.getAbsolutePath() + File.separator + "current"));
    	this._isSub = true;
    	this.appId = parentIndex.appId;
    	this.dbId = parentIndex.dbId;
    	this.rmiHost = parentIndex.rmiHost;
    	this.rmiPort = parentIndex.rmiPort;
    	this.useCompoundFiles = parentIndex.useCompoundFiles;
        this.remoteIndexName = Utilities.buildRmiName(this.rmiHost, this.rmiPort, this.appId, this.dbId);
    }

    public void configure(Configuration configuration) throws ConfigurationException {

        configureQueryParser(configuration);

        //configuring the names for the remote objects
        boolean remoteConf = configuration.getAttributeAsBoolean(LuceneDocumentBase.DBELEM_ATTRIBUTE_REMOTE_ACCESS, false);
        /*TODO:FIXME-if we don't have a remote index configuration we nullify the name so that
        calling methods will not have access to a remote object name*/
        if (!remoteConf) this.remoteIndexName = null;

        if (remoteConf) {
            try {
                bindRemoteObjectToRegistry(this.remoteIndexName, new RemoteIndex(this));
            } catch (SDXException sdxE) {
                throw new ConfigurationException(sdxE.getMessage(), sdxE);
            } catch (RemoteException e) {
                throw new ConfigurationException(e.getMessage(), e);
            }
        }

        //configuring the maxFieldLength for indexation, default is 10000 based on lucene defaults
        this.maxFieldLength = configuration.getAttributeAsInteger(ATTRIBUTE_MAX_FIELD_LENGTH, DEFAULT_MAX_FIELD_LENGTH);

    }

    private void configureQueryParser(Configuration configuration) throws ConfigurationException {
        //configuring the queryparser
        Configuration qpConf = configuration.getChild(ELEMENT_NAME_QUERYPARSER, false);
        String qpClassName = DEFAULT_QUERYPARSER;
        if (qpConf != null)
            qpClassName = qpConf.getAttribute(ATTRIBUTE_CLASS, DEFAULT_QUERYPARSER);
        //TODO : add better checking here -pb
        try {
            this.queryParserClass = Class.forName(qpClassName);
        } catch (ClassNotFoundException e) {
            SDXException sdxE = new SDXException(super.getLog(), SDXExceptionCode.ERROR_CONFIGURE_QUERY_PARSER, null, e);
            throw new ConfigurationException(sdxE.getMessage(), sdxE);
        }

    }


    private void bindRemoteObjectToRegistry(String name, Remote obj) throws SDXException {

        try {
            Naming.rebind(name, obj);
        } catch (IOException e) {
            String[] args = new String[1];
            args[0] = name;
            throw new SDXException(super.getLog(), SDXExceptionCode.ERROR_BIND_REMOTE_INDEX, args, e);
        }

    }

    /**
     * Initializes the Lucene database.
     *
     * <p>It the index exists, nothing is done here. If it is doesn't exist, it will be created.</p>
     */
    public void init() throws SDXException {
    	if(!isSubIndex())
    		super.init(false);
    	else
    		super.init(false);

    	if(!this._isSub) {
    		writeCreationTimestampFile();
    		indexModified(false);
    	}
        //ensuring we have a metaAnalyzer for fields
        if (this.metaAnalyzer == null)
            throw new SDXException(super.getLog(), SDXExceptionCode.ERROR_LUCENE_INDEX_INIT, null, null);
    }

    protected void writeCreationTimestampFile() {
        try {
            File createFile = new File(super.fsdFile.getParentFile(), "creation-timestamp");
            if (!createFile.exists())
                createFile.createNewFile();

            // FIXME: Why change the date of this file if it exists ? If so, the create date is not the creation date but the the last LuceneIndex intialisation, for example the last Tomcat start ! [MP]
            this.creationTimestamp = new java.util.Date(createFile.lastModified());
        } catch (IOException e) {
            LoggingUtils.logError(super.getLog(), e.getMessage(), e);
        }

    }

    public void indexModified(boolean createLastModficationTimestampFile) throws SDXException{
        writeLastModificationTimestampFile(createLastModficationTimestampFile);
        //setDocCount(search(new TermQuery(new Term(LuceneDataStore.ALL_FIELD, LuceneDataStore.ALL_VALUE))).length());
    }

    protected void setDocCount(int i){
        _docCount = i;
    }

    public int docCount(){
       return _docCount;
    }

    public void writeLastModificationTimestampFile(boolean create) {
        try {
					java.util.Date now = new java.util.Date();
					File lastModFile = new File(super.fsdFile.getParentFile(), "last-modification-timestamp");
        	if (!lastModFile.exists()) lastModFile.createNewFile();
        	lastModFile.setLastModified(now.getTime());
					this.lastModificationTimestamp = now;
        } catch (IOException e) {
            LoggingUtils.logError(super.getLog(), e.getMessage(), e);
        }
    }


    /** Returns the type of a field.
     *
     * @param name  The name of the field for which the type is desired.
     * @return      The int field code (see Field doc)
     */
    public int getFieldType(String name) {
        return fields.getFieldType(name);
    }

    /** Returns the locale for a field.
     *
     * @param name The name of the field for which the Locale is desired.
     */
    public Locale getLocale(String name) {
        return fields.getLocale(name);
    }

    /**
     * Returns the default field for this index.
     */
    public Field getDefaultField() {
        return fields.getDefaultField();
    }


    /** Returns a field given a name.
     *
     * @param name The name of the field for which the Field is desired.
     */
    public Field getField(String name) {
        return fields.getField(name);
    }

    /** Get's an array of all the field names for the search index
     *
     * @return
     */
    /*TODO?: discuss the usefulness of this method and the proper information needed as requested by fred?
	public String[] getFieldNames(){
		if (fields != null){
			fields.
			fields
		}

	}
	 */


    /**Gets the MetaAnalyzer*/
    public MetaAnalyzer getMetaAnalyzer() {
        return metaAnalyzer;

    }

    /**Sets the MetaAnalyzer
     *
     * @param mAnalyzer A MetaAnalyzer object containing a FieldList object for this index.
     * @throws SDXException
     */
    public void setMetaAnalyzer(MetaAnalyzer mAnalyzer) throws SDXException {
        //setting the metaAnalyzer
        this.metaAnalyzer = mAnalyzer;
        //assigning the class field
        this.fields = this.metaAnalyzer.getFieldList();
        super.analyzer = this.metaAnalyzer;

    }

    /**
     * Stores a Lucene document within the database.
     *
     * @param   ldoc     The Lucene document to store.
     * @param   batchIndex     Indicates wheter a tempBatch index is taking place or not. Useful for efficiency of index optimizations
     */
    public synchronized void writeDocument(org.apache.lucene.document.Document ldoc, boolean batchIndex) throws SDXException {
    	writeDocument(ldoc, batchIndex, true);
    }

    /**
     * Stores a Lucene document within the database.
     *
     * @param   ldoc     The Lucene document to store.
     * @param   batchIndex     Indicates wheter a tempBatch index is taking place or not. Useful for efficiency of index optimizations
     * @param	autoOptimize	Is the DocumentBase autoOptimize ?
     */
    public synchronized void writeDocument(org.apache.lucene.document.Document ldoc, boolean batchIndex, boolean autoOptimize) throws SDXException {
        IndexWriter w = null;
        try {

        	//if the DocumentBase is not autoOptimize, or we have a single document, then indexes directly in the current index
        	if( !autoOptimize || !batchIndex ){
        		this.write(ldoc);
        	}
        	//the DocumentBase is autoOptimize, indexes in a RAMDirectory to increase performance
        	else{
        		//if we have batchIndex and don't already have one, we create the temporary directory for the tempBatch
        		if (  batchIndex && this.tempBatch == null ) {
                    //initializing the directory
                    this.tempBatch = new RAMDirectory();
                    IndexWriter initTempBatch = new IndexWriter(tempBatch, getMetaAnalyzer(), true);
                    initTempBatch.setUseCompoundFile(this.useCompoundFiles);
                    initTempBatch.close();
                }
        		//if we still have a tempBatch  we add to it
                if (this.tempBatch != null) {
                    //getting a writer for the document tempBatch directory
                    w = this.getWriter(this.tempBatch);
                    if (w != null) {
                        //adding the document
                        w.addDocument(ldoc);
                        w.close();
                        w = null;
                    }
                }
        	}

        } catch (IOException e) {
            String[] args = new String[2];
            args[0] = super.getIndexPath();
            args[1] = e.getMessage();
            throw new SDXException(super.getLog(), SDXExceptionCode.ERROR_LUCENE_WRITE, args, e);
        } finally {
            try {
                if (w != null) w.close();
            } catch (IOException e) {
                String[] args = new String[2];
                args[0] = super.getIndexPath();
                args[1] = e.getMessage();
                throw new SDXException(super.getLog(), SDXExceptionCode.ERROR_LUCENE_WRITE, args, e);
            }
        }

    }

    public synchronized void recycleLuceneIndexSearcher() throws SDXException{
    	recycleSearcher();
    }

    public synchronized void recycleIndexReader() throws SDXException{
    	IndexWriter w;
		try {
			w = getWriter(fsd);
			if(w==null) w = getWriter();
			if(w==null) w = new IndexWriter(fsd, analyzer, false);
			w.close();
		} catch (IOException e) {
            String[] args = new String[2];
            args[0] = super.getIndexPath();
            args[1] = e.getMessage();
            throw new SDXException(super.getLog(), SDXExceptionCode.ERROR_LUCENE_WRITE, args, e);
		}
    }


    /**
     * Deletes a document from the index.
     *
     * @param	docId	        The document's id.
     */
    public synchronized void deleteDocument(String docId) throws SDXException {
        super.delete(docId);
    }

    public QueryParser getQueryParser() throws IOException, SDXException {
        try {
            if (this.queryParserClass == null)
                return null;
            else
                return (QueryParser) queryParserClass.newInstance();
        } catch (InstantiationException e) {
            throw new SDXException(super.getLog(), SDXExceptionCode.ERROR_GET_QUERY_PARSER, null, e);
        } catch (IllegalAccessException e) {
            throw new SDXException(super.getLog(), SDXExceptionCode.ERROR_GET_QUERY_PARSER, null, e);
        }
    }

    public String getRemoteIndexName() {
        return remoteIndexName;
    }

    /**Free's the resources associated with this index
     * USE WITH CARE!
     *
     * @throws IOException  Lucene IOExceptions
     */
    public void freeResources() throws IOException {
        if (this.tempBatch != null) {
            this.tempBatch.close();
            this.tempBatch = null;
        }
        super.freeResources();
    }

    /**Sets the indexing parameters
     *
     *@param params The relevant parameters for this lucene index.
     */
    public void setParameters(LuceneIndexParameters params) {
        //TODO: should we reset these after indexation or should we leave them set until the server is restarted?
        this.maxMergeDocs = params.getMaxMergeDocs();
        this.mergeFactor = params.getMergeFactor();
    }

    public synchronized IndexReader getReader() throws SDXException {
        return super.getReader();
    }

    protected synchronized IndexWriter getWriter(Directory directory) throws IOException {
        IndexWriter w = super.getWriter(directory);
        //setting parameters to the index writer if they differ from the defaults
        if (this.maxFieldLength != DEFAULT_MAX_FIELD_LENGTH)
            w.maxFieldLength = this.maxFieldLength;
        if (this.maxMergeDocs != DEFAULT_MAX_MERGE_DOCS)
            w.maxMergeDocs = this.maxMergeDocs;
        if (this.mergeFactor != DEFAULT_MERGE_FACTOR)
            w.mergeFactor = this.mergeFactor;
        return w;
    }

    /** Gets a searcher.
     *
     * @return      A lucene Searcher.
     */
    public Searchable getSearcher() {
        return super.getSearcher();
    }


    public String getFieldValue(IndexableDocument doc, String fieldName) throws SDXException {
        Utilities.checkDocument(super.getLog(), doc);
        if (!Utilities.checkString(fieldName)) return null;
        Hits h = search(new TermQuery(new Term(ID_FIELD, doc.getId())));
        // The return it if it exists
        if (h.length() == 1) {
            try {
                org.apache.lucene.document.Field field = h.doc(0).getField(fieldName);
                if (field != null)
                    return field.stringValue();
                else
                    return null;
            } catch (IOException e) {
                String[] args = new String[2];
                args[0] = fsd.toString();
                args[1] = e.getMessage();
                throw new SDXException(super.getLog(), SDXExceptionCode.ERROR_LUCENE_RETRIEVE_DOCUMENT, args, e);
            }
        } else
            return null;

    }

    public Date getCreationDate() {
        return creationTimestamp;
    }

    public Date getLastModificationDate() {
        return lastModificationTimestamp;
    }

    public void toSAX(ContentHandler hdl) throws SAXException {
        String sdxNsUri = Framework.SDXNamespaceURI;
        String sdxNsPrefix = Framework.SDXNamespacePrefix;
        String localName = Node.Name.SEARCH_LOCATION;
        String qName = sdxNsPrefix + ":" + localName;
        AttributesImpl atts = new AttributesImpl();
        //TODO : use constant names-pb
        atts.addAttribute("", Node.Name.SEARCH_ENGINE, Node.Name.SEARCH_ENGINE, Node.Type.CDATA, "lucene");
        if (Utilities.checkString(this.appId)) atts.addAttribute("", Node.Name.APPID, Node.Name.APPID, Node.Type.CDATA, this.appId);
        if (Utilities.checkString(this.dbId)) atts.addAttribute("", Node.Name.DB_ID, Node.Name.DB_ID, Node.Type.CDATA, this.dbId);
        //TODO : add other infos ? Count, directory... -pb
        atts.addAttribute("", "creationDate", "creationDate", Node.Type.CDATA, fr.gouv.culture.sdx.utils.Date.formatUtcISO8601Date(getCreationDate()));
        atts.addAttribute("", "lastModificationDate", "lastModificationDate", Node.Type.CDATA, fr.gouv.culture.sdx.utils.Date.formatUtcISO8601Date(getLastModificationDate()));
        //Let's build the index representation
        hdl.startElement(sdxNsUri, localName, qName, atts);
        //All over
        hdl.endElement(sdxNsUri, localName, qName);
    }

    protected String getClassNameSuffix() {
        return Index.CLASS_NAME_SUFFIX;
    }

    /** Index file optimization
     * @throws SDXException*/
    public synchronized void optimize() throws SDXException {
        try {
			//this.mergeCurrentBatch(true);
			IndexWriter w = new IndexWriter(fsdFile, getMetaAnalyzer(), false);
			w.optimize();
			w.close();
			w = null;
		} catch (IOException e) {
       		String[] args = new String[2];
       		args[0] = super.getIndexPath();
       		args[1] = e.getMessage();
       		throw new SDXException(super.getLog(), SDXExceptionCode.ERROR_LUCENE_WRITE, args, e);
       	}
    }

    /**Merges any batch in memory and optimize it if necessary*/
    public synchronized void mergeCurrentBatch(boolean optimize) throws SDXException {

        //check if the index directory exists
        if(!fsdFile.exists() || fsd == null) return;

        if(tempBatch == null) return;//nothing to merge

       	try {
       		//creating the new file container for the original directory
            File tmpDir = new File(this.fsdFile.getParent(), "sdx_tmp");
            //a directory for the new index
            File tmpNewDir = Utilities.createTempDirectory(super.getLog(), "new", "Index", tmpDir);
            tmpNewDir.delete();

            Directory[] dirs = new Directory[2];
            //adding the original
            dirs[0] = this.fsd;
            //adding the batch
            dirs[1] = this.tempBatch;
            //verifying the directory
            Utilities.checkDirectory(tmpNewDir.getAbsolutePath(), super.getLog());
            //creating a new temp lucene directory for the batch merge
            FSDirectory tempNewLuceneDir = super.getFSDirectory(tmpNewDir, true);
            //getting the writer for the new temp dir
            IndexWriter w = new IndexWriter(tempNewLuceneDir, getMetaAnalyzer(), true);
            w.setUseCompoundFile(this.useCompoundFiles);
            //adding the two indicies to the new directory
            w.addIndexes(dirs);
            //freeing resources after the merge
            w.close();
            w = null;
            tempNewLuceneDir.close();
            tempNewLuceneDir = null;

            //freeing resources so we can rename the directories
            this.freeResources();

            //delete the main and the current directory
            FileUtil.deleteDirectory(fsdFile);

            //now use the safe copy to get the index back
            Utilities.safeCopy(tmpNewDir, fsdFile);

            //and make it functionnal again
            init();

            //now delete all the temp dirs
            FileUtil.deleteDirectory(tmpDir);

       	} catch (IOException e) {
       		String[] args = new String[2];
       		args[0] = super.getIndexPath();
       		args[1] = e.getMessage();
       		throw new SDXException(super.getLog(), SDXExceptionCode.ERROR_LUCENE_WRITE, args, e);
       	}
       	return;
    }

    /**
	 * Tell if the index is a subIndex or not
	 */
    public boolean isSubIndex()
    {
    	return _isSub;
    }

	/**
	 * Filter for index content (all but directories)
	 */
    FilenameFilter indexContentFilter = new FilenameFilter() {
        public boolean accept(File dir, String name) {
        	return !Utilities.isNameMatchIndexFiles(name) && !name.startsWith("current");
        }
    };

	/** Save LuceneIndex files
	 * @see fr.gouv.culture.sdx.utils.save.Saveable#backup(fr.gouv.culture.sdx.utils.save.SaveParameters)
	 */
	public void backup(SaveParameters save_config) throws SDXException{
		super.backup(save_config);

	}
	/** Restore LuceneIndex files
	 * @see fr.gouv.culture.sdx.utils.save.Saveable#restore(fr.gouv.culture.sdx.utils.save.SaveParameters)
	 */
	public void restore(SaveParameters save_config) throws SDXException{
		// TODO Auto-generated method stub

	}

}
