/*
Copyright (C) 2000-2010  Ministere de la culture et de la communication (France), AJLSM
See LICENCE file
*/
package fr.gouv.culture.sdx.repository;

import fr.gouv.culture.sdx.exception.SDXException;
import fr.gouv.culture.sdx.exception.SDXExceptionCode;
import org.apache.avalon.excalibur.datasource.DataSourceComponent;

import java.sql.Connection;
import java.sql.SQLException;

/**This classes represents a SQL Connection to JDBC Source
 *
 */
public class JDBCRepositoryConnection extends AbstractRepositoryConnection {

    /** The SQL connection. */
    private Connection _conn;

    /** The data source. */
    private DataSourceComponent dataSource;

    /**
     * Creates a connection using a standard SQL Connection.
     *
     * @param dataSource    The component from cocoon representing the database
     */
    public JDBCRepositoryConnection(DataSourceComponent dataSource) throws SQLException {
        this.dataSource = dataSource;
        this._conn = dataSource.getConnection();
        this._conn.setAutoCommit(false);//defaulted
    }

    public JDBCRepositoryConnection(Connection conn) throws SQLException {
        this._conn = conn;
        this._conn.setAutoCommit(false);//defaulted
    }


    /**
     * Commit all pending operations within this active connection.
     */
    public void commit() throws SDXException {
        try {
            if (this._conn != null) this._conn.commit();
        } catch (SQLException e) {
            String[] args = new String[2];
            args[0] = dataSource.toString();
            args[1] = e.getMessage();
            throw new SDXException(super.getLog(), SDXExceptionCode.ERROR_COMMIT, args, e);
        }
    }

    /**
     * Undo all pending operations within with this connection.
     */
    public void rollback() throws SDXException {
        try {
            if (this._conn != null) this._conn.rollback();
        } catch (SQLException e) {
            String[] args = new String[2];
            args[0] = dataSource.toString();
            args[1] = e.getMessage();
            throw new SDXException(super.getLog(), SDXExceptionCode.ERROR_ROLLBACK, args, e);
        }
    }

    /**
     * Sets this connection in autocommit mode or not.
     *
     * @param   autoCommit      true enables auto-commit ; false disables auto-commit.
     */
    public void setAutoCommit(boolean autoCommit) throws SDXException {
        try {
            if (this._conn != null) this._conn.setAutoCommit(autoCommit);
        } catch (SQLException e) {
            String[] args = new String[2];
            args[0] = dataSource.toString();
            args[1] = e.getMessage();
            throw new SDXException(super.getLog(), SDXExceptionCode.ERROR_SET_AUTO_COMMIT, args, e);
        }
    }

    /**
     * Returns the SQL connection to the datasource component
     */
    public Connection getConnection() {
        return this._conn;
    }

    /**
     *Returns the datasource component from which this connection was derived
     */
    public DataSourceComponent getDataSource() {
        return this.dataSource;
    }

    /**Releases the SQL Connection to the datasource
     *
     * @throws SDXException
     */
    protected void releaseSqlConnection() throws SDXException {
        try {
            this._conn.close();
        } catch (SQLException e) {
            String[] args = new String[1];
            args[0] = e.getMessage();
            throw new SDXException(super.getLog(), SDXExceptionCode.ERROR_CLOSE_SQL_CONNECTION, args, e);
        }
    }
}
