
#include "JavaEnv.h"
#include "JavaClass.h"

//  routines supporting Java classes:

//  Construct a java (cover) class in anticipation of operating on
//  objects of that class type.  The java cover class's main job in life
//  is to remember it's class id and help you find methods

@implementation JavaClass

+ new:  (char *)name  {
  self = [JavaClass alloc];

  fprintf(stderr, "creating... java class %s\n", name);
  if (javaEnv == 0)
    javaEnv = [JavaEnv new];

  return [self init: name env: javaEnv];
}

- init: (char *)name env: (id)_env  {
  [super init];
  fprintf(stderr, "JavaClass %s in init\n", name);
  /* save a copy of our environment pointer */
  env = [_env getEnv];
  /* extract and save our java class id */
  clsid = [_env findClass: name];
  if (clsid == 0) {
    fprintf(stderr, "Couldn't find Java class %s\n", name);
    exit(1);
  }
  fprintf(stderr, "JavaClass leaving init\n");
  return self;
}

- (JNIEnv *)getEnv {
  return env;
}



//  various flavors of constructor


// constructor with no args
- (jobject) construct {
  jobject lobj, obj;

  // look up this flavor of constructor
  jmethodID constrID = [self findMethod: "<init>" signature: "()V"];

  if (constrID == 0)
    return 0;

  // call the constructor
  lobj = (*env)->NewObject(env, clsid, constrID);

  // turn this into a global reference
  obj = (*env)->NewGlobalRef(env, lobj);
  (*env)->DeleteLocalRef(env, lobj);

  // return the object id
  return obj;
}

// constructor with two int args
- (jobject) construct: (int) a : (int) b {
  jobject lobj, obj;

  // look up this flavor of constructor
  jmethodID constrID = [self findMethod: "<init>" signature: "(II)V"];

  if (constrID == 0)
    return 0;

  // call the constructor
  lobj = (*env)->NewObject(env, clsid, constrID, (jint)a, (jint)b);

  // turn this into a global reference
  obj = (*env)->NewGlobalRef(env, lobj);
  (*env)->DeleteLocalRef(env, lobj);

  // return the object id
  return obj;
}

// constructor with four int args
- (jobject) construct: (int) a : (int) b : (int)c : (int)d {
  jobject lobj, obj;

  // look up this flavor of constructor
  jmethodID constrID = [self findMethod: "<init>" signature: "(IIII)V"];

  if (constrID == 0)
    return 0;

  // call the constructor
  lobj = (*env)->NewObject(env, clsid, constrID, (jint)a, (jint)b, 
	(jint)c, (jint)d);

  // turn this into a global reference
  obj = (*env)->NewGlobalRef(env, lobj);
  (*env)->DeleteLocalRef(env, lobj);

  // return the object id
  return obj;
}


/* get the method id for a particular java method in this class */
- (jmethodID) findMethod: (char *)method signature: (char *)sig {

  jmethodID mid = (*env)->GetMethodID(env, clsid, method, sig);
  if (mid == 0) {
    fprintf(stderr, "warning: can't find method %s\n", method);
    return mid;
  }
  fprintf(stderr, "Method: %s\n", method);

  return mid;
}

/* get the method id for a particular java method in this class */
- (jfieldID) getFieldID: (char *)field signature: (char *)sig {

  jfieldID fid = (*env)->GetFieldID(env, clsid, field, sig);
  if (fid == 0) {
    fprintf(stderr, "error: can't find field %s\n", field);
    exit(1);
  }
  fprintf(stderr, "Field: %s\n", field);

  return fid;
}

/* get the method id for a particular <static> java method in this class */
- (jmethodID) findStaticMethod: (char *)method signature: (char *)sig {

  jmethodID mid = (*env)->GetStaticMethodID(env, clsid, method, sig);
  if (mid == 0) {
    fprintf(stderr, "warning: can't find static method %s\n", method);
    exit(1);
  }

  return mid;
}

/* call a static java method; no args, returns object */
- (jobject) callObjectStaticMethod: (jmethodID) mid {

  if (mid == 0) {
    fprintf(stderr, "callObjectStaticMethod: illegal method id\n");
    exit(1);
  }
    
  return (*env)->CallStaticObjectMethod(env, clsid, mid);
}

@end

