/* Copyright (c) 2007 Axel Wachtler
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

   * Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   * Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   * Neither the name of the authors nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. */

/* $Id: diagradio.c,v 1.1 2008/03/21 11:14:24 awachtler Exp $ */
/**
 * @file
 * @brief Implementation of the @ref grpAppRdiag
 *
 * @ingroup grpAppRdiag
 */
/**
 * This application provides some basic test functions of the radio chip.
 *
 */
#include <string.h>
#include "radio.h"
#include "ioutil.h"
#include "timer.h"
/* === Types ===============================================*/

typedef struct
{
    uint8_t new;
    uint8_t len;
    uint8_t crc;
    uint8_t lqi;
    uint8_t rssi;
    uint8_t seq;
} rx_frame_t;


typedef struct
{
    uint32_t rxcnt;
    uint32_t crcerr;
    uint32_t rssi;
    uint32_t lqi;

    uint32_t txcnt;
    time_t   start;
} statistic_t;

/* === Prototypes ==========================================*/
uint8_t set_next_channel(int8_t chaninc);
uint8_t set_next_pwr(int8_t pwrinc);
uint8_t set_next_cca(int8_t ccainc);
bool toggle_rxon_idle(void);
void send_frame(uint8_t seq);
void send_continous(bool active);

void show_statistic(bool reset);
void help(void);
time_t blink(timer_arg_t t);

/* === Globals =============================================*/
static uint8_t rxbuf[MAX_FRAME_SIZE];
static uint8_t txbuf[MAX_FRAME_SIZE];
static rx_frame_t rxfrm;
statistic_t stat;
int8_t verbose;
bool   conttx;
uint8_t tx_length = 42;
#define TBLINK_PERIOD (500)
#define NL "\n\r"
timer_hdl_t  th_blink;

//uint64_t extaddr_eeprom EEMEM;
//trx_param_t trxp_eeprom EEMEM;

/** factory defaults of radio parameter  */
trx_param_t PROGMEM trxp_flash = {chan: 13, txp: 0, cca: 1, edt: 11, clkm: 0, crc: 0x0355};

/**
 * @brief Main function of diagradio application.
 *
 * This routine performs the initialization of the
 * hardware modules and stays in a endless loop, which
 * interpretes the commands, received from the host interface.
 *
 */
int main(void)
{

uint16_t inchar;
uint8_t tmp, seq, *p;
trx_param_t trxp;
trx_cfg_t cfgt;

    memset(txbuf, 0x55, sizeof(txbuf));
    memset(rxbuf, 0xff, sizeof(rxbuf));
    LED_INIT();
    KEY_INIT();
    timer_init();
    hif_init(9600);
    radio_init(rxbuf, MAX_FRAME_SIZE);

    sei();

    PRINT(NL"Radio Diag 0.12"NL);
    tmp = trx_reg_read(RG_TRX_STATUS);
    PRINTF(">RADIO INIT[%s]: %s (status=0x%02x)"NL, BOARD_NAME, (tmp == 8 ? "OK" : "FAIL"), tmp );
    PRINTF("Timer-Tick %lx"NL, TIMER_TICK);
    PRINT("Config: ");

    /* CONFIG */
    cfgt = radio_config_recall();
    if (cfgt != CFG_NONE)
    {
        PRINTF("use settings from %s"NL, (cfgt == CFG_EEPROM ? "eeprom" : "flash"));
    }

    th_blink = timer_start(blink,TBLINK_PERIOD,0);
    seq = 0;

    while(1)
    {
        inchar = hif_getc();
        if(inchar<0x100)
        {
            switch ((char)inchar)
            {

                case '+':
                    tmp = set_next_channel(+1);
                    PRINTF(">CHAN: %d"NL, tmp);
                    break;
                case '-':
                    tmp = set_next_channel(-1);
                    PRINTF(">CHAN: %d"NL, tmp);
                    break;

                case 'p':
                    tmp = set_next_pwr(-1);
                    PRINTF(">PWR: %d"NL, tmp);
                    break;
                case 'P':
                    tmp = set_next_pwr(1);
                    PRINTF(">PWR: %d"NL, tmp);
                    break;

                case 'c':
                    tmp = radio_do_cca();
                    PRINTF(">CCA: status=%d"NL, tmp);
                    break;
                case 'C':
                    tmp = set_next_cca(1);
                    PRINTF(">CCA: mode=%d"NL, tmp);
                    break;

                case 'R':
                    tmp = toggle_rxon_idle();
                    PRINTF(">RXON_IDLE: %d"NL, tmp);
                    break;

                case 'r':
                    radio_set_state(STATE_RX);
                    PRINT(">SATE_RX"NL);
                    break;

                case 't':
                    radio_set_state(STATE_TX);
                    PRINT(">SATE_TX"NL);
                    break;

                case 'o':
                    radio_set_state(STATE_OFF);
                    PRINT(">SATE_OFF"NL);
                    break;

                case 'l':
                    tx_length += 1;
                    tx_length &= 0x7f;
                    PRINTF(">TX LEN = %d"NL,tx_length);
                    break;

                case 'L':
                    tx_length += 10;
                    tx_length &= 0x7f;
                    PRINTF(">TX LEN = %d"NL,tx_length);
                    break;

                case 's':
                    send_frame(tx_length);
                    break;

                case 'S':
                    conttx = conttx ? 0 : 1;
                    send_continous(conttx);
                    PRINTF(">SEND CONTINOUS %s"NL, (conttx ? "START":"STOP"));
                    break;

                case 'i':
                    show_statistic(0);
                    break;

                case 'I':
                    show_statistic(1);
                    break;

                case 'h':
                    help();
                    break;

                case 'v':
                    if(verbose>0) verbose --;
                    PRINTF(">VERBOSE: %d"NL,verbose);
                    break;

                case 'V':
                    if(verbose<2) verbose ++;
                    PRINTF(">VERBOSE: %d"NL,verbose);
                    break;

                case 'g':
                    trx_parms_get(&trxp);
                    PRINTF("{"\
                           "chan: %d, txp: %d,"\
                           " cca: %d,\n\r edt: %d,"\
                           " clkm: %d, crc: 0x%04x}"NL,
                            trxp.chan, trxp.txp, trxp.cca,
                            trxp.edt, trxp.clkm,trxp.crc);
                    p = (uint8_t*)&trxp;
                    PRINTF("avrdude: w ee 8 0x%x 0x%x 0x%x 0x%x 0x%x"NL,
                           p[0],p[1],p[2],p[3],p[4]);
                    break;

                case 'G':
                    radio_config_store();
                    PRINT("params stored in EEPROM"NL);
                    break;


                case 'b':
                    JUMP_BOOT_LOADER();
                    PRINT("There is no BootLoader defined"NL);
                    break;
            }
        }

        if(rxfrm.new == 1)
        {
            if (verbose > 0)
            {
                PRINTF("++FRAME len=%d, crc=%3s, lqi=%d rssi=%d seq=%d"NL,
                    rxfrm.len, rxfrm.crc ? "ERR" : "OK",
                    rxfrm.lqi, rxfrm.rssi, rxfrm.seq);
            }
            if (verbose > 1)
            {
                DUMP(rxfrm.len, rxbuf);
            }
            rxfrm.new = 0;
        }

    }
    return 0;

}


/**
 * @brief Increment/decrement channel.
 */

uint8_t set_next_channel(int8_t chaninc)
{
radio_param_t rp;

    radio_get_param(phyCurrentChannel, &rp);
    rp.curr_channel += chaninc;

    if (rp.curr_channel > MAX_CHANNEL)
    {
        rp.curr_channel = MIN_CHANNEL;
    }
    if (rp.curr_channel < MIN_CHANNEL)
    {
        rp.curr_channel = MAX_CHANNEL;
    }
    radio_set_param(phyCurrentChannel, &rp);
    return rp.curr_channel;

}

/**
 * @brief Enable/disable mode RX_ON_IDLE.
 */
bool toggle_rxon_idle(void)
{
radio_param_t rp;

    radio_get_param(phyRxOnIdle, &rp);
    rp.rxon_idle = rp.rxon_idle == true ? false : true;

    radio_set_param(phyRxOnIdle, &rp);
    return rp.rxon_idle;

}

/**
 * @brief Increment/decrement TX power.
 */
uint8_t set_next_pwr(int8_t pwrinc)
{
radio_param_t rp;

    radio_get_param(phyTransmitPower, &rp);
    rp.tx_pwr += pwrinc;

    if (rp.tx_pwr == 16)
    {
        rp.tx_pwr = 0;
    }
    else if (rp.tx_pwr > 17)
    {
        rp.tx_pwr = 15;
    }


    radio_set_param(phyTransmitPower, &rp);
    return rp.tx_pwr;
}


/**
 * @brief Select CCA Mode.
 */
uint8_t set_next_cca(int8_t ccainc)
{
radio_param_t rp;

    radio_get_param(phyCCAMode, &rp);
    rp.cca_mode += ccainc;

    if (rp.cca_mode == 4)
    {
        rp.cca_mode = 0;
    }
    else if (rp.cca_mode > 5)
    {
        rp.cca_mode = 3;
    }
    radio_set_param(phyCCAMode, &rp);
    return rp.cca_mode;
}

/**
 * @brief Callback for errors in radio module functions.
 */
void usr_radio_error(radio_error_t err)
{
uint8_t regval, r2, r3;
uint16_t t;
    regval = trx_reg_read(RG_TRX_STATUS);
    r2 = trx_reg_read(0x30);
    r3 = trx_reg_read(RG_VREG_CTRL);
    while(1)
    {

        PRINTF("\n\rRADIO ERROR : %d radio status : 0x%02x:0x%02x:0x%02x\n\r",
                err, regval, r2, r3);
        t = 2000;
        while(t--)
        {
            DELAY_US(1000);
        }
    }
}

/**
 * @brief Transmit a frame with a given payload length.
 */
void send_frame(uint8_t frmlen)
{
    static uint8_t seqnb;

    seqnb ++;
    if (frmlen < 5)
    {
        frmlen = 5;
    }
    if (frmlen > MAX_FRAME_SIZE)
    {
        frmlen = MAX_FRAME_SIZE;
    }
    txbuf[0] = 1;
    txbuf[1] = 0;
    txbuf[2] = seqnb;
    txbuf[frmlen - 2] = 0;
    txbuf[frmlen - 1] = 0;
    radio_set_state(STATE_TX);
    radio_send_frame(frmlen, txbuf, 0);

    PRINTF(">SEND len=%d"NL,frmlen);
    if (verbose > 1)
    {
        DUMP(frmlen, txbuf);
    }
}

/**
 * @brief Send frames permanently (next frame is triggered at TX_END_IRQ).
 */
void send_continous(bool active)
{
radio_param_t rp;

    if(active)
    {
        rp.rxon_idle = 2;
        radio_set_param(phyRxOnIdle, &rp);
        send_frame(tx_length);
    }
    else
    {
        rp.rxon_idle = 0;
        radio_set_param(phyRxOnIdle, &rp);
    }

}



/**
 * @brief Callback function for frame reception.
 */
uint8_t * usr_radio_receive_frame(uint8_t len, uint8_t *frm, uint8_t lqi,
                                  uint8_t rssi, uint8_t crc)
{
    rxfrm.new = 1;
    rxfrm.len = len;
    rxfrm.crc = crc;
    rxfrm.lqi = lqi;
    rxfrm.rssi = rssi;
    if (rxfrm.len > 3)
    {
        rxfrm.seq = frm[2];
    }

    stat.crcerr += ( (crc != 0)||(len==0) ? 1 : 0);
    stat.rxcnt ++;
    stat.lqi += lqi;
    stat.rssi += rssi;
    return frm;
}

/**
 * @brief Callback function for TX_END IRQ.
 */
void usr_radio_tx_done(radio_tx_done_t status)
{
static uint8_t sn;
   stat.txcnt++;
   if(conttx)
   {
        /* force next frame sending */
        TRX_SLPTR_HIGH();
        TRX_SLPTR_LOW();
        trx_sram_write(3, 1, &sn);
        sn ++;
   }
}

/**
 * @brief Display RX/TX transceiver statistic and state.
 */
void show_statistic(bool reset)
{

time_t now;
radio_param_t rp;

    now = timer_systime();
    PRINTF(
        ">STAT"
        " duration: %ld ticks"NL,
            now - stat.start);


    radio_get_param(phyCurrentChannel, &rp);

    PRINTF(
        " RX:"
        " frames: %ld"
        " crcerr: %ld"NL,
            stat.rxcnt, stat.crcerr);
    PRINTF(
        " RX: channel %d"
        " avg rssi: %ld"
        " avg lqi:  %ld"NL,
            rp.curr_channel,
            stat.rssi/stat.rxcnt, stat.lqi/stat.rxcnt);

    PRINTF(
        " TX:"
        " frames: %ld"NL,
            stat.txcnt);
    if (reset)
    {
        memset(&stat, 0, sizeof(stat));
        stat.start = timer_systime();
    }

}

/**
 * @brief Print help for hotkeys.
 */
void help(void)
{
    PRINT("i/I : show / show and reset statistic"NL);
    PRINT("o   : set state OFF"NL);
    PRINT("r   : set state RX"NL);
    PRINT("t   : set state TX"NL);
    PRINT("+/- : incr./decr. channel"NL);
    PRINT("s   : send a test frame"NL);
    PRINT("S   : start/stop continous sending frames"NL);
    PRINT("R   : toggle RXON_IDLE parameter"NL);
    PRINT("P/p : incr./decr. power"NL);
    PRINT("c   : do CCA measurement"NL);
    PRINT("C   : change CCA mode"NL);
    PRINT("V/v : incr./decr. verbosity (0...2)"NL);
    PRINT("G/g : store/show seetings"NL);
}

/**
 *  @brief Life LED timer service routine.
 */
time_t blink(timer_arg_t t)
{
static volatile uint8_t btick;
    LED_SET_VALUE(btick++);
    /* restart the timer again */
    return  TBLINK_PERIOD;
}

