/* Copyright (c) 2011 - 2013 Daniel Thiele, Axel Wachtler
 All rights reserved.

 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions
 are met:

 * Redistributions of source code must retain the above copyright
 notice, this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 notice, this list of conditions and the following disclaimer in the
 documentation and/or other materials provided with the distribution.
 * Neither the name of the authors nor the names of its contributors
 may be used to endorse or promote products derived from this software
 without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 POSSIBILITY OF SUCH DAMAGE. */

/**
 * @file
 *
 *
 *
 * @brief
 * @ingroup
 */


#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/sleep.h>
#include <util/delay.h>

#if defined(__AVR_ATmega128RFA1__)

static volatile uint8_t adcfinished = 0;
static int8_t adc_offset = 0;

 ISR(ADC_vect, ISR_BLOCK)
{
	adcfinished = 1;
}

/*
 * \brief Trigger sleep based ADC measurement
 * Function is blocking until flag "adcfinished" is set by ISR
 *
 * @return ADC register value
 */
static inline int16_t adc_measure(void)
{
	set_sleep_mode(SLEEP_MODE_ADC);
	/* dummy cycle */
	adcfinished = 0;
	do
	{
		sleep_mode();
		/* sleep, wake up by ADC IRQ */

		/* check here for ADC IRQ because sleep mode could wake up from
		 * another source too
		 */
	} while (0 == adcfinished); /* set by ISR */
	return ADC ;
}

static int8_t measure_adc_offset(void)
{
	uint16_t val;

	ADCSRA = (1 << ADEN) | (1 << ADPS2) | (1 << ADPS1); /* PS 64 */
	ADCSRB = 0;
	ADMUX = (1 << REFS1) | (1 << REFS0) | (1 << MUX3); /* reference: 1.6V, differential ADC0-ADC0 10x */

	_delay_us(200); /* some time to settle */

	ADCSRA |= (1 << ADIF); /* clear flag */
	ADCSRA |= (1 << ADIE);

	/* dummy cycle after REF change (suggested by datasheet) */
	adc_measure();

	_delay_us(100); /* some time to settle */

	val = adc_measure();

	ADCSRA &= ~((1 << ADEN) | (1 << ADIE));

	return (val);
}

void measure_init_rfa(void)
{
	adc_offset = measure_adc_offset();
}

/**
 * \brief Supply voltage measurement
 * Method: set 1.1V reference as input and AVCC as reference
 * 	this returns a negative result: AVCC = 1.1V - result
 *
 * \return The MCU internal voltage in [mV]
 *
 * \author Daniel Thiele
 */
float measure_tmcu_rfa(void)
{
	int32_t val = 0;
	uint8_t nbavg = 5;

	ADCSRA = (1 << ADEN) | (1 << ADPS2) | (1 << ADPS1); /* PS 64 */
	ADCSRB = (1 << MUX5);

	ADMUX = (1 << REFS1) | (1 << REFS0) | (1 << MUX3) | (1 << MUX0); /* reference: 1.6V, input Temp Sensor */
	_delay_us(200); /* some time to settle */

	ADCSRA |= (1 << ADIF); /* clear flag */
	ADCSRA |= (1 << ADIE);

	/* dummy cycle after REF change (suggested by datasheet) */
	adc_measure();

	_delay_us(100); /* some time to settle */

	for(uint8_t i=0;i<nbavg;i++){
		val += adc_measure() - adc_offset;
	}

	ADCSRA &= ~((1 << ADEN) | (1 << ADIE));

	return (1.13 * val / (float)nbavg - 272.8);
}

/*
 * \brief Measure internal voltage of ATmega128RFA1
 *
 * Cannot be measured via ADC, use Batmon of TRX part
 *
 */
float measure_vmcu_rfa(void)
{
	uint16_t v; /* voltage in [mV] */
	uint8_t vth;

	for(vth=0;vth<32;vth++){
		BATMON = vth & 0x1F;

		if(0 == (BATMON & (1<<BATMON_OK))){
			break;
		}
	}

	if(vth & 0x10){
		v = 2550 + 75*(vth&0x0F); /* high range */
	}else{
		v = 1700 + 50*(vth&0x0F); /* low range */
	}

	return v;
}
#endif /* defined(__AVR_ATmega128RFA1__) */
/* EOF */
