' docoll Windows rsync client script

' Copyright (C) 2011 Charles Atkinson
'
' This program is free software; you can redistribute it and/or modify
' it under the terms of the GNU General Public License as published by
' the Free Software Foundation; either version 2 of the License, or
' (at your option) any later version.
'
' This program is distributed in the hope that it will be useful,
' but WITHOUT ANY WARRANTY; without even the implied warranty of
' MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
' GNU General Public License for more details.
'
' You should have received a copy of the GNU General Public License
' along with this program; if not, write to the Free Software
' Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

' Purpose: synchronises configured file trees to an rsync server

' Usage: described in "docoll Windows rsync client sysadmin guide".

' Developed and tested using Windows Script Host Version 5.6 (WXP) and 5.8 (W7)

' Notes for developers:
'      * Pro-forma function and subroutine header:
'      '*********************************************************
'      ' Purpose: What the procedure does (not how).
'      ' Assumptions: 
'      '     List of any external variable, control, or other element whose
'      '     state affects this procedure.
'      ' Effects:
'      '     List of the procedure's effect on each external variable, control,
'      '     or other element.
'      ' Inputs: 
'      '     Explanation of each argument that is not obvious. Each argument
'      '     should be on a separate line with inline comments.
'      ' Return Values: Explanation of the value returned (functions only).
'      '*********************************************************
'     * Variable naming conventions
'       > Prefixes:
'       Script scope: s (before other prefixes)
'       Boolean: bln
'       Integer: int
'       Object: obj
'       String: str
'   * Error handling: the script tries to continue to maximise chance of
'     synchronisation

' Function and Subroutine call tree
'    +
'    |
'    +-- Initialise
'    |   |
'    |   +-- ParseArguments
'    |   |
'    |   +-- ParseCfg
'    |   |   |
'    |   |   +-- ParseSection
'    |   |   |
'    |   |   +-- ParseKey
'    |   |
'    |   +-- ChkPath
'    |
'    +-- Synchronise
'    |   |
'    |   +-- SynchroniseOneDrive
'    |       |
'    |       +-- GenerateBackupDir
'    |
'    +-- Finalise
'
' Functions and subroutines called from several places: ConvertPathToCygdrive, FormatErrDescription, Log, MakeTimestamp, MyTrim 

Option Explicit

' Declare script scope constants
Const ALPHABET="ABCDEFGHIJKLMNOPQRSTUVWXYZ"
Const CONFIGS_ARCHIVE_FOLDER = 3
Const CONFIGS_DRIVE_LETTER = 0
Const CONFIGS_EXCLUDE_FROM = 2
Const CONFIGS_FILES_FROM = 1
Const CONFIGS_GLOBAL = -1
Const CONFIGS_INVALID = -2
Const DATE_PART_ABBREVIATE = True
Const DEFAULT_CONF_FILE = "docoll_cwrsync.ini"
Const DEFAULT_CONF_DIR = "conf"
Const DEFAULT_LOG_DIR = "logs"
Const DQUOTE = """"
Const FSO_OPEN_TEXT_FILE_FOR_READING = 1
Const R = 0

' Declare script scope variables
Private sblnDebug
Private sblnError
Private sblnWarn
Private sintLogRetention
Private sobjFSO
Private sobjLogFile
Private sobjWshShell
Private sstrBackupDir
Private sstrComputerName
Private sstrConfFile
Private sstrDocollHome
Private sstrLogFile
Private sstrLogDir
Private sstrPassword
Private sstrMyName
Private sstrCwRsyncHome
Private sstrSystemDrive
Private sstrServerID
Private sstrConfigs(25, 3)

' Function and Sub definitions in alphabetical order.  Execution begins after
' the last one.

'*********************************************************
' Purpose: Checks the given path exists and has the requested access
' Effects: 
'   * Adds any error mesage to strErrorMsg
' Return:
'   True if the given path exists and has the requested access
'   False otherwise
'*********************************************************
Function ChkPath(strPath, strAccess, ByRef strErrorMsg)

    Dim objFile

    ' Could use FSO' File object's Attributes method but this is more direct 
    ' and simpler.
    Select Case strAccess
        Case R
            On Error Resume Next
            Err.Clear
            Set objFile = sobjFSO.OpenTextFile(strPath, FSO_OPEN_TEXT_FILE_FOR_READING)
            If Err.Number = 0 Then
                ChkPath = True
                objFile.Close
            Else
                FormatErrDescription
                strErrorMsg = strErrorMsg & vbCrLf & "  Could not open " & strPath & " for reading: " & Err.Number & " " & Err.Description
                ChkPath = False
            End If
            On Error GoTo 0
        Case Else
            Log "E", "Function Log called with invalid access of " & strClass
            ChkPath = False
    End Select

End Function

'*********************************************************
' Purpose: Converts the path to cygdrive format
' Input: path must begin with "<drive letter>:"
' Return: the converted path
'*********************************************************
Function ConvertPathToCygdrive (strPath)

    Dim strDriveLetter
    
    strDriveLetter = Ucase(Mid(strPath, 1, 1))
    strPath = Mid(strPath, 3)
    strPath = "/cygdrive/" & strDriveLetter & Replace(strPath, "\", "/")
    ConvertPathToCygdrive = strPath
    
End Function

'*********************************************************
' Purpose: cleans up and quits
'*********************************************************
Sub Finalise (intReturnValue)

    Dim objFile, objFiles, objFolder

    If sblnError Then
        Log "E", "There was at least one error"
        intReturnValue = 1
    ElseIf sblnWarn Then
        Log "W", "There was at least one warning"
        intReturnValue = 1
    End If

    ' Final message
    Log "I", "Quitting with return value " & intReturnValue 

    ' Delete old log files
    If Not IsEmpty(sstrLogDir) Then
        Set objFolder = sobjFSO.GetFolder(sstrLogDir) 
        Set objFiles = objFolder.Files
        For each objFile in objFiles    
            If ( _
                    LCase(Right(Cstr(objFile.Name), 3)) = "log" _
                    Or LCase(Right(Cstr(objFile.Name), 7)) = "log.rtf" _ 
                ) _
                And objFile.DateLastModified < (Date() - sintLogRetention) Then
                On Error Resume Next
                Err.Clear
                objFile.Delete
                If Err.Number <> 0 Then
                    FormatErrDescription
                    Log "E", "Could not delete old log file " & Cstr(objFile.Name) & Err.Number & " " & Err.Description
                End If
                On Error GoTo 0
            End If 
        Next
    End If

    ' Close log file
    On Error Resume Next
    sobjLogFile.Close
    On Error GoTo 0

    WScript.Quit(intReturnValue)

End Sub

'*********************************************************
' Purpose: formats Err.Description for use in error messages
'*********************************************************
Sub FormatErrDescription ()
    If Err.Description <> Empty And Err.Description <> Null Then
        Err.Description = Err.Description
    Else
        Select Case Err.Number
            Case 52 Err.Description = "Bad file name"
            Case 53 Err.Description = "File not found"
            Case 70 Err.Description = "Permission denied"
        End Select
    End If
End Sub

'*********************************************************
' Purpose: generates the backup directory from the archive folder string
' Actions: 
'   * Substitutes any timestamp tokens
'   * Converts to cygdrive
'*********************************************************
Function GenerateBackupDir (strArchiveFolder)

    Dim strBackupDir,  strAMorPM

    If sblnDebug Then
        Log "D", "Function GenerateBackupDir started" 
    End If

    ' %y% for year, two digit
    strArchiveFolder = Replace(strArchiveFolder, "%y%", Right(DatePart( "yyyy", Now), 2))
    ' %Y% for year, four digit
    strArchiveFolder = Replace(strArchiveFolder, "%Y%", DatePart( "yyyy", Now))
    ' %b% for month name, example Jan
    strArchiveFolder = Replace(strArchiveFolder, "%b%", MonthName(DatePart("m", Now), DATE_PART_ABBREVIATE))
    ' %B% for month name, example January
    strArchiveFolder = Replace(strArchiveFolder, "%B%", MonthName(DatePart("m", Now)))
    ' %n% for month number (01 to 12)
    strArchiveFolder = Replace(strArchiveFolder, "%n%", Right( "0" & DatePart("m", Now), 2))
    ' %a% for day name, example Mon
    strArchiveFolder = Replace(strArchiveFolder, "%a%", WeekdayName(DatePart("w", Now), DATE_PART_ABBREVIATE))
    ' %A% for day name, example Monday
    strArchiveFolder = Replace(strArchiveFolder, "%A%", WeekdayName(DatePart("w", Now)))
    ' %d% for day of month, example 01
	strArchiveFolder = Replace(strArchiveFolder, "%d%", Right( "0" & Day(Now), 2))
    ' %H% for hour (00  23)
    strArchiveFolder = Replace(strArchiveFolder, "%H%", Right( "0" & DatePart("h", Now), 2))
    ' %I% for hour (01 to 12)
    strArchiveFolder = Replace(strArchiveFolder, "%I%", Right( "0" & DatePart("h", Now) Mod 12, 2))
    ' %p% for AM or PM.  According to ISO, 12:00 is ambiguous so arbirarily make it PM.
    If DatePart("h", Now) < 12 Then
        strAMorPM = "AM"
    Else
        strAMorPM = "PM"
    End If
    strArchiveFolder = Replace(strArchiveFolder, "%p%", strAMorPM)
    ' %M% for minute (00 to 59)
    strArchiveFolder = Replace(strArchiveFolder, "%M%", Right( "0" & DatePart("n", Now), 2))
    ' %S% for second (00 to 60)
    strArchiveFolder = Replace(strArchiveFolder, "%S%", Right( "0" & DatePart("s", Now), 2))

    strBackupDir = Replace(strArchiveFolder, "\", "/")
    If sblnDebug Then
        Log "D", "Function GenerateBackupDir returning '" & strBackupDir & "'" 
    End If
    
    GenerateBackupDir = strBackupDir 
End Function

'*********************************************************
' Purpose: initialises the script in preparation for running Synchronise
' Actions: 
'   * Creates access objects
'   * Parses any command line arguments 
'   * Parses the configuration file
'   * Sets up logging
'*********************************************************
Sub Initialise ()

    ' Declare variables
    Dim blnChkConfig
    Dim intIdx
    Dim strArchiveFolder
    Dim strBuf
    Dim strDriveLetter
    Dim strErrorMsg
    Dim strFilesFrom
    Dim strExcludeFrom
    Dim strInfoMsg
    Dim strTimestamp

    ' Create access objects
    Set sobjWshShell = WScript.CreateObject("WScript.Shell")
    Set sobjFSO = CreateObject("Scripting.FileSystemObject")

    ' Set unchanging configuration variables
    sstrMyName = WScript.ScriptName
    sstrSystemDrive = sobjWshShell.ExpandEnvironmentStrings("%systemdrive%")
    
    ' Get location of cwRsync and docoll folders
    strBuf = sobjWshShell.ExpandEnvironmentStrings("%ProgramFiles(x86)%")
    If strBuf = "%ProgramFiles(x86)%" Then
        strBuf = sobjWshShell.ExpandEnvironmentStrings("%ProgramFiles%")
    End If
    sstrCwRsyncHome = strBuf & "\cwrsync"
    sstrDocollHome = sstrCwRsyncHome & "\docoll"
    
    ' Parse command line
    blnChkConfig = False
    sblnDebug = False
    sstrConfFile = sstrDocollHome & "\" & DEFAULT_CONF_DIR & "\" & DEFAULT_CONF_FILE
    ParseArguments sstrConfFile, blnChkConfig

    ' Parse configuration file
    sintLogRetention = 28
    sstrComputerName = sobjWshShell.ExpandEnvironmentStrings("%computername%")
    sstrLogDir = sstrDocollHome & "\" & DEFAULT_LOG_DIR
    strErrorMsg = ""
    ParseCfg sstrConfFile, strErrorMsg
    If IsEmpty(sstrPassword) Then
        strErrorMsg = strErrorMsg & vbCrLf & "  Password not set"
    End If
    If IsEmpty(sstrServerID) Then
        strErrorMsg = strErrorMsg & vbCrLf & "  Server-ID not set"
    End If

    ' Set up logging
    sblnError = False
    sblnWarn = False
    ' TODO: if folder exists, ensure it is writeable.  Saner choice of alternative folder?
    If Not sobjFSO.FolderExists(sstrLogDir) Then
        sstrLogDir = sstrSystemDrive & "\"
        Wscript.Echo "Creating log file in " & sstrLogDir
    End If
    strTimestamp = MakeTimestamp
    sstrLogFile = sstrLogDir & "\docoll_cwrsync." & strTimestamp & ".log"
    On Error Resume Next
    Err.Clear
    Set sobjLogFile = sobjFSO.CreateTextFile(sstrLogFile)
    If Err.Number <> 0 Then
        ' Can't log anything so at least help any interactive problem investigator
        FormatErrDescription
        Wscript.Echo "Could not create log file " & sstrLogFile & ". Error " & Err.Number & " " & Err.Description
    End If

    ' Post-process the configuration data and build configuration report
    For intIdx = LBound(sstrConfigs, 1) To UBound(sstrConfigs, 1)
        strDriveLetter= sstrConfigs(intIdx, CONFIGS_DRIVE_LETTER)
        If Not IsEMpty(strDriveLetter) Then
            strInfoMsg = strInfoMsg & vbCrLf & "  Drive " & strDriveLetter
            ' Archive folder
            strArchiveFolder = sstrConfigs(intIdx, CONFIGS_ARCHIVE_FOLDER)
            If IsEmpty(strArchiveFolder) Then
                strArchiveFolder = "_Changed and deleted files\%Y%\%b%\%d%\%H%-%M%"
            Else
				If InStr( strArchiveFolder, "/" ) > 0 Then
                    strErrorMsg = strErrorMsg & vbCrLf & "  Archive-folder path cannot include /: " & strArchiveFolder
                    ' Disable synchronising this drive
                    sstrConfigs(intIdx, CONFIGS_DRIVE_LETTER) = ""
                End If
            End If
            sstrConfigs(intIdx, CONFIGS_ARCHIVE_FOLDER) = strArchiveFolder
			strInfoMsg = strInfoMsg & vbCrLf & "    Archive-folder: " & strArchiveFolder
            ' FilesFrom
            strFilesFrom = sstrConfigs(intIdx, CONFIGS_FILES_FROM)
            If Not IsEmpty(strFilesFrom) Then
                strFilesFrom = sstrDocollHome & "\" & DEFAULT_CONF_DIR & "\" & strFilesFrom
                strInfoMsg = strInfoMsg & vbCrLf & "    Files-from: " & strFilesFrom
                If ChkPath(strFilesFrom, R, strErrorMsg) Then
                    sstrConfigs(intIdx, CONFIGS_FILES_FROM) = strFilesFrom
                Else
                    ' File not readable; disable synchronising this drive
                    sstrConfigs(intIdx, CONFIGS_DRIVE_LETTER) = ""
                End If
            Else
               strErrorMsg = strErrorMsg & vbCrLf & "  No files-from keyword for drive " & strDriveLetter
            End If
            ' ExcludeFrom
            strExcludeFrom = sstrConfigs(intIdx, CONFIGS_EXCLUDE_FROM)
            If Not IsEmpty(strExcludeFrom) Then
                strExcludeFrom = sstrDocollHome & "\" & DEFAULT_CONF_DIR & "\" & strExcludeFrom
                strInfoMsg = strInfoMsg & vbCrLf & "    Exclude-from: " & strExcludeFrom
                If ChkPath(strExcludeFrom, R, strErrorMsg) Then
                    sstrConfigs(intIdx, CONFIGS_EXCLUDE_FROM) = strExcludeFrom
                Else
                    ' File not readable; disable synchronising this drive
                    sstrConfigs(intIdx, CONFIGS_DRIVE_LETTER) = ""
                End If
            End If
        End If
    Next

    ' Report any configuration errors
    If strErrorMsg <> "" Then
       Log "E", "Configuration file " & sstrConfFile & strErrorMsg
       Wscript.Echo "Configuration file " & sstrConfFile & strErrorMsg
    End If
    
    ' Quit if /V verification switch was used
    If blnChkConfig Then
        If strErrorMsg = "" Then
            Wscript.Echo "Configuration file " & sstrConfFile & " is OK"
        End If
        Finalise 0
    End If

    ' Log configuration values
    Log "I", "Configuration (from " & sstrConfFile & "):" & strInfoMsg

    ' Debug: log contents of sstrConfigs
    If sblnDebug Then
        Log "D", "Computer name: " & sstrComputerName 
        Log "D", "Log directory: " & sstrLogDir 
        Log "D", "Log retention: " & sintLogRetention 
        Log "D", "Server ID: " & sstrServerID 
        For intIdx = LBound(sstrConfigs, 1) To UBound(sstrConfigs, 1)
            Log "D", "sstrConfigs(" & intIdx & "): " _
                & sstrConfigs(intIdx, CONFIGS_DRIVE_LETTER) _
                & ", " & sstrConfigs(intIdx, CONFIGS_FILES_FROM) _
                & ", " & sstrConfigs(intIdx, CONFIGS_EXCLUDE_FROM) _ 
                & ", " & sstrConfigs(intIdx, CONFIGS_ARCHIVE_FOLDER)
        Next
        Log "D", "Sub Initialise returning" 
    End If

End Sub

'*********************************************************
' Purpose: logs messages
' Assumptions: log file is open for reading as sobjLogFile
' Inputs:
'   strClass: D, E, F, I or W for Debug, Error, Fatal, Information or Warning
'   strMsg: message text
'*********************************************************
Sub Log (strClass, strMsg)

    Dim strPrefix

    Select Case strClass
        Case "D" strPrefix = "DEBUG: " 
        Case "E" 
            strPrefix = "ERROR: " 
            sblnError = True
        Case "F" 
            strPrefix = "FATAL ERROR: " 
            sblnError = True
        Case "I" 
        Case "W" 
            strPrefix = "WARN: " 
            sblnWarn = True
        Case Else
            Log "E", "Sub Log called with invalid strClass of " & strClass
    End Select

    On Error Resume Next
    sobjLogFile.WriteLine(strPrefix & strMsg)    
    On Error GoTo 0

    If strClass = "F" Then
        Wscript.Echo strPrefix & strMsg
        Finalise 1
    End If

End Sub

'*********************************************************
' Purpose: returns a timestamp
'*********************************************************
Function MakeTimestamp ()
    MakeTimestamp = Right( "0" & DatePart( "d", Now), 2) _
        & "-" & MonthName(DatePart("m", Now), DATE_PART_ABBREVIATE ) _
        & "-" & DatePart( "yyyy", Now) _
        & "@" & Right( "0" & DatePart( "h", Now), 2) _
        & "-" & Right( "0" & DatePart( "n", Now), 2) _
        & "-" & Right( "0" & DatePart( "s", Now), 2)    
End Function

'*********************************************************
' Purpose: trims aribtrary characters
' Inputs:
'   strString: string to be trimmed
'   strChars: string of characters to trim
'*********************************************************
Function MyTrim (strString, strChars)

    Dim blnContinue, intIdx, strArray(), strChar
    
    ' Convert strChars to an array of single characters
    ReDim strArray(Len(strChars) - 1)
    intIdx = 1
    Do 
        strArray(intIdx - 1) = Mid(strChars, intIdx, 1)
        intIdx = intIdx + 1
    Loop While intIDx <= Len(strChars)

    ' Trim
    Do
        If strString = "" Then
            Exit Do
        End If
        For intIdx = LBound(strArray) To UBound(strArray)
            blnContinue = False
            strChar = strArray(intIdx)
            If strString <> "" Then
                If Mid(strString, 1, 1) = strChar Then
                    If Len(strString) > 1 Then
                        strString = Mid(strString, 2)
                        blnContinue = True
                    Else
                        strString = ""
                    End If
                End If
            End If
            If strString <> "" Then
                If Mid(strString, Len(strString), 1) = strChar Then
                    If Len(strString) > 1 Then
                        strString = Mid(strString, 1, Len(strString) - 1)
                        blnContinue = True
                    Else
                        strString = ""
                    End If
                End If
            End If
        Next
    Loop While blnContinue

    MyTrim = strString

End Function

'*********************************************************
' Purpose: parses the command line arguments
' Effects: may set Initialise's sstrConfFile and/or blnChkConfig
'*********************************************************
Sub ParseArguments(ByRef sstrConfFile, ByRef blnChkConfig)

    Dim intIdx, strArgLetter, strArgValue, strErrorMsg
    
    If Wscript.Arguments.Count = 0 Then
        Exit Sub
    End If
    
    For intIdx = 0 to Wscript.Arguments.Count - 1
        strArgLetter = Mid(LCase(Wscript.Arguments(intIdx)), 1, 2)
        If Len(Wscript.Arguments(intIdx)) > 2 Then
            strArgValue = Mid(Wscript.Arguments(intIdx), 4)
        Else
            strArgValue = ""
        End If
        Select Case strArgLetter
            Case "/v" blnChkConfig = True
            Case "/c" 
                If sstrConfFile <> "" Then
                    sstrConfFile = strArgValue
                Else
                    strErrorMsg = strErrorMsg & vbCrLf & "Switch /C requires a configuration file path"
                End If
            Case "/d"
                sblnDebug = True
            Case "/h"
                Wscript.Echo "Usage:" _
                    & vbCrLf & sstrMyName & " [/C:<configuration file>] [/H] [/V] " _
                    & vbCrLf & "  /C names the configuration file (default " & sstrConfFile & ")." _
                    & vbCrLf & "  /H prints this help and exits." _
                    & vbCrLf & "  /V validates the configuration file."
                 WScript.Quit(0)
            Case Else
                strErrorMsg = strErrorMsg & vbCrLf & "Invalid switch " & Wscript.Arguments(intIdx) & " (switch /H prints help)"
        End Select
    Next

    ' Report any errors
    If strErrorMsg <> "" Then
        Wscript.Echo "Command line error(s) " & strErrorMsg
        Finalise 1
    End If

End Sub

'*********************************************************
' Purpose: parses the configuration file
'*********************************************************
Sub ParseCfg (sstrConfFile, ByRef strErrorMsg)

    Dim objCfgFile
    Dim strLine
    Dim iIndex

    ' Open the configuration file
    On Error Resume Next
    Err.Clear
    Set objCfgFile = sobjFSO.OpenTextFile(sstrConfFile, FSO_OPEN_TEXT_FILE_FOR_READING)
    If Err.Number <> 0 Then
        FormatErrDescription
        Log "F", "Could not open configuration file " & sstrConfFile & " " & Err.Description
    End If
    On Error GoTo 0

    ' For each line ...
    iIndex = CONFIGS_INVALID
    Do Until objCfgFile.AtEndOfStream
        strLine = objCfgFile.Readline
        strLine = MyTrim(strLine, " " & vbTab)
        Select Case Left(strLine, 1)
            Case "[" iIndex = ParseSection(strLine, strErrorMsg)
            Case "", ";"  
            Case Else ParseKey iIndex, strLine, strErrorMsg 
        End Select
    Loop

    objCfgFile.Close

End Sub

'*********************************************************
' Purpose: parses a keyword line from the configuration file
' Effects:
'    Loads array sstrConfigs, member (iIndex, *) with the keyword value
'    where:
'        iIndex is 0 for A, 1 for 2 ... 25 for Z
'        *=CONFIGS_FILES_FROM: for files-from value
'        *=CONFIGS_EXCLUDE_FROM: for exclude-from value
'*********************************************************
Sub ParseKey (intIndex, strLine, ByRef strErrorMsg)

    Dim intMyIndex, strBuf, strKeyword, strValue

    ' Get the keyword and the value
    intMyIndex = InStr(strLine, "=" )
    If intMyIndex = 0 Then
        strErrorMsg = strErrorMsg & vbCrLf & "  Keyword line has no '=': " & strLine
        Exit Sub
    End If
    strBuf = Mid(strLine, 1, intMyIndex - 1)
    strKeyword = Lcase(MyTrim(strBuf, " " & vbTab))
    strBuf = Mid(strLine, intMyIndex + 1)
    strValue = MyTrim(strBuf, " " & vbTab)

    ' Store the value
    Select Case strKeyword
        Case "archive-folder", "archive_folder"
             If intIndex < 0 Then
                strErrorMsg = strErrorMsg & vbCrLf & "  Keyword Archive-folder found outside a drive letter section"
                Exit Sub
            End If
            sstrConfigs(intIndex, CONFIGS_ARCHIVE_FOLDER) = strValue
       Case "client-name", "client_name"
             If intIndex <> CONFIGS_GLOBAL Then
                strErrorMsg = strErrorMsg & vbCrLf & "  Keyword Client-name found outside the Global section"
                Exit Sub
            End If
            sstrComputerName = strValue
        Case "exclude-from", "exclude_from"
             If intIndex < 0 Then
                strErrorMsg = strErrorMsg & vbCrLf & "  Keyword Exclude-from found outside a drive letter section"
                Exit Sub
            End If
            sstrConfigs(intIndex, CONFIGS_EXCLUDE_FROM) = strValue
        Case "files-from", "files_from"
            If intIndex < 0 Then
                strErrorMsg = strErrorMsg & vbCrLf & "  Keyword Files-from found outside a drive letter section"
                Exit Sub
            End If
            sstrConfigs(intIndex, CONFIGS_FILES_FROM) = strValue
       Case "log-dir", "log_dir"
             If intIndex <> CONFIGS_GLOBAL Then
                strErrorMsg = strErrorMsg & vbCrLf & "  Keyword Log-dir found outside the Global section"
                Exit Sub
            End If
            sstrLogDir = strValue
       Case "log-retention", "log_retention"
             If intIndex <> CONFIGS_GLOBAL Then
                strErrorMsg = strErrorMsg & vbCrLf & "  Keyword Log-retention found outside the Global section"
                Exit Sub
            End If
            sintLogRetention = Cint(strValue)
       Case "password"
             If intIndex <> CONFIGS_GLOBAL Then
                strErrorMsg = strErrorMsg & vbCrLf & "  Keyword Password found outside the Global section"
                Exit Sub
            End If
            sstrPassword = strValue
       Case "server-id", "server_id"
             If intIndex <> CONFIGS_GLOBAL Then
                strErrorMsg = strErrorMsg & vbCrLf & "  Keyword Server-ID found outside the Global section"
                Exit Sub
            End If
            sstrServerID = strValue
        Case Else
            strErrorMsg = strErrorMsg & vbCrLf & "  Keyword invalid: " & strLine
    End Select

End Sub

'*********************************************************
' Purpose: parses a section line from the configuration file
' Effects:
'    Loads array sstrConfigs, member (iIndex, CONFIGS_DRIVE_LETTER) with the drive letter
' Return:
'    -2 on error
'    -1 on finding the [Global] section
'    Index of drive letter on finding [<drive letter>] section
'    with 0 for A, 1 for 2 ... 25 for Z
'*********************************************************
Function ParseSection (strLine, ByRef strErrorMsg)

    Dim intIndex, strBuf, strSection

    ' Get the section name
    If Mid(strLine, Len(strLine)) <> "]" Then
        strErrorMsg = strErrorMsg & vbCrLf & "  No closing ] on section " & strLine
        ParseSection = CONFIGS_INVALID
        Exit Function
    End if
    strBuf = Mid(strLine, 2)
    strBuf = Mid(strBuf, 1, Len(strBuf) - 1)
    strSection = Ucase(MyTrim(strBuf, " " & vbTab))
    If Len(strSection) = 0 Then
        strErrorMsg = strErrorMsg & vbCrLf & "  Empty section " & strLine
        ParseSection = CONFIGS_INVALID
        Exit Function
    End If

    ' Look for the [Global] section
    If strSection = "GLOBAL" Then
        ParseSection = CONFIGS_GLOBAL
        Exit Function
    End If

    ' Look for a [<drive letter>] section
    intIndex = InStr(ALPHABET, strSection) - 1
    If Len(strSection) = 1 And intIndex >= 0  Then
        sstrConfigs(intIndex, CONFIGS_DRIVE_LETTER) = strSection
        ParseSection = intIndex
        Exit Function
    End If

    strErrorMsg = strErrorMsg & vbCrLf & "  Invalid configuration section " & strLine
    ParseSection = CONFIGS_INVALID

End Function

'*********************************************************
' Purpose: synchronises each configured drive 
'*********************************************************
Function Synchronise ()

    Dim intIdx, objEnv, strBuf, strCwrsyncBin, strDriveLetter, strLogFile
    Dim strMsg, strProcessPath, strUserPath    

    If sblnDebug Then
        Log "D", "Function Synchronise started" 
    End If
    
    ' Generate log file path
    ' .rtf is used because rsync logs with Linux line ends
    strLogFile = ConvertPathToCygdrive(sstrLogDir & "\rsync." & MakeTimestamp() & ".log.rtf")
    
    ' Debug: log environment variables before changing any
    If sblnDebug Then
        Set objEnv = sobjWshShell.Environment("USER")
        For Each strBuf In objEnv
            strMsg = strMsg & vbCrLf & "  " & strBuf
        Next
        Log "D", "USER environment variables before any changes:" & strMsg
        strMsg = ""
        Set objEnv = sobjWshShell.Environment("SYSTEM")
        For Each strBuf In objEnv
            strMsg = strMsg & vbCrLf & "  " & strBuf
        Next
        Log "D", "SYSTEM environment variables before any changes:" & strMsg
    End If

    ' Get environment variables
    strProcessPath = sobjWshShell.ExpandEnvironmentStrings("%Path%")
    ' Next line worked on WXP but not on W7-32
    ' strUserPath = sobjWshShell.RegRead("HKCU\Environment\PATH")
    strUserPath = sobjWshShell.Environment("USER").Item("PATH")    
    
    ' Set environent variables for rsync.exe
    sobjWshShell.Environment("USER").Item("CWRSYNCHOME") = sstrCwRsyncHome
    sobjWshShell.Environment("USER").Item("CWOLDPATH") = strProcessPath
    sobjWshShell.Environment("USER").Item("CYGWIN") = "nontsec"
    strCwrsyncBin = sstrCwRsyncHome & "\bin"
    If Instr(strProcessPath, strCwrsyncBin ) = 0 Then
        sobjWshShell.Environment("USER").Item("PATH") = strCwrsyncBin & ";" & strUserPath
    End If    
    ' RSYNC_PASSWORD is used because the rsync --password-file option
    ' cannot be used because this script is intended to be run by SYSTEM and 
    ' rsync requires that the password file is only readable by "root".
    ' 
    ' Experimentation showed different behaviour regards passing the password 
    ' to rsync, depending on how this script was run:
    ' * For a Scheduled Task running as SYSTEM, the SYSTEM environment
    '   variable worked; the USER one did not.
    ' * Running docoll_cwrsync.vbs from the command line, neither worked but
    '   sending the password plus CRLF on stdin worked.
    ' * Running 
    '   %ComSpec% /c CScript //NoLogo "C:\Program Files\cwRsync\docoll_cwrsync.vbs"
    '   from the command line, neither worked but sending the password plus
    '   CRLF on stdin worked.
    ' Therfore, for robustness, this script both sets the SYSTEM RSYNC_PASSWORD
    ' and writes the password to the rsync process' stdin.
    sobjWshShell.Environment("SYSTEM").Item("RSYNC_PASSWORD") = sstrPassword

    ' Debug: log environment variables
    If sblnDebug Then
        Set objEnv = sobjWshShell.Environment("USER")
        For Each strBuf In objEnv
            strMsg = strMsg & vbCrLf & "  " & strBuf
        Next
        Log "D", "USER environment variables:" & strMsg
        strMsg = ""
        Set objEnv = sobjWshShell.Environment("SYSTEM")
        For Each strBuf In objEnv
            strMsg = strMsg & vbCrLf & "  " & strBuf
        Next
        Log "D", "SYSTEM environment variables:" & strMsg
    End If
    
    ' Run rsync for each configured drive
    For intIdx = LBound(sstrConfigs, 1) To UBound(sstrConfigs, 1)
        strDriveLetter = sstrConfigs(intIdx, CONFIGS_DRIVE_LETTER)
        If Not IsEmpty(strDriveLetter) Then
            SynchroniseOneDrive intIdx, strDriveLetter, strLogFile
        End If
    Next
    
    ' Remove and restore environment variables
    sobjWshShell.Environment("SYSTEM").Remove("RSYNC_PASSWORD")
    sobjWshShell.Environment("USER").Remove("CWOLDPATH")
    sobjWshShell.Environment("USER").Remove("CWRSYNCHOME")
    sobjWshShell.Environment("USER").Remove("CYGWIN")
    sobjWshShell.Environment("USER").Item("PATH") = strUserPath

    If sblnDebug Then
        Log "D", "Function Synchronise returning" 
    End If
    
End Function

'*********************************************************
' Purpose: synchronises files on a single drive by running cwrsync 
'*********************************************************
Sub SynchroniseOneDrive (intIdx, strDriveLetter, strLogFile)

    Const EXEC_STATUS_RUNNING = 0
    Dim objExec, strCommand, strBackupDir, strBackupDirRoot, strCygdriveLetter
    Dim strExcludeFrom, strExcludeFromOpt, strFilesFrom, strLogClass 
    Dim strPasswordLine, strStderr, strStdout
    ' Set "constant"
    strPasswordLine = "Password: " & vbLf
    
    If sblnDebug Then
        Log "D", "Started sub SynchroniseOneDrive for drive " & strDriveLetter 
    End If

    ' Build command string
    strCygdriveLetter = "/cygdrive/./" & strDriveLetter
    strFilesFrom = ConvertPathToCygdrive(sstrConfigs(intIdx, CONFIGS_FILES_FROM))
    strExcludeFrom = sstrConfigs(intIdx, CONFIGS_EXCLUDE_FROM)
    If IsEmpty(strExcludeFrom) Then
        strExcludeFromOpt = ""
    Else
        strExcludeFromOpt = " --exclude-from " & DQUOTE & ConvertPathToCygdrive(strExcludeFrom) & DQUOTE
    End If
    strBackupDir = GenerateBackupDir(sstrConfigs(intIdx, CONFIGS_ARCHIVE_FOLDER))
	strBackupDirRoot = Left(strBackupDir, InStr(strBackupDir, "/"))
    strCommand = DQUOTE & sstrCwRsyncHome & "\bin" & "\rsync" & DQUOTE _
        & " --backup" _ 
        & " --backup-dir=" & DQUOTE & strBackupDir & DQUOTE _
        & " --compress" _
        & " --exclude=" & DQUOTE & strBackupDirRoot & DQUOTE _
        & strExcludeFromOpt _
        & " --files-from=" & DQUOTE & strFilesFrom & DQUOTE _
        & " --log-file=" & DQUOTE & strLogFile & DQUOTE _
        & " --partial" _
        & " --partial-dir=.rsync-partial" _
        & " --prune-empty-dirs" _
        & " --recursive" _
        & " --relative" _
        & " --times" _
        & " " & strCygdriveLetter _
        & " " & sstrComputerName & "@" & sstrServerID & "::" & sstrComputerName & "/" & strDriveLetter
    Log "I", strDriveLetter & " drive rsync command: " & strCommand
    
    ' Run command
    On Error Resume Next
    Err.Clear
    Set objExec = sobjWshShell.Exec(strCommand)
    If Err.Number <> 0 Then
        On Error GoTo 0
        FormatErrDescription
        Log "E", "Error starting rsync: " & Err.Number & " " & Err.Description
        If sblnDebug Then
            Log "D", "Sub SynchroniseOneDrive exiting" 
        End If
        Exit Sub
    End If
    On Error GoTo 0
    
    ' Write password to the command's process
    objExec.StdIn.Write sstrPassword & vbCrLf
    
    ' Monitor the command's process
    Do While objExec.Status = EXEC_STATUS_RUNNING
        Wscript.Sleep 1000    ' milli-seconds
        strStderr = objExec.StdErr.ReadAll
        If strStderr = strPasswordLine Then
            strStderr = ""
        End If
        If (Not objExec.StdOut.AtEndOfStream) Or (strStderr <> "") Then
            Exit Do    
        End If
    Loop
    Wscript.Sleep 1000    ' Allow time for more output
    strStdout =  objExec.StdErr.ReadAll
    strStderr = strStderr & objExec.StdErr.ReadAll
    If strStdout <> "" Then
        Log "E", "Unexpected stdout from rsync: " _
            & vbCrLf & strStdOut
    End If
    If strStderr <> "" Then
        ' Sometimes the "Password: " line is found again
        If Instr(strSTderr, strPasswordLine) > 0 Then
            strStderr = Mid(strStderr, Len(strPasswordLine) + 1) 
        End If
        Log "E", "Unexpected stderr from rsync: " _
            & vbCrLf & strStderr
    End If
    If objExec.Status = EXEC_STATUS_RUNNING Then
        objExec.Terminate
    End If
    If objExec.ExitCode = 0 Then
        strLogClass = "I"
    Else
        strLogClass = "E"
    End If
    Log strLogClass, "Exit code from rsync for " & strDriveLetter & " drive: " & objExec.ExitCode

    If sblnDebug Then
        Log "D", "Sub SynchroniseOneDrive exiting" 
    End If
    
End Sub

'*********************************************************
' Main sequence.  Execution begins here
'*********************************************************
Initialise
' Allow time for network to be connected in case scheduled to run at system startup
Wscript.Sleep 30000    ' milli-seconds
Synchronise
Finalise 0 
