% FUNCTION [...] = gf_slice([operation [, args]])
%
%   General constructor for slice objects.
%
%   Creation of a mesh slice. Mesh slices are very similar to a
%   P1-discontinuous mesh_fem on which interpolation is very fast. The slice is
%   built from a mesh object, and a description of the slicing operation, for
%   example::
%   
%   sl = gf_slice({'planar',+1,[0;0],[1;0]}, m, 5);
%   
%   cuts the original mesh with the half space {y>0}. Each convex of the
%   original mesh `m` is simplexified (for example a quadrangle is splitted
%   into 2 triangles), and each simplex is refined 5 times.
%   
%   Slicing operations can be:
%   
%   * cutting with a plane, a sphere or a cylinder
%   * intersection or union of slices
%   * isovalues surfaces/volumes
%   * "points", "streamlines" (see below)
%   
%   If the first argument is a mesh_fem `mf` instead of a mesh, and if it is
%   followed by a `mf`-field `u` (with size(u,1) ==
%   gf_mesh_fem_get(mesh_fem MF, 'nbdof')), then the deformation `u` will be applied to the
%   mesh before the slicing operation.
%   
%   The first argument can also be a slice.
%   
%
%   * sl = gf_slice(sliceop, {slice sl|{mesh m| mesh_fem mf, vec U}, int refine}[, mat CVfids])
%   Create a slice using `sliceop` operation.
%   
%   `sliceop` operation is specified with Matlab CELL arrays (i.e.
%   with braces) . The first element is the
%   name of the operation, followed the slicing options:
%   
%   * {'none'} :
%   Does not cut the mesh.
%   
%   * {'planar', int orient, vec p, vec n} :
%   Planar cut. `p` and `n` define a half-space, `p` being a point belong to
%   the boundary of the half-space, and `n` being its normal. If `orient` is
%   equal to -1 (resp. 0, +1), then the slicing operation will cut the mesh
%   with the "interior" (resp. "boundary", "exterior") of the half-space.
%   `orient` may also be set to +2 which means that the mesh will be sliced,
%   but both the outer and inner parts will be kept.
%   
%   * {'ball', int orient, vec c, scalar r} :
%   Cut with a ball of center `c` and radius `r`.
%   
%   * {'cylinder', int orient, vec p1, vec p2, scalar r} :
%   Cut with a cylinder whose axis is the line `(p1, p2)` and whose radius
%   is `r`.
%   
%   * {'isovalues', int orient, mesh_fem mf, vec U, scalar s} :
%   Cut using the isosurface of the field `U` (defined on the mesh_fem `mf`).
%   The result is the set `{x such that `U`(x) <= `s`}` or `{x such that
%   `U`(x)=`s`}` or `{x such that `U`(x) >= `s`}` depending on the value of
%   `orient`.
%   
%   * {'boundary'[, SLICEOP]} :
%   Return the boundary of the result of SLICEOP, where SLICEOP is any
%   slicing operation. If SLICEOP is not specified, then the whole mesh is
%   considered (i.e. it is equivalent to {'boundary',{'none'}}).
%   
%   * {'explode', mat Coef} :
%   Build an 'exploded' view of the mesh: each convex is shrinked (0 <
%   `Coef` <= 1). In the case of 3D convexes, only their faces are kept.
%   
%   * {'union', SLICEOP1, SLICEOP2} :
%   Returns the union of slicing operations.
%   
%   * {'intersection', SLICEOP1, SLICEOP2} :
%   Returns the intersection of slicing operations, for example::
%   
%   sl = gf_slice({intersection',{'planar',+1,[0;0;0],[0;0;1]},
%   {'isovalues',-1,mf2,u2,0}},mf,u,5)
%   
%   * {'comp', SLICEOP} :
%   Returns the complementary of slicing operations.
%   
%   * {'diff', SLICEOP1, SLICEOP2} :
%   Returns the difference of slicing operations.
%   
%   * {'mesh', mesh m} :
%   Build a slice which is the intersection of the sliced mesh with another
%   mesh. The slice is such that all of its simplexes are stricly contained
%   into a convex of each mesh.
%   
%
%   * sl = gf_slice('streamlines', mesh_fem mf, mat U, mat S)
%   Compute streamlines of the (vector) field `U`, with seed points given
%   by the columns of `S`.
%
%   * sl = gf_slice('points', mesh m, mat Pts)
%   Return the "slice" composed of points given by the columns of `Pts`
%   (useful for interpolation on a given set of sparse points, see
%   ``gf_compute('interpolate on',sl)``.
%
%   * sl = gf_slice('load', string filename[, mesh m])
%   Load the slice (and its linked mesh if it is not given as an argument)
%   from a text file.
%
%
function [varargout]=gf_slice(varargin)
  if (nargout),
    [varargout{1:nargout}]=gf_matlab('slice', varargin{:});
  else
    gf_matlab('slice', varargin{:});
    if (exist('ans', 'var') == 1), varargout{1}=ans; end;
  end;
% autogenerated mfile;
