/*
	This file is part of jrisk.
	
	jrisk is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.
	
	jrisk is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
	
	You should have received a copy of the GNU General Public License
	along with jrisk; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
 * Created on 14-feb-2005 by Alessio Treglia
 * Copyright  2005 Alessio Treglia
 */

import java.util.Set;
import java.util.HashSet;

/**
 * Un territorio ha come caratteristiche un nome, un numero
 * di territori confinanti e un numero di armate poste su di
 * esso. Inoltre, ogni territorio appartiente ad un continente,
 * cio un gruppo di territori posti relativamente vicini, o
 * comunque tutti nella stessa area geografica.
 *
 * @author Alessio Treglia
 * @version 1.0.0
 */
public class Territorio implements Comparable {
    /** Nome del territorio */
    private String nome;
    /** Numero di armate presenti sul territorio */
    private int armate;
    /** Territori confinanti */
    private Set territoriConfinanti;
    /** Nome del continente del quale fa parte */
    private String continente;
    /** Minimo numero di armate che devono essere presenti sul territorio */
    private static final int MIN_ARMATE = 1;
    
    /**
     * Inizializza un oggetto Territorio. Imposta un nome (i.e. "cina", "egitto"),
     * e un numero di armate iniziali.
     * 
     * @param nome Nome del territorio
     * @param armate Armate iniziali
     */
    public Territorio(String nome, int armate) {
        this.nome = nome;
        this.armate = (armate < MIN_ARMATE ? MIN_ARMATE : armate);
        this.territoriConfinanti = new HashSet();
    }
    
    /**
     * Restituisce una stringa contenente il nome del territorio.
     * 
     * @return il nome del territorio in formato <code>String</code>
     */
    public String getNome() { return this.nome; }
    
    /**
     * Restituisce il numero di armate presenti sul territorio.
     * 
     * @return il numero di armate presenti sul <code>Territorio</code>
     */
    public int getArmate() { return this.armate; }
    /**
     * Restituisce il nome del continente del quale fa parte
     * in formato stringa
     * 
     * @return il continente del quale fa parte
     */
    public String getContinente() { return this.continente; }
    /**
     * Restituisce una collezione di territori confinanti.
     * 
     * @return un <code>Set</code> contenente i riferimenti
     * ai territori confinanti
     */
    public Set getTerritoriConfinanti() { return this.territoriConfinanti; }
    
    /**
     * Trasferisce su di esso un n numero di armate.
     * 
     * @param n Numero di armate da trasferire sul <code>Territorio</code>
     */
    public void setArmate(int n) { this.armate = n; }
    /**
     * Imposta il nome del continente del quale il territorio
     * fa parte. Verifica che il nome inserito sia un continente
     * accettabile. Restituisce lo stato dell'operazione.
     * 
     * @param nome nome del continente
     * @return <code>true</code> in caso di successo
     */
    public boolean setContinente(String nome) {
        if(ElencoTerritoriContinenti.continenteValido(nome)) {
            this.continente = nome;
            return true;
        }
        return false;
    }
    /**
     * Imposta un territorio come confinante.
     * 
     * @param t <code>Territorio</code> con cui confina
     */
    public void setConfinante(Territorio t) {
        territoriConfinanti.add(t);
    }
    /**
     * Imposta una collezione di territori come
     * confinanti con se stesso.
     * 
     * @param confinanti <code>Set</code> di territori confinanti
     */
    public void setConfinanti(Set confinanti) {
        this.territoriConfinanti = confinanti;
    }
    
    /**
     * Verifica la contiguit con un altro territorio.
     * Se la verifica  positiva restituisce true.
     * 
     * @param t <code>Territorio</code> del quale verifica la
     * contiguit
     * @return <code>true</code>  confinante, <code>false</code>
     * in caso contrario
     */
    public boolean confinaCon(Territorio t) {
        if(territoriConfinanti.contains(t))
            return true;
        return false;
    }
    
    /**
     * Sposta un numero di armate verso un territorio confinante.
     * Controlla che il territorio di destinazione sia confinante e che
     * vengano spostate un numero accettabile di armate (su ogni territorio
     * deve sempre rimanere almeno un'armata di difesa)
     * 
     * @param n numero di armate da spostare
     * @param destinazione <code>Territorio</code> verso il quale spostare le armate
     * @return <code>true</code> in caso di successo
     */
    public boolean muovi(int n, Territorio destinazione) {
        /*
         * Controlla che il territorio destinazione sia
         * confinante e che questo non sia vuoto
         */
        if(!this.isVuoto() && territoriConfinanti.contains(destinazione)) {
        /*
         * Controlla che sia accettabile spostare n armate...
         */
            if( n >= 0 && n < this.armate) {               
               this.armate -= n;						// elimina le armate dal territorio di partenza
               /* e le aggiunge al territorio di destinazione */
               destinazione.setArmate( destinazione.getArmate() + n );
            } else {
                /*
                 * ...altrimenti ne sposta il massimo numero possibile
                 */
                destinazione.setArmate(destinazione.getArmate() + this.armate - MIN_ARMATE);
                this.armate = MIN_ARMATE;
            }           
            return true;
        }        
        return false;
    }
    
    /**
     * Se sul territorio sono presenti due o pi armate, allora
     * si pu attaccare da esso.
     * 
     * @return <code>true</code> se sul territorio ci sono almeno due armate.
     * In caso contrario il <code>Territorio</code> pu solo difendersi.
     */
    public boolean puoAttaccare() { return (armate > MIN_ARMATE); }
    /**
     * Verifica che un territorio abbia sufficienti armate per
     * attaccare con n dadi
     * 
     * @param n numero dadi lanciati dall'attacco
     * @return <code>true</code> se n  un numero accettabile e
     * se vi sono sufficienti armate sul terrritorio
     */
    public boolean puoAttaccare(int n) {
        return ( (n > 0 && n < 4) && (armate >= n) );
    }
    
    /**
     * Restituisce il numero massimo di dadi che pu lanciare
     * in attacco il giocatore che possiede il territorio.
     * 
     * @return numero masssimo di dadi utilizzabili in un attacco
     */
    public int maxDadiAttacco() {
        if(puoAttaccare())
            if(armate < 4)
            	return this.armate;
        	else
        	    return 3;
        return 0;
    }
    
    /**
     * Restituisce il numero massimo di dadi che pu lanciare
     * in difesa il giocatore che possiede il territorio.
     * 
     * @return numero massimo di dadi utilizzabili per difendersi
     */
    public int maxDadiDifesa() {
        //return (maxDadiAttacco() > 0 ? maxDadiAttacco() : MIN_ARMATE);
        return (armate >= 3 ? 3 : armate);
    }
    
    /**
     * Controlla che sul territorio ci siano delle armate
     * 
     * @return <code>false</code> se sul <code>Territorio</code>
     * ci sono delle armate
     */
    public boolean isVuoto() { return (armate < 1); }
    /**
     * Restituisce il numero minimo di armate che devono essere
     * presenti sul territorio.
     * 
     * @return il numero minimo di armate che devono essere
     * presenti sul <code>Territorio</code> per poterlo occupare
     */
    public static int getMinimoArmate() { return MIN_ARMATE; }
    /**
     * Funzione di hash.
     * 
     * @return restituisce il codice hash dell'oggetto
     */
    public int hashCode() {
        return (this.nome.hashCode() + this.continente.hashCode());
    }
    
    /**
     * Esegue il confronto fra i codici hash di due territori.
     * 
     * @param ob <code>Territorio</code> da confrontare
     * @return un valore minore, uguale o maggiore di 0 a seconda
     * che l'oggetto chiamante sia minore, uguale o maggiore dell'
     * oggetto del parametro.
     */
    public int compareTo(Object ob) {
        Territorio altroTerritorio = (Territorio) ob;
        if(this.hashCode() < altroTerritorio.hashCode())
            return -1;
        if(this.hashCode() > altroTerritorio.hashCode())
            return 1;
        return 0;
    }
    
    /**
     * Restituisce una descrizione in formato stringa dell'oggetto.
     * 
     * @return una <code>String</code> contenente la descrizione dell'oggetto
     */
    public String toString() {
        /*StringBuffer buffer = new StringBuffer();
        
        buffer.append(nome + "\t");
        buffer.append(armate + "\t");
        buffer.append(continente + "\t");
        buffer.append(territoriConfinanti.size());
        
        return buffer.toString();*/
        return (this.nome + " (" + this.continente + ")");
    }
}
