/*
	This file is part of jrisk.
	
	jrisk is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.
	
	jrisk is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
	
	You should have received a copy of the GNU General Public License
	along with jrisk; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
 * Created on 16-feb-2005 by Alessio Treglia
 * Copyright  2005 Alessio Treglia
 */
import java.util.Set;
import java.util.Iterator;
import java.util.List;
import java.util.Collections;

/**
 * <p>Gestisce la distribuzione delle armate ai giocatori ad
 * ogni inizio di turno. Calcola le armate in base ai
 * territori posseduti, in pi somma ad esse un numero di
 * armate bonus per ogni intero continente posseduto.</p>
 * 
 * @author Alessio Treglia
 * @version 1.0.0
 */
public class DistributoreArmate {
    /** Numero di territori per ogni armata */
    private static final int ARMATE_OGNI_TERRITORI = 3;
    /** Numero di territori bonus per l'America del Nord */
    private static final int ARMATE_NORDAMERICA = 5;
    /** Numero di territori bonus per l'America del Sud */
    private static final int ARMATE_SUDAMERICA = 2;
    /** Numero di territori bonus per l'Europa */
    private static final int ARMATE_EUROPA = 5;
    /** Numero di territori bonus per l'Africa */
    private static final int ARMATE_AFRICA = 3;
    /** Numero di territori bonus per l'Asia */
    private static final int ARMATE_ASIA = 7;
    /** Numero di territori bonus per l'Oceania */
    private static final int ARMATE_OCEANIA = 2;
    
    /**
     * Scorre l'elenco dei territori posseduti da un giocatore,
     * aggiorna il conteggio dei territori di ogni continente.
     * Infine controlla se il giocatore possiede qualche continente
     * intero. Se  cos, restituisce un numero di armate bonus
     * per ogni continente posseduto.
     * 
     * @param giocatore <code>Giocatore</code> al quale assegnare le armate
     * @return un numero di armate bonues, calcolato in base ai
     * continenti controllati per interi.
     */
    private static int calcolaArmatePerContinente(Giocatore giocatore) {
        Set territori = giocatore.getTerritori();				// Riceve la lista dei territori dal giocatore
    	Territorio temp;										// Riferimento ad un oggetto temporaneo
        /* Contatori di territori posseduti su ogni continente */
    	int nordAmerica, sudAmerica, europa, africa, asia, oceania;
        Iterator it = territori.iterator();						// Iteratore per la lista dei territori
        int armate = 0;											// Numero armate bonus
        
        /*
         * Inizializzo i contatori dei territori
         */
        nordAmerica = 0;
        sudAmerica = 0;
        europa = 0;
        africa = 0;
        asia = 0;
        oceania = 0;
        /*
         * Per ogni territorio, ricevo il nome del continente
         * al quale appariente, dunque ne incremento il
         * contatore.
         */
        while(it.hasNext()) {
            temp = (Territorio) it.next();
            if(temp.getContinente().equals("nordamerica"))
                nordAmerica++;
            if(temp.getContinente().equals("sudamerica"))
                sudAmerica++;
            if(temp.getContinente().equals("europa"))
                europa++;
            if(temp.getContinente().equals("africa"))
                africa++;
            if(temp.getContinente().equals("asia"))
                asia++;
            if(temp.getContinente().equals("oceania"))
                oceania++;
        }
              
        /*
         * Verifica se il giocatore controlla interamente qualche
         * continente. Se il risultato  true, allora somma alle
         * armate bonus totali quelle del continente posseduto.
         */
        if(nordAmerica == ElencoTerritoriContinenti.numeroTerritoriNordAmerica())
            armate += ARMATE_NORDAMERICA;
        if(sudAmerica == ElencoTerritoriContinenti.numeroTerritoriSudAmerica())
            armate += ARMATE_SUDAMERICA;
        if(europa == ElencoTerritoriContinenti.numeroTerritoriEuropa())
            armate += ARMATE_EUROPA;
        if(asia == ElencoTerritoriContinenti.numeroTerritoriAsia())
            armate += ARMATE_ASIA;
        if(africa == ElencoTerritoriContinenti.numeroTerritoriAfrica())
    	    armate += ARMATE_AFRICA;
        if(oceania == ElencoTerritoriContinenti.numeroTerritoriOceania())
    	    armate += ARMATE_OCEANIA;
        
        return armate;											// Restituisce il numero di armate bonus
    }
    
    /**
     * Verifica la presenza di tre carte "cannone" all'interno
     * di una collezione di carte.
     * 
     * @param carte collezione di carte da analizzare
     * @return <code>true</code> se sono presenti 3 carte "cannone"
     */
    private static boolean trisCannoni(List carte) {
        if(Collections.frequency(carte, "cannone")>2)
            return true;
        return false;
    }
    /**
     * Verifica la presenza di tre carte "fante" all'interno
     * di una collezione di carte.
     * 
     * @param carte collezione di carte da analizzare
     * @return <code>true</code> se sono presenti 3 carte "fante"
     */
    private static boolean trisFanti(List carte) {
        if(Collections.frequency(carte, "fante")>2)
            return true;
        return false;
    }
    /**
     * Verifica la presenza di tre carte "cavallo" all'interno
     * di una collezione di carte.
     * 
     * @param carte collezione di carte da analizzare
     * @return <code>true</code> se sono presenti 3 carte "cavallo"
     */
    private static boolean trisCavalli(List carte) {
        if(Collections.frequency(carte, "cavallo")>2)
            return true;
        return false;
    }
    /**
     * Verifica la presenza di tre carte diverse ("jolly" escluso)
     * all'interno di una collezione di carte.
     * 
     * @param carte collezione di carte da analizzare
     * @return <code>true</code> se sono presenti 3 carte "fante"
     */
    private static boolean comboCarte(List carte) {
        if(carte.contains("cannone") && carte.contains("fante") && carte.contains("cavallo"))
            return true;
        return false;
    }
    /**
     * Verifica la presenza di due carte dello stesso valore
     * pi una carta jolly all'interno di una collezione
     * di carte.
     * 
     * @param carte collezione di carte da analizzare
     * @return <code>String</code> contenente il valore della coppia
     * di carte uguali che affiancano il jolly
     */
    private static String trisJolly(List carte) {
        if (carte.contains("jolly")) {
            if(Collections.frequency(carte, "cannone") > 1)
                return "cannone";
            if(Collections.frequency(carte,"fante") > 1)
                return "fante";
            if(Collections.frequency(carte,"cavallo") > 1)
                return "cavallo";
        }
        return null;
    }
    
    /**
     * Calcola un numero di armate bonus in base alle
     * combinazione di carte possedute dal giocatore.
     * Ogni combinazione convertita in armate prevede
     * anche il ritiro delle carte dal giocatore.
     * 
     * @param g riferimento al <code>Giocatore</code>
     * @return numero di armate bonus spettanti al
     * <code>Giocatore</code> in base alle carte possedute
     */
    public static int calcolaArmatePerCarte(Giocatore g) {
        int armate = 0;											// Numero armate bonus
        List carte = g.getCarte();								// Riferimento alle carte del giocatore
        boolean comboFinite = false;
        String coppia;											// Memorizza la coppia di carte uguali
    															// in un tris con il jolly
        /* Se il giocatore ha 3 o pi carte
         * verifica la presenza di combinazioni
         * fra esse.
         */
        //if(carte.size()>2) {
        while(carte.size() > 2 && !comboFinite) {
            if(trisJolly(carte) != null) {						// Se ve ne sono due uguali pi un jolly
                armate += ElencoCarte.armateTrisJolly();		// assegnagli tante armate quante gli spettano                
                g.rimuoviCarteStessoValore(trisJolly(carte), 2);// prendi le carte dal giocatore e scartale
                g.rimuoviCarta("jolly");						// rimuovi il jolly
            } else
            if(comboCarte(carte)) {								// Se invece ha 3 carte diverse
                armate += ElencoCarte.armateCombo();			// dagli le giuste armate
                g.rimuoviCarta("cavallo");						// e scarta le carte usate nella combinazione
                g.rimuoviCarta("fante");
                g.rimuoviCarta("cannone");
            } else
            if(trisCavalli(carte)) {							// Lo stesso discorso vale anche per i tris
                armate += ElencoCarte.armateTrisCavalli();		// ...armate tris di cavalli
                g.rimuoviCarteStessoValore("cavallo",3);                
            } else
            if(trisFanti(carte)) {
                armate += ElencoCarte.armateTrisFanti();		//...armate tris di fanti
                g.rimuoviCarteStessoValore("fante",3);
            } else
            if(trisCannoni(carte)) {
                armate += ElencoCarte.armateTrisCannoni();
                g.rimuoviCarteStessoValore("cannone",3);		// ...armate tris di cannoni
            } else
                comboFinite = true;
            carte = g.getCarte();
        }
        
        return armate;											// Restituisci il numero di armate bonus
    }
    
    /**
     * Calcola il numero totale di territori posseduti dal
     * giocatore al quale distribuire le armate. Divide per
     * 3 il numero totale di territori posseduti da esso,
     * e a questo numero aggiunge anche il numero di armate
     * bonus spettanti al giocatore dall'eventuale controllo
     * di interi continenti. 
     * 
     * @param giocatore <code>Giocatore</code> al quale distribuire le
     * armate
     * @return il numero di armate spettanti al giocatori
     * in base ai territori posseduti pi le armate bonus
     * per i continenti.
     */
    public static int daiArmate(Giocatore giocatore) {
        return (
                (giocatore.getTerritori().size() / ARMATE_OGNI_TERRITORI) +
                calcolaArmatePerContinente(giocatore) +
                calcolaArmatePerCarte(giocatore)
                );
    }
}
