/*
 * @(#)IntegerType.java          1.0            6 September 1999
 *
 * This work is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * Copyright (c) 1999 Ericsson Telecom. All rights reserved.
 * Copyright (c) 2002 Per Cederberg. All rights reserved.
 */

package net.percederberg.mib.type;

/**
 * A class for handling integer type information.
 *
 * @version  1.0
 * @author   Per Cederberg, per@percederberg.net
 */
public class IntegerType extends Type {

   /**
    * A value range constraint, or null for any value.
    */
   private ValueRangeConstraint range;

   /**
    * Creates a new integer type without value restrictions.
    */
   public IntegerType() {
      this(null);
   }

   /**
    * Creates a new integer type with value range restrictions.
    *
    * @param  range     the value range
    */
   public IntegerType(ValueRangeConstraint range) {
      this.range = range;
   }

   /**
    * Checks if this type is equals to some other object. It will only
    * be reported equal to other integer types having the same type
    * constraints.
    *
    * @param  obj     an object
    *
    * @return true if the objects are equal, or
    *         false otherwise
    */
   public boolean equals(Object obj) {
      if (obj instanceof IntegerType) {
         IntegerType t = (IntegerType)obj;
         return equalValue(this.range, t.range);
      } else if (obj instanceof NamedType) {
         NamedType t = (NamedType)obj;
         return t.equals(this);
      } else {
         return false;
      }
   }

   /**
    * Checks if two values are equal, considering two null values to
    * be equal.
    *
    * @param  a       the first object
    * @param  b       the second object
    * @return true if the objects are equal, false otherwise
    */
   private boolean equalValue(Object a, Object b) {
      if (a == null) {
         return a == b;
      } else {
         return a.equals(b);
      }
   }

   /**
    * Returns a string description of this type.
    *
    * @return a string description of the type
    */
   public String toString() {
      if (range != null) {
         return "Integer [" + range + "]";
      } else {
         return "Integer";
      }
   }

   /**
    * Transfers the type information from this type to a type converter
    * object. The calling conventions declared in the TypeConverter
    * class are followed.
    *
    * @param   converter     a type converter
    */
   public void transferType(TypeConverter converter) {
      converter.createInteger();
      if (range != null) {
         range.transferConstraint(converter);
      }
   }

}

