/*
 * @(#)NamedType.java           1.1             25 October 1999
 *
 * This work is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * Copyright (c) 1999 Ericsson Telecom. All rights reserved.
 * Copyright (c) 2002 Per Cederberg. All rights reserved.
 */

package net.percederberg.mib.type;

import net.percederberg.mib.symbol.TypeSymbol;

/**
 * A class for handling type information for named types.
 *
 * @version  1.1
 * @author   Per Cederberg, per@percederberg.net
 */
public class NamedType extends Type {

   /**
    * The type symbol containing the type information.
    */
   private TypeSymbol  sym;

   /**
    * The additional type constraint information.
    */
   private Constraint  constraint;

   /**
    * Creates a new named type based on the given type symbol.
    *
    * @param  sym     a type symbol
    */
   public NamedType(TypeSymbol sym) {
      this(sym, null);
   }

   /**
    * Creates a new named type based on the given type symbol.
    *
    * @param  sym        a type symbol
    * @param constraint  the named type constraint
    */
   public NamedType(TypeSymbol sym, Constraint constraint) {
      this.sym = sym;
      this.constraint = constraint;
   }

   /**
    * Checks if this types equals any other object.
    *
    * @param  obj     an object
    *
    * @return true if the objects are equal, or
    *         false otherwise
    */
   public boolean equals(Object obj) {
      if (constraint == null) {
         return sym.getType().equals(obj);
      } else {
         return this == obj;
      }
   }

   /**
    * Returns a string description of this type.
    *
    * @return a string description of the type
    */
   public String toString() {
      if (constraint == null) {
          return sym.getName();
      } else {
          return sym.getName() + " [" + constraint + "]";
      }
   }

   /**
    * Transfers the type information from this type to a type converter
    * object. The calling conventions declared in the TypeConverter
    * class are followed.
    *
    * @param   converter     a type converter
    */
   public void transferType(TypeConverter converter) {
      this.sym.getType().transferType(converter);
      converter.transferNameAlias(sym.getName());
      if (constraint != null) {
         constraint.transferConstraint(converter);
      }
   }

}

