/*
 * @(#)DepthFirstAdapter.java       1.2          9 August 1999
 *
 * This work is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * Copyright (c) 1999 Ericsson Telecom. All rights reserved.
 * Copyright (c) 2002 Per Cederberg. All rights reserved.
 */

package net.percederberg.mib.asn1.analysis;

import net.percederberg.mib.asn1.node.Node;

/**
 * This class provides a depth first tree traversal implementation for
 * the parse tree. This class should be subclassed and the appropriate
 * inX and outX methods overridden for handling various types of
 * parse tree node analysis. All the case methods are declared final
 * to avoid overriding them.
 *
 * @version  1.2
 * @author   Per Cederberg, per@percederberg.net
 */
public class DepthFirstAdapter extends TreeWalker implements Switch {

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inStart(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outStart(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseStart(Node node) {
        inStart(node);
        defaultCase(node);
        outStart(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inModuleDefinition(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outModuleDefinition(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseModuleDefinition(Node node) {
        inModuleDefinition(node);
        defaultCase(node);
        outModuleDefinition(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTagDefault(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTagDefault(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTagDefault(Node node) {
        inTagDefault(node);
        defaultCase(node);
        outTagDefault(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inModuleIdentifier(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outModuleIdentifier(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseModuleIdentifier(Node node) {
        inModuleIdentifier(node);
        defaultCase(node);
        outModuleIdentifier(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inModuleBody(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outModuleBody(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseModuleBody(Node node) {
        inModuleBody(node);
        defaultCase(node);
        outModuleBody(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inExports(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outExports(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseExports(Node node) {
        inExports(node);
        defaultCase(node);
        outExports(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inImports(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outImports(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseImports(Node node) {
        inImports(node);
        defaultCase(node);
        outImports(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSymbolsFromModuleList(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSymbolsFromModuleList(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSymbolsFromModuleList(Node node) {
        inSymbolsFromModuleList(node);
        defaultCase(node);
        outSymbolsFromModuleList(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSymbolsFromModule(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSymbolsFromModule(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSymbolsFromModule(Node node) {
        inSymbolsFromModule(node);
        defaultCase(node);
        outSymbolsFromModule(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSymbolList(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSymbolList(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSymbolList(Node node) {
        inSymbolList(node);
        defaultCase(node);
        outSymbolList(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSymbol(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSymbol(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSymbol(Node node) {
        inSymbol(node);
        defaultCase(node);
        outSymbol(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inAssignmentList(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outAssignmentList(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseAssignmentList(Node node) {
        inAssignmentList(node);
        defaultCase(node);
        outAssignmentList(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inAssignment(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outAssignment(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseAssignment(Node node) {
        inAssignment(node);
        defaultCase(node);
        outAssignment(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inMacroDefinition(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outMacroDefinition(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseMacroDefinition(Node node) {
        inMacroDefinition(node);
        defaultCase(node);
        outMacroDefinition(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inMacroBody(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outMacroBody(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseMacroBody(Node node) {
        inMacroBody(node);
        defaultCase(node);
        outMacroBody(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSkipToEND(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSkipToEND(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSkipToEND(Node node) {
        inSkipToEND(node);
        defaultCase(node);
        outSkipToEND(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inMacroReference(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outMacroReference(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseMacroReference(Node node) {
        inMacroReference(node);
        defaultCase(node);
        outMacroReference(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTypeAssignment(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTypeAssignment(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTypeAssignment(Node node) {
        inTypeAssignment(node);
        defaultCase(node);
        outTypeAssignment(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inType(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outType(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseType(Node node) {
        inType(node);
        defaultCase(node);
        outType(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inBuiltinType(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outBuiltinType(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseBuiltinType(Node node) {
        inBuiltinType(node);
        defaultCase(node);
        outBuiltinType(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inIntegerType(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outIntegerType(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseIntegerType(Node node) {
        inIntegerType(node);
        defaultCase(node);
        outIntegerType(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inNamedNumberList(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outNamedNumberList(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseNamedNumberList(Node node) {
        inNamedNumberList(node);
        defaultCase(node);
        outNamedNumberList(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inNamedNumber(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outNamedNumber(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseNamedNumber(Node node) {
        inNamedNumber(node);
        defaultCase(node);
        outNamedNumber(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSignedNumber(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSignedNumber(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSignedNumber(Node node) {
        inSignedNumber(node);
        defaultCase(node);
        outSignedNumber(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inStringType(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outStringType(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseStringType(Node node) {
        inStringType(node);
        defaultCase(node);
        outStringType(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inBitStringType(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outBitStringType(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseBitStringType(Node node) {
        inBitStringType(node);
        defaultCase(node);
        outBitStringType(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSequenceType(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSequenceType(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSequenceType(Node node) {
        inSequenceType(node);
        defaultCase(node);
        outSequenceType(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSequenceOfType(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSequenceOfType(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSequenceOfType(Node node) {
        inSequenceOfType(node);
        defaultCase(node);
        outSequenceOfType(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSetType(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSetType(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSetType(Node node) {
        inSetType(node);
        defaultCase(node);
        outSetType(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSetOfType(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSetOfType(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSetOfType(Node node) {
        inSetOfType(node);
        defaultCase(node);
        outSetOfType(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inElementTypeList(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outElementTypeList(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseElementTypeList(Node node) {
        inElementTypeList(node);
        defaultCase(node);
        outElementTypeList(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inElementType(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outElementType(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseElementType(Node node) {
        inElementType(node);
        defaultCase(node);
        outElementType(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inNamedType(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outNamedType(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseNamedType(Node node) {
        inNamedType(node);
        defaultCase(node);
        outNamedType(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inChoiceType(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outChoiceType(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseChoiceType(Node node) {
        inChoiceType(node);
        defaultCase(node);
        outChoiceType(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inEnumeratedType(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outEnumeratedType(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseEnumeratedType(Node node) {
        inEnumeratedType(node);
        defaultCase(node);
        outEnumeratedType(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSelectionType(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSelectionType(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSelectionType(Node node) {
        inSelectionType(node);
        defaultCase(node);
        outSelectionType(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTaggedType(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTaggedType(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTaggedType(Node node) {
        inTaggedType(node);
        defaultCase(node);
        outTaggedType(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTag(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTag(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTag(Node node) {
        inTag(node);
        defaultCase(node);
        outTag(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inClassNumber(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outClassNumber(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseClassNumber(Node node) {
        inClassNumber(node);
        defaultCase(node);
        outClassNumber(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inClass(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outClass(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseClass(Node node) {
        inClass(node);
        defaultCase(node);
        outClass(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inAnyType(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outAnyType(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseAnyType(Node node) {
        inAnyType(node);
        defaultCase(node);
        outAnyType(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inDefinedType(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outDefinedType(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseDefinedType(Node node) {
        inDefinedType(node);
        defaultCase(node);
        outDefinedType(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inConstraintList(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outConstraintList(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseConstraintList(Node node) {
        inConstraintList(node);
        defaultCase(node);
        outConstraintList(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inConstraint(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outConstraint(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseConstraint(Node node) {
        inConstraint(node);
        defaultCase(node);
        outConstraint(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inValueConstraint(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outValueConstraint(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseValueConstraint(Node node) {
        inValueConstraint(node);
        defaultCase(node);
        outValueConstraint(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inValueRange(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outValueRange(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseValueRange(Node node) {
        inValueRange(node);
        defaultCase(node);
        outValueRange(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inLowerEndPoint(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outLowerEndPoint(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseLowerEndPoint(Node node) {
        inLowerEndPoint(node);
        defaultCase(node);
        outLowerEndPoint(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inUpperEndPoint(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outUpperEndPoint(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseUpperEndPoint(Node node) {
        inUpperEndPoint(node);
        defaultCase(node);
        outUpperEndPoint(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSizeConstraint(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSizeConstraint(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSizeConstraint(Node node) {
        inSizeConstraint(node);
        defaultCase(node);
        outSizeConstraint(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inAlphabetConstraint(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outAlphabetConstraint(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseAlphabetConstraint(Node node) {
        inAlphabetConstraint(node);
        defaultCase(node);
        outAlphabetConstraint(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inValueAssignment(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outValueAssignment(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseValueAssignment(Node node) {
        inValueAssignment(node);
        defaultCase(node);
        outValueAssignment(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inValue(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outValue(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseValue(Node node) {
        inValue(node);
        defaultCase(node);
        outValue(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inDefinedValue(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outDefinedValue(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseDefinedValue(Node node) {
        inDefinedValue(node);
        defaultCase(node);
        outDefinedValue(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inBuiltinValue(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outBuiltinValue(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseBuiltinValue(Node node) {
        inBuiltinValue(node);
        defaultCase(node);
        outBuiltinValue(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inBooleanValue(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outBooleanValue(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseBooleanValue(Node node) {
        inBooleanValue(node);
        defaultCase(node);
        outBooleanValue(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSpecialRealValue(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSpecialRealValue(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSpecialRealValue(Node node) {
        inSpecialRealValue(node);
        defaultCase(node);
        outSpecialRealValue(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inNullValue(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outNullValue(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseNullValue(Node node) {
        inNullValue(node);
        defaultCase(node);
        outNullValue(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inNamedValue(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outNamedValue(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseNamedValue(Node node) {
        inNamedValue(node);
        defaultCase(node);
        outNamedValue(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inObjectIdentifierValue(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outObjectIdentifierValue(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseObjectIdentifierValue(Node node) {
        inObjectIdentifierValue(node);
        defaultCase(node);
        outObjectIdentifierValue(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inObjIdComponentList(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outObjIdComponentList(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseObjIdComponentList(Node node) {
        inObjIdComponentList(node);
        defaultCase(node);
        outObjIdComponentList(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inObjIdComponent(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outObjIdComponent(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseObjIdComponent(Node node) {
        inObjIdComponent(node);
        defaultCase(node);
        outObjIdComponent(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inNameAndNumberForm(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outNameAndNumberForm(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseNameAndNumberForm(Node node) {
        inNameAndNumberForm(node);
        defaultCase(node);
        outNameAndNumberForm(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inBinaryString(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outBinaryString(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseBinaryString(Node node) {
        inBinaryString(node);
        defaultCase(node);
        outBinaryString(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inHexString(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outHexString(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseHexString(Node node) {
        inHexString(node);
        defaultCase(node);
        outHexString(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inCharString(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outCharString(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseCharString(Node node) {
        inCharString(node);
        defaultCase(node);
        outCharString(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inNumber(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outNumber(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseNumber(Node node) {
        inNumber(node);
        defaultCase(node);
        outNumber(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inIdentifier(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outIdentifier(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseIdentifier(Node node) {
        inIdentifier(node);
        defaultCase(node);
        outIdentifier(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inModuleReference(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outModuleReference(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseModuleReference(Node node) {
        inModuleReference(node);
        defaultCase(node);
        outModuleReference(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTypeReference(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTypeReference(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTypeReference(Node node) {
        inTypeReference(node);
        defaultCase(node);
        outTypeReference(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inDefinedMacroType(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outDefinedMacroType(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseDefinedMacroType(Node node) {
        inDefinedMacroType(node);
        defaultCase(node);
        outDefinedMacroType(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inDefinedMacroName(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outDefinedMacroName(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseDefinedMacroName(Node node) {
        inDefinedMacroName(node);
        defaultCase(node);
        outDefinedMacroName(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSnmpModuleIdentityMacroType(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSnmpModuleIdentityMacroType(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSnmpModuleIdentityMacroType(Node node) {
        inSnmpModuleIdentityMacroType(node);
        defaultCase(node);
        outSnmpModuleIdentityMacroType(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSnmpObjectIdentityMacroType(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSnmpObjectIdentityMacroType(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSnmpObjectIdentityMacroType(Node node) {
        inSnmpObjectIdentityMacroType(node);
        defaultCase(node);
        outSnmpObjectIdentityMacroType(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSnmpObjectTypeMacroType(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSnmpObjectTypeMacroType(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSnmpObjectTypeMacroType(Node node) {
        inSnmpObjectTypeMacroType(node);
        defaultCase(node);
        outSnmpObjectTypeMacroType(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSnmpNotificationTypeMacroType(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSnmpNotificationTypeMacroType(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSnmpNotificationTypeMacroType(Node node) {
        inSnmpNotificationTypeMacroType(node);
        defaultCase(node);
        outSnmpNotificationTypeMacroType(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSnmpTrapTypeMacroType(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSnmpTrapTypeMacroType(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSnmpTrapTypeMacroType(Node node) {
        inSnmpTrapTypeMacroType(node);
        defaultCase(node);
        outSnmpTrapTypeMacroType(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSnmpTextualConventionMacroType(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSnmpTextualConventionMacroType(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSnmpTextualConventionMacroType(Node node) {
        inSnmpTextualConventionMacroType(node);
        defaultCase(node);
        outSnmpTextualConventionMacroType(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSnmpObjectGroupMacroType(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSnmpObjectGroupMacroType(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSnmpObjectGroupMacroType(Node node) {
        inSnmpObjectGroupMacroType(node);
        defaultCase(node);
        outSnmpObjectGroupMacroType(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSnmpNotificationGroupMacroType(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSnmpNotificationGroupMacroType(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSnmpNotificationGroupMacroType(Node node) {
        inSnmpNotificationGroupMacroType(node);
        defaultCase(node);
        outSnmpNotificationGroupMacroType(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSnmpModuleComplianceMacroType(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSnmpModuleComplianceMacroType(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSnmpModuleComplianceMacroType(Node node) {
        inSnmpModuleComplianceMacroType(node);
        defaultCase(node);
        outSnmpModuleComplianceMacroType(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSnmpAgentCapabilitiesMacroType(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSnmpAgentCapabilitiesMacroType(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSnmpAgentCapabilitiesMacroType(Node node) {
        inSnmpAgentCapabilitiesMacroType(node);
        defaultCase(node);
        outSnmpAgentCapabilitiesMacroType(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSnmpUpdatePart(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSnmpUpdatePart(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSnmpUpdatePart(Node node) {
        inSnmpUpdatePart(node);
        defaultCase(node);
        outSnmpUpdatePart(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSnmpOrganizationPart(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSnmpOrganizationPart(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSnmpOrganizationPart(Node node) {
        inSnmpOrganizationPart(node);
        defaultCase(node);
        outSnmpOrganizationPart(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSnmpContactPart(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSnmpContactPart(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSnmpContactPart(Node node) {
        inSnmpContactPart(node);
        defaultCase(node);
        outSnmpContactPart(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSnmpDescrPart(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSnmpDescrPart(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSnmpDescrPart(Node node) {
        inSnmpDescrPart(node);
        defaultCase(node);
        outSnmpDescrPart(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSnmpRevisionPart(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSnmpRevisionPart(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSnmpRevisionPart(Node node) {
        inSnmpRevisionPart(node);
        defaultCase(node);
        outSnmpRevisionPart(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSnmpStatusPart(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSnmpStatusPart(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSnmpStatusPart(Node node) {
        inSnmpStatusPart(node);
        defaultCase(node);
        outSnmpStatusPart(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSnmpReferPart(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSnmpReferPart(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSnmpReferPart(Node node) {
        inSnmpReferPart(node);
        defaultCase(node);
        outSnmpReferPart(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSnmpSyntaxPart(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSnmpSyntaxPart(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSnmpSyntaxPart(Node node) {
        inSnmpSyntaxPart(node);
        defaultCase(node);
        outSnmpSyntaxPart(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTypeOrBits(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTypeOrBits(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTypeOrBits(Node node) {
        inTypeOrBits(node);
        defaultCase(node);
        outTypeOrBits(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inNamedBits(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outNamedBits(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseNamedBits(Node node) {
        inNamedBits(node);
        defaultCase(node);
        outNamedBits(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inNamedBit(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outNamedBit(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseNamedBit(Node node) {
        inNamedBit(node);
        defaultCase(node);
        outNamedBit(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSnmpUnitsPart(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSnmpUnitsPart(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSnmpUnitsPart(Node node) {
        inSnmpUnitsPart(node);
        defaultCase(node);
        outSnmpUnitsPart(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSnmpAccessPart(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSnmpAccessPart(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSnmpAccessPart(Node node) {
        inSnmpAccessPart(node);
        defaultCase(node);
        outSnmpAccessPart(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSnmpIndexPart(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSnmpIndexPart(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSnmpIndexPart(Node node) {
        inSnmpIndexPart(node);
        defaultCase(node);
        outSnmpIndexPart(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTypeOrValueList(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTypeOrValueList(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTypeOrValueList(Node node) {
        inTypeOrValueList(node);
        defaultCase(node);
        outTypeOrValueList(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTypeOrValue(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTypeOrValue(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTypeOrValue(Node node) {
        inTypeOrValue(node);
        defaultCase(node);
        outTypeOrValue(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSnmpDefValPart(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSnmpDefValPart(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSnmpDefValPart(Node node) {
        inSnmpDefValPart(node);
        defaultCase(node);
        outSnmpDefValPart(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSnmpObjectsPart(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSnmpObjectsPart(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSnmpObjectsPart(Node node) {
        inSnmpObjectsPart(node);
        defaultCase(node);
        outSnmpObjectsPart(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inValueList(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outValueList(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseValueList(Node node) {
        inValueList(node);
        defaultCase(node);
        outValueList(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSnmpEnterprisePart(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSnmpEnterprisePart(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSnmpEnterprisePart(Node node) {
        inSnmpEnterprisePart(node);
        defaultCase(node);
        outSnmpEnterprisePart(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSnmpVarPart(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSnmpVarPart(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSnmpVarPart(Node node) {
        inSnmpVarPart(node);
        defaultCase(node);
        outSnmpVarPart(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inVarTypes(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outVarTypes(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseVarTypes(Node node) {
        inVarTypes(node);
        defaultCase(node);
        outVarTypes(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSnmpDisplayPart(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSnmpDisplayPart(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSnmpDisplayPart(Node node) {
        inSnmpDisplayPart(node);
        defaultCase(node);
        outSnmpDisplayPart(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSnmpNotificationsPart(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSnmpNotificationsPart(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSnmpNotificationsPart(Node node) {
        inSnmpNotificationsPart(node);
        defaultCase(node);
        outSnmpNotificationsPart(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSnmpModulePart(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSnmpModulePart(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSnmpModulePart(Node node) {
        inSnmpModulePart(node);
        defaultCase(node);
        outSnmpModulePart(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSnmpMandatoryPart(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSnmpMandatoryPart(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSnmpMandatoryPart(Node node) {
        inSnmpMandatoryPart(node);
        defaultCase(node);
        outSnmpMandatoryPart(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSnmpCompliancePart(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSnmpCompliancePart(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSnmpCompliancePart(Node node) {
        inSnmpCompliancePart(node);
        defaultCase(node);
        outSnmpCompliancePart(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inComplianceGroup(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outComplianceGroup(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseComplianceGroup(Node node) {
        inComplianceGroup(node);
        defaultCase(node);
        outComplianceGroup(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inComplianceObject(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outComplianceObject(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseComplianceObject(Node node) {
        inComplianceObject(node);
        defaultCase(node);
        outComplianceObject(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSnmpWriteSyntaxPart(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSnmpWriteSyntaxPart(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSnmpWriteSyntaxPart(Node node) {
        inSnmpWriteSyntaxPart(node);
        defaultCase(node);
        outSnmpWriteSyntaxPart(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSnmpProductReleasePart(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSnmpProductReleasePart(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSnmpProductReleasePart(Node node) {
        inSnmpProductReleasePart(node);
        defaultCase(node);
        outSnmpProductReleasePart(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSnmpModuleSupportPart(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSnmpModuleSupportPart(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSnmpModuleSupportPart(Node node) {
        inSnmpModuleSupportPart(node);
        defaultCase(node);
        outSnmpModuleSupportPart(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSnmpVariationPart(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSnmpVariationPart(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSnmpVariationPart(Node node) {
        inSnmpVariationPart(node);
        defaultCase(node);
        outSnmpVariationPart(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inSnmpCreationPart(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outSnmpCreationPart(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseSnmpCreationPart(Node node) {
        inSnmpCreationPart(node);
        defaultCase(node);
        outSnmpCreationPart(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inEOF(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outEOF(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseEOF(Node node) {
        inEOF(node);
        defaultCase(node);
        outEOF(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTDOT(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTDOT(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTDOT(Node node) {
        inTDOT(node);
        defaultCase(node);
        outTDOT(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTCOMMA(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTCOMMA(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTCOMMA(Node node) {
        inTCOMMA(node);
        defaultCase(node);
        outTCOMMA(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTSEMI_COLON(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTSEMI_COLON(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTSEMI_COLON(Node node) {
        inTSEMI_COLON(node);
        defaultCase(node);
        outTSEMI_COLON(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTLEFTPAREN(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTLEFTPAREN(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTLEFTPAREN(Node node) {
        inTLEFTPAREN(node);
        defaultCase(node);
        outTLEFTPAREN(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTRIGHTPAREN(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTRIGHTPAREN(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTRIGHTPAREN(Node node) {
        inTRIGHTPAREN(node);
        defaultCase(node);
        outTRIGHTPAREN(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTLEFTBRACE(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTLEFTBRACE(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTLEFTBRACE(Node node) {
        inTLEFTBRACE(node);
        defaultCase(node);
        outTLEFTBRACE(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTRIGHTBRACE(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTRIGHTBRACE(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTRIGHTBRACE(Node node) {
        inTRIGHTBRACE(node);
        defaultCase(node);
        outTRIGHTBRACE(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTLEFTBRACKET(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTLEFTBRACKET(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTLEFTBRACKET(Node node) {
        inTLEFTBRACKET(node);
        defaultCase(node);
        outTLEFTBRACKET(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTRIGHTBRACKET(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTRIGHTBRACKET(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTRIGHTBRACKET(Node node) {
        inTRIGHTBRACKET(node);
        defaultCase(node);
        outTRIGHTBRACKET(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTMINUS(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTMINUS(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTMINUS(Node node) {
        inTMINUS(node);
        defaultCase(node);
        outTMINUS(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTLESSTHAN(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTLESSTHAN(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTLESSTHAN(Node node) {
        inTLESSTHAN(node);
        defaultCase(node);
        outTLESSTHAN(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTBAR(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTBAR(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTBAR(Node node) {
        inTBAR(node);
        defaultCase(node);
        outTBAR(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTASSIGN(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTASSIGN(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTASSIGN(Node node) {
        inTASSIGN(node);
        defaultCase(node);
        outTASSIGN(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTDEFINITIONS(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTDEFINITIONS(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTDEFINITIONS(Node node) {
        inTDEFINITIONS(node);
        defaultCase(node);
        outTDEFINITIONS(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTEXPLICIT(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTEXPLICIT(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTEXPLICIT(Node node) {
        inTEXPLICIT(node);
        defaultCase(node);
        outTEXPLICIT(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTIMPLICIT(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTIMPLICIT(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTIMPLICIT(Node node) {
        inTIMPLICIT(node);
        defaultCase(node);
        outTIMPLICIT(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTTAGS(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTTAGS(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTTAGS(Node node) {
        inTTAGS(node);
        defaultCase(node);
        outTTAGS(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTBEGIN(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTBEGIN(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTBEGIN(Node node) {
        inTBEGIN(node);
        defaultCase(node);
        outTBEGIN(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTEND(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTEND(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTEND(Node node) {
        inTEND(node);
        defaultCase(node);
        outTEND(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTEXPORTS(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTEXPORTS(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTEXPORTS(Node node) {
        inTEXPORTS(node);
        defaultCase(node);
        outTEXPORTS(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTIMPORTS(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTIMPORTS(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTIMPORTS(Node node) {
        inTIMPORTS(node);
        defaultCase(node);
        outTIMPORTS(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTFROM(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTFROM(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTFROM(Node node) {
        inTFROM(node);
        defaultCase(node);
        outTFROM(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTMACRO(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTMACRO(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTMACRO(Node node) {
        inTMACRO(node);
        defaultCase(node);
        outTMACRO(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTINTEGER(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTINTEGER(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTINTEGER(Node node) {
        inTINTEGER(node);
        defaultCase(node);
        outTINTEGER(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTREAL(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTREAL(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTREAL(Node node) {
        inTREAL(node);
        defaultCase(node);
        outTREAL(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTBOOLEAN(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTBOOLEAN(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTBOOLEAN(Node node) {
        inTBOOLEAN(node);
        defaultCase(node);
        outTBOOLEAN(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTNULL(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTNULL(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTNULL(Node node) {
        inTNULL(node);
        defaultCase(node);
        outTNULL(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTBIT(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTBIT(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTBIT(Node node) {
        inTBIT(node);
        defaultCase(node);
        outTBIT(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTOCTET(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTOCTET(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTOCTET(Node node) {
        inTOCTET(node);
        defaultCase(node);
        outTOCTET(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTSTRING(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTSTRING(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTSTRING(Node node) {
        inTSTRING(node);
        defaultCase(node);
        outTSTRING(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTENUMERATED(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTENUMERATED(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTENUMERATED(Node node) {
        inTENUMERATED(node);
        defaultCase(node);
        outTENUMERATED(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTSEQUENCE(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTSEQUENCE(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTSEQUENCE(Node node) {
        inTSEQUENCE(node);
        defaultCase(node);
        outTSEQUENCE(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTSET(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTSET(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTSET(Node node) {
        inTSET(node);
        defaultCase(node);
        outTSET(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTOF(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTOF(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTOF(Node node) {
        inTOF(node);
        defaultCase(node);
        outTOF(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTCHOICE(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTCHOICE(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTCHOICE(Node node) {
        inTCHOICE(node);
        defaultCase(node);
        outTCHOICE(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTUNIVERSAL(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTUNIVERSAL(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTUNIVERSAL(Node node) {
        inTUNIVERSAL(node);
        defaultCase(node);
        outTUNIVERSAL(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTAPPLICATION(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTAPPLICATION(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTAPPLICATION(Node node) {
        inTAPPLICATION(node);
        defaultCase(node);
        outTAPPLICATION(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTPRIVATE(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTPRIVATE(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTPRIVATE(Node node) {
        inTPRIVATE(node);
        defaultCase(node);
        outTPRIVATE(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTANY(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTANY(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTANY(Node node) {
        inTANY(node);
        defaultCase(node);
        outTANY(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTDEFINED(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTDEFINED(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTDEFINED(Node node) {
        inTDEFINED(node);
        defaultCase(node);
        outTDEFINED(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTBY(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTBY(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTBY(Node node) {
        inTBY(node);
        defaultCase(node);
        outTBY(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTOBJECT(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTOBJECT(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTOBJECT(Node node) {
        inTOBJECT(node);
        defaultCase(node);
        outTOBJECT(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTIDENTIFIER(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTIDENTIFIER(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTIDENTIFIER(Node node) {
        inTIDENTIFIER(node);
        defaultCase(node);
        outTIDENTIFIER(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTINCLUDES(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTINCLUDES(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTINCLUDES(Node node) {
        inTINCLUDES(node);
        defaultCase(node);
        outTINCLUDES(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTMIN(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTMIN(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTMIN(Node node) {
        inTMIN(node);
        defaultCase(node);
        outTMIN(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTMAX(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTMAX(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTMAX(Node node) {
        inTMAX(node);
        defaultCase(node);
        outTMAX(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTSIZE(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTSIZE(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTSIZE(Node node) {
        inTSIZE(node);
        defaultCase(node);
        outTSIZE(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTWITH(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTWITH(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTWITH(Node node) {
        inTWITH(node);
        defaultCase(node);
        outTWITH(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTCOMPONENT(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTCOMPONENT(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTCOMPONENT(Node node) {
        inTCOMPONENT(node);
        defaultCase(node);
        outTCOMPONENT(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTCOMPONENTS(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTCOMPONENTS(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTCOMPONENTS(Node node) {
        inTCOMPONENTS(node);
        defaultCase(node);
        outTCOMPONENTS(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTPRESENT(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTPRESENT(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTPRESENT(Node node) {
        inTPRESENT(node);
        defaultCase(node);
        outTPRESENT(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTABSENT(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTABSENT(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTABSENT(Node node) {
        inTABSENT(node);
        defaultCase(node);
        outTABSENT(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTOPTIONAL(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTOPTIONAL(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTOPTIONAL(Node node) {
        inTOPTIONAL(node);
        defaultCase(node);
        outTOPTIONAL(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTDEFAULT(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTDEFAULT(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTDEFAULT(Node node) {
        inTDEFAULT(node);
        defaultCase(node);
        outTDEFAULT(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTTRUE(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTTRUE(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTTRUE(Node node) {
        inTTRUE(node);
        defaultCase(node);
        outTTRUE(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTFALSE(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTFALSE(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTFALSE(Node node) {
        inTFALSE(node);
        defaultCase(node);
        outTFALSE(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTPLUS_INFINITY(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTPLUS_INFINITY(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTPLUS_INFINITY(Node node) {
        inTPLUS_INFINITY(node);
        defaultCase(node);
        outTPLUS_INFINITY(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTMINUS_INFINITY(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTMINUS_INFINITY(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTMINUS_INFINITY(Node node) {
        inTMINUS_INFINITY(node);
        defaultCase(node);
        outTMINUS_INFINITY(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTMODULE_IDENTITY(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTMODULE_IDENTITY(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTMODULE_IDENTITY(Node node) {
        inTMODULE_IDENTITY(node);
        defaultCase(node);
        outTMODULE_IDENTITY(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTOBJECT_IDENTITY(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTOBJECT_IDENTITY(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTOBJECT_IDENTITY(Node node) {
        inTOBJECT_IDENTITY(node);
        defaultCase(node);
        outTOBJECT_IDENTITY(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTOBJECT_TYPE(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTOBJECT_TYPE(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTOBJECT_TYPE(Node node) {
        inTOBJECT_TYPE(node);
        defaultCase(node);
        outTOBJECT_TYPE(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTNOTIFICATION_TYPE(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTNOTIFICATION_TYPE(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTNOTIFICATION_TYPE(Node node) {
        inTNOTIFICATION_TYPE(node);
        defaultCase(node);
        outTNOTIFICATION_TYPE(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTTRAP_TYPE(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTTRAP_TYPE(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTTRAP_TYPE(Node node) {
        inTTRAP_TYPE(node);
        defaultCase(node);
        outTTRAP_TYPE(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTTEXTUAL_CONVENTION(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTTEXTUAL_CONVENTION(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTTEXTUAL_CONVENTION(Node node) {
        inTTEXTUAL_CONVENTION(node);
        defaultCase(node);
        outTTEXTUAL_CONVENTION(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTOBJECT_GROUP(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTOBJECT_GROUP(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTOBJECT_GROUP(Node node) {
        inTOBJECT_GROUP(node);
        defaultCase(node);
        outTOBJECT_GROUP(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTNOTIFICATION_GROUP(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTNOTIFICATION_GROUP(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTNOTIFICATION_GROUP(Node node) {
        inTNOTIFICATION_GROUP(node);
        defaultCase(node);
        outTNOTIFICATION_GROUP(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTMODULE_COMPLIANCE(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTMODULE_COMPLIANCE(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTMODULE_COMPLIANCE(Node node) {
        inTMODULE_COMPLIANCE(node);
        defaultCase(node);
        outTMODULE_COMPLIANCE(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTAGENT_CAPABILITIES(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTAGENT_CAPABILITIES(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTAGENT_CAPABILITIES(Node node) {
        inTAGENT_CAPABILITIES(node);
        defaultCase(node);
        outTAGENT_CAPABILITIES(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTLAST_UPDATED(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTLAST_UPDATED(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTLAST_UPDATED(Node node) {
        inTLAST_UPDATED(node);
        defaultCase(node);
        outTLAST_UPDATED(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTORGANIZATION(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTORGANIZATION(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTORGANIZATION(Node node) {
        inTORGANIZATION(node);
        defaultCase(node);
        outTORGANIZATION(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTCONTACT_INFO(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTCONTACT_INFO(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTCONTACT_INFO(Node node) {
        inTCONTACT_INFO(node);
        defaultCase(node);
        outTCONTACT_INFO(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTDESCRIPTION(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTDESCRIPTION(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTDESCRIPTION(Node node) {
        inTDESCRIPTION(node);
        defaultCase(node);
        outTDESCRIPTION(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTREVISION(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTREVISION(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTREVISION(Node node) {
        inTREVISION(node);
        defaultCase(node);
        outTREVISION(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTSTATUS(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTSTATUS(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTSTATUS(Node node) {
        inTSTATUS(node);
        defaultCase(node);
        outTSTATUS(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTREFERENCE(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTREFERENCE(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTREFERENCE(Node node) {
        inTREFERENCE(node);
        defaultCase(node);
        outTREFERENCE(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTSYNTAX(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTSYNTAX(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTSYNTAX(Node node) {
        inTSYNTAX(node);
        defaultCase(node);
        outTSYNTAX(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTBITS(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTBITS(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTBITS(Node node) {
        inTBITS(node);
        defaultCase(node);
        outTBITS(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTUNITS(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTUNITS(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTUNITS(Node node) {
        inTUNITS(node);
        defaultCase(node);
        outTUNITS(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTACCESS(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTACCESS(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTACCESS(Node node) {
        inTACCESS(node);
        defaultCase(node);
        outTACCESS(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTMAX_ACCESS(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTMAX_ACCESS(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTMAX_ACCESS(Node node) {
        inTMAX_ACCESS(node);
        defaultCase(node);
        outTMAX_ACCESS(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTMIN_ACCESS(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTMIN_ACCESS(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTMIN_ACCESS(Node node) {
        inTMIN_ACCESS(node);
        defaultCase(node);
        outTMIN_ACCESS(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTINDEX(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTINDEX(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTINDEX(Node node) {
        inTINDEX(node);
        defaultCase(node);
        outTINDEX(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTAUGMENTS(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTAUGMENTS(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTAUGMENTS(Node node) {
        inTAUGMENTS(node);
        defaultCase(node);
        outTAUGMENTS(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTIMPLIED(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTIMPLIED(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTIMPLIED(Node node) {
        inTIMPLIED(node);
        defaultCase(node);
        outTIMPLIED(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTDEFVAL(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTDEFVAL(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTDEFVAL(Node node) {
        inTDEFVAL(node);
        defaultCase(node);
        outTDEFVAL(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTOBJECTS(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTOBJECTS(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTOBJECTS(Node node) {
        inTOBJECTS(node);
        defaultCase(node);
        outTOBJECTS(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTENTERPRISE(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTENTERPRISE(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTENTERPRISE(Node node) {
        inTENTERPRISE(node);
        defaultCase(node);
        outTENTERPRISE(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTVARIABLES(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTVARIABLES(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTVARIABLES(Node node) {
        inTVARIABLES(node);
        defaultCase(node);
        outTVARIABLES(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTDISPLAY_HINT(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTDISPLAY_HINT(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTDISPLAY_HINT(Node node) {
        inTDISPLAY_HINT(node);
        defaultCase(node);
        outTDISPLAY_HINT(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTNOTIFICATIONS(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTNOTIFICATIONS(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTNOTIFICATIONS(Node node) {
        inTNOTIFICATIONS(node);
        defaultCase(node);
        outTNOTIFICATIONS(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTMODULE(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTMODULE(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTMODULE(Node node) {
        inTMODULE(node);
        defaultCase(node);
        outTMODULE(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTMANDATORY_GROUPS(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTMANDATORY_GROUPS(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTMANDATORY_GROUPS(Node node) {
        inTMANDATORY_GROUPS(node);
        defaultCase(node);
        outTMANDATORY_GROUPS(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTGROUP(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTGROUP(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTGROUP(Node node) {
        inTGROUP(node);
        defaultCase(node);
        outTGROUP(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTWRITE_SYNTAX(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTWRITE_SYNTAX(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTWRITE_SYNTAX(Node node) {
        inTWRITE_SYNTAX(node);
        defaultCase(node);
        outTWRITE_SYNTAX(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTPRODUCT_RELEASE(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTPRODUCT_RELEASE(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTPRODUCT_RELEASE(Node node) {
        inTPRODUCT_RELEASE(node);
        defaultCase(node);
        outTPRODUCT_RELEASE(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTSUPPORTS(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTSUPPORTS(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTSUPPORTS(Node node) {
        inTSUPPORTS(node);
        defaultCase(node);
        outTSUPPORTS(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTVARIATION(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTVARIATION(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTVARIATION(Node node) {
        inTVARIATION(node);
        defaultCase(node);
        outTVARIATION(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTCREATION_REQUIRES(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTCREATION_REQUIRES(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTCREATION_REQUIRES(Node node) {
        inTCREATION_REQUIRES(node);
        defaultCase(node);
        outTCREATION_REQUIRES(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTBINSTRING(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTBINSTRING(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTBINSTRING(Node node) {
        inTBINSTRING(node);
        defaultCase(node);
        outTBINSTRING(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTHEXSTRING(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTHEXSTRING(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTHEXSTRING(Node node) {
        inTHEXSTRING(node);
        defaultCase(node);
        outTHEXSTRING(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTCSTRING(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTCSTRING(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTCSTRING(Node node) {
        inTCSTRING(node);
        defaultCase(node);
        outTCSTRING(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTIDENT(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTIDENT(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTIDENT(Node node) {
        inTIDENT(node);
        defaultCase(node);
        outTIDENT(node);
    }

    /**
     * Called when entering a node of this type. By default, this
     * method calls the defaultIn() method.
     *
     * @param node   a parse tree node
     */
    protected void inTNUMBER(Node node) {
        defaultIn(node);
    }

    /**
     * Called when exiting a node of this type. By default, this
     * method calls the defaultOut() method.
     *
     * @param node   a parse tree node
     */
    protected void outTNUMBER(Node node) {
        defaultOut(node);
    }

    /**
     * Depth-first traverses this node. This method calls other
     * methods as appropriate.
     *
     * @param node    a parse tree node
     */
    public final void caseTNUMBER(Node node) {
        inTNUMBER(node);
        defaultCase(node);
        outTNUMBER(node);
    }

    /**
     * A default method called from all inX methods. Override
     * this method in a subclass to handle the default case.
     *
     * @param  node   a parse tree node
     */
    protected void defaultIn(Node node) {
    }

    /**
     * A default method called from all outX methods. Override
     * this method in a subclass to handle the default case.
     *
     * @param  node   a parse tree node
     */
    protected void defaultOut(Node node) {
    }

    /**
     * Traverses the child nodes recursively by switching them
     * on this object.
     *
     * @param  node   a parse tree node
     */
    protected final void defaultCase(Node node) {
        int   i = 0;

        while (i < node.children()) {
            node.childAt(i).apply(this);
            i++;
        }
    }
}
