/*
 * @(#)Mib.java
 *
 * This work is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * As a special exception, the copyright holders of this library 
 * give you permission to link this library with independent modules
 * to produce an executable, regardless of the license terms of 
 * these independent modules, and to copy and distribute the 
 * resulting executable under terms of your choice, provided that 
 * you also meet, for each linked independent module, the terms and 
 * conditions of the license of that module. An independent module 
 * is a module which is not derived from or based on this library. 
 * If you modify this library, you may extend this exception to your 
 * version of the library, but you are not obligated to do so. If 
 * you do not wish to do so, delete this exception statement from 
 * your version. 
 *
 * Copyright (c) 1999 Ericsson Telecom. All rights reserved.
 * Copyright (c) 2003 Per Cederberg. All rights reserved.
 */

package net.percederberg.mib;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

import net.percederberg.mib.asn1.node.Node;
import net.percederberg.mib.asn1.parser.AsnParser;
import net.percederberg.mib.asn1.parser.ParseException;
import net.percederberg.mib.symbol.Symbol;

/**
 * A class representing a MIB file. It also serves as a symbol table
 * for the symbols in the MIB file. Each Mib object represents a
 * distinct ASN.1 module, and several objects are thus required to
 * represent multiple modules. An ASN.1 symbol can be uniqely
 * identified by it's MIB (module) and symbol name.<p>
 *
 * This class also stores all the errors, warnings and messages output
 * during the MIB file reading. The MIB file parsing will only be
 * interrupted by a critical error (syntax) error.
 *
 * @version  1.1
 * @author   Per Cederberg, per@percederberg.net
 */
public class Mib extends Object {

    /**
     * The MIB file.
     */
    private File file = null;

    /**
     * The MIB name.
     */
    private String name = null;

    /**
     * The hash table containing all symbols. The hash table is needed
     * in order to find a named symbol rapidly.
     */
    private Hashtable symbolTable = new Hashtable();

    /**
     * A vector containing all symbols in order. The symbol list is
     * needed to preserve the symbol order from the file.
     */
    private Vector symbolList = new Vector();

    /**
     * A vector with all error messages.
     */
    private Vector errors = new Vector();

    /**
     * A vector with all warning messages.
     */
    private Vector warnings = new Vector();

    /**
     * A vector with all other messages.
     */
    private Vector messages = new Vector();

    /**
     * Creates a MIB by reading and parsing the specified file.
     *
     * @param filename  the name of a the MIB file to read
     *
     * @throws FileNotFoundException if the file couldn't be found
     * @throws ParseException if the file couldn't be parsed correctly
     */
    public Mib(String filename)
        throws FileNotFoundException, ParseException {

        this(new File(filename));
    }

    /**
     * Creates a MIB by reading and parsing the specified file.
     *
     * @param file      the MIB file to read
     *
     * @throws FileNotFoundException if the file couldn't be found
     * @throws ParseException if the file couldn't be parsed correctly
     */
    public Mib(File file)
        throws FileNotFoundException, ParseException {

        // Initialize
        this.file = file;
        this.name = "";

        // Parse MIB file
        Node parseTree = AsnParser.parse(new FileInputStream(file));
        parseTree.apply(new FirstPassAnalysis(this));
        if (getErrorCount() == 0) {
            parseTree.apply(new SecondPassAnalysis(this));
        }
    }

    /**
     * Checks if this MIB is equal to some other object. This method
     * compares the name of the MIB with the string representation of
     * the specified object.
     *
     * @param obj       the object to compare with
     *
     * @return true if the objects are equal, or
     *         false otherwise
     */
    public boolean equals(Object obj) {
        return name.equals(obj.toString());
    }

    /**
     * Returns the MIB name.
     *
     * @return the MIB name
     */
    public String toString() {
        return name;
    }

    /**
     * Returns the MIB file.
     *
     * @return the MIB file
     */
    public File getFile() {
        return file;
    }

    /**
     * Returns the MIB name.
     *
     * @return the MIB name
     */
    public String getName() {
        return name;
    }

    /**
     * Returns the number of symbols in this MIB.
     *
     * @return the number of symbols in this MIB
     */
    public int getSymbolCount() {
        return symbolList.size();
    }

    /**
     * Returns a specified symbol in this MIB. The symbol index is
     * specified in the range from 0 to getSymbolCount() - 1.
     *
     * @param index    the symbol index
     *
     * @return the symbol found, or
     *         null if not found
     */
    public Symbol getSymbol(int index) {
        if (index >= 0 && index < symbolList.size()) {
            return (Symbol) symbolList.get(index);
        } else {
            return null;
        }
    }

    /**
     * Returns a specified symbol in this MIB. The symbol name should
     * be specified in a case-sensitive manner.
     *
     * @param name     the symbol name
     *
     * @return the symbol found, or
     *         null if not found
     */
    public Symbol getSymbol(String name) {
        return (Symbol) symbolTable.get(name);
    }

    /**
     * Returns an enumeration of all the symbols in this MIB. The
     * symbols will appear in the same order as in the MIB file.
     *
     * @return an enumeration with all the symbols in this MIB
     */
    public Enumeration getAllSymbols() {
        return symbolList.elements();
    }

    /**
     * Returns the number of parse errors in this MIB.
     *
     * @return the number of parse errors in this MIB
     */
    public int getErrorCount() {
        return errors.size();
    }

    /**
     * Returns a specified error message. The index is specified in
     * the range from 0 to getErrorCount() - 1.
     *
     * @param index    the error index
     *
     * @return the error message, or
     *         null if not found
     */
    public String getError(int index) {
        if (index < 0 || index >= errors.size()) {
            return null;
        } else {
            return errors.get(index).toString();
        }
    }

    /**
     * Returns the number of parse warnings in this MIB.
     *
     * @return the number of parse warnings in this MIB
     */
    public int getWarningCount() {
        return warnings.size();
    }

    /**
     * Returns a specified warning message. The index is specified in
     * the range from 0 to getWarningCount() - 1.
     *
     * @param index    the warning index
     *
     * @return the warning message, or
     *         null if not found
     */
    public String getWarning(int index) {
        if (index < 0 || index >= warnings.size()) {
            return null;
        } else {
            return warnings.get(index).toString();
        }
    }

    /**
     * Returns the number of parse messages in this MIB.
     *
     * @return the number of parse messages in this MIB
     */
    public int getMessageCount() {
        return messages.size();
    }

    /**
     * Returns a specified parse message. The index is specified in
     * the range from 0 to getMessageCount() - 1.
     *
     * @param index    the message index
     *
     * @return the parse message, or
     *         null if not found
     */
    public String getMessage(int index) {
        if (index < 0 || index >= messages.size()) {
            return null;
        } else {
            return messages.get(index).toString();
        }
    }

    /**
     * Returns a string with all the parse message. The messages will
     * be ordered by category.
     *
     * @return a string with all parse messages
     */
    public String getAllMessages() {
        return vectorToString(errors) + vectorToString(warnings) +
               vectorToString(messages);
    }

    /**
     * Sets the MIB name.
     *
     * @param name    the new MIB name
     */
    void setName(String name) {
        this.name = name;
    }

    /**
     * Adds a symbol to the symbol table.
     *
     * @param   sym      the symbol to add
     */
    void addSymbol(Symbol sym) {
        symbolTable.put(sym.getName(), sym);
        symbolList.addElement(sym);
    }

    /**
     * Adds an error message without line information.
     *
     * @param   message      the error message to add
     */
    void addError(String message) {
        errors.add(new Message(Message.ERROR, message));
    }

    /**
     * Adds an error message with line information.
     *
     * @param   message      the error message to add
     * @param   firstLine    the first line affected
     * @param   lastLine     the last line affected
     */
    void addError(String message, int firstLine, int lastLine) {
        errors.add(new Message(Message.ERROR,
                               message,
                               firstLine,
                               lastLine));
    }

    /**
     * Adds a warning message without line information.
     *
     * @param   message      the warning message to add
     */
    void addWarning(String message) {
        warnings.add(new Message(Message.WARNING, message));
    }

    /**
     * Adds a warning message with line information.
     *
     * @param   message      the warning message to add
     * @param   firstLine    the first line affected
     * @param   lastLine     the last line affected
     */
    void addWarning(String message, int firstLine, int lastLine) {
        warnings.add(new Message(Message.WARNING,
                                 message,
                                 firstLine,
                                 lastLine));
    }

    /**
     * Adds a message without line information.
     *
     * @param   message      the message to add
     */
    void addMessage(String message) {
        messages.add(new Message(Message.OTHER, message));
    }

    /**
     * Adds a message with line information.
     *
     * @param   message      the message to add
     * @param   firstLine    the first line affected
     * @param   lastLine     the last line affected
     */
    void addMessage(String message, int firstLine, int lastLine) {
        messages.add(new Message(Message.OTHER,
                                 message,
                                 firstLine,
                                 lastLine));
    }

    /**
     * Returns a string with the vector elements concatenated with
     * newline characters.
     *
     * @param v     a vector
     *
     * @return a string with the contents of the vector
     */
    private String vectorToString(Vector v) {
        Enumeration   e;
        String        result = "";

        if (v == null) {
            // Do nothing
            return "";
        }
        e = v.elements();
        while (e.hasMoreElements()) {
            result = result + e.nextElement() + "\n";
        }
        return result;
    }
}


/**
 * A class that handles a single error or warning message.
 *
 * @version  1.0
 * @author   Per Cederberg, per@percederberg.net
 */
class Message extends Object {

    /**
     * The error message type.
     */
    public static final int ERROR    = 0;

    /**
     * The warning message type.
     */
    public static final int WARNING  = 1;

    /**
     * The other message type.
     */
    public static final int OTHER    = 2;

    /**
     * The message type string.
     */
    private static final String[] MESSAGE_DESCRIPTIONS = {
        "ERROR",
        "Warning",
        "Message" };

    /**
     * The message type.
     */
    protected int     type;

    /**
     * The message text.
     */
    protected String  message;

    /**
     * The first line that the message applies to.
     */
    protected int     firstLine;

    /**
     * The last line that the message applies to.
     */
    protected int     lastLine;

    /**
     * Creates a new message with the OTHER type and no line numbers.
     *
     * @param   message     the message text
     */
    public Message(String message) {
        this(OTHER, message);
    }

    /**
     * Creates a new message with no line numbers.
     *
     * @param   type        the message type
     * @param   message     the message text
     */
    public Message(int type, String message) {
        this(type, message, -1, -1);
    }

    /**
     * Creates a new message with the given parameters.
     *
     * @param   type        the message type
     * @param   message     the message text
     * @param   firstLine   the first line to refer to
     * @param   lastLine    the last line to refer to
     */
    public Message(int type,
                   String message,
                   int firstLine,
                   int lastLine) {

        this.type = type;
        this.message = message;
        this.firstLine = firstLine;
        if (lastLine <= 0) {
            this.lastLine = firstLine;
        } else {
            this.lastLine = lastLine;
        }
    }

    /**
     * Returns a string description of this message.
     *
     * @return a string description of the message
     */
    public String toString() {
        return MESSAGE_DESCRIPTIONS[type] + linesToString() + ": " +
            message;
    }

    /**
     * Returns a string description of the line numbers.
     *
     * @return a line number description string
     */
    private String linesToString() {
        if (firstLine > 0 && firstLine != lastLine) {
            return " (lines " + firstLine + "-" + lastLine + ")";
        } else if (firstLine > 0) {
            return " (line " + firstLine + ")";
        } else {
            return "";
        }
    }
}
