/*
 * @(#)TreePrinter.java
 *
 * This work is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * As a special exception, the copyright holders of this library 
 * give you permission to link this library with independent modules
 * to produce an executable, regardless of the license terms of 
 * these independent modules, and to copy and distribute the 
 * resulting executable under terms of your choice, provided that 
 * you also meet, for each linked independent module, the terms and 
 * conditions of the license of that module. An independent module 
 * is a module which is not derived from or based on this library. 
 * If you modify this library, you may extend this exception to your 
 * version of the library, but you are not obligated to do so. If 
 * you do not wish to do so, delete this exception statement from 
 * your version. 
 *
 * Copyright (c) 1999 Ericsson Telecom. All rights reserved.
 * Copyright (c) 2003 Per Cederberg. All rights reserved.
 */

package net.percederberg.mib.asn1.analysis;

import java.io.PrintWriter;
import net.percederberg.mib.asn1.node.Node;

/**
 * This class provides a tree printing mechanism. The tree is
 * traversed in depth first order and all nodes in it are printed.
 *
 * @version  1.0
 * @author   Per Cederberg, per@percederberg.net
 */
public class TreePrinter extends DepthFirstAdapter {

    /**
     * The current indentation level.
     */
    private int currentIndent = 0;

    /**
     * The indentation increment.
     */
    private int indentStep = 2;

    /**
     * The output stream.
     */
    private PrintWriter out;

    /**
     * Creates a new tree printer with an indentation step of 2.
     *
     * @param out     the output stream
     */
    public TreePrinter(PrintWriter out) {
        this(out, 2);
    }

    /**
     * Creates a new tree printer with the given indentation step.
     *
     * @param out          the output stream
     * @param indentStep   the size of each indentation step
     */
    public TreePrinter(PrintWriter out, int indentStep) {
        this.out = out;
        this.indentStep = indentStep;
    }

    /**
     * A default method called from all inX methods. Override
     * this method in a subclass to handle the default case.
     *
     * @param node   a parse tree node
     */
    protected void defaultIn(Node node) {
        int   i;

        for (i = 0; i < currentIndent; i++) {
            out.print(" ");
        }
        out.println(node.toString());
        currentIndent += indentStep;
    }

    /**
     * A default method called from all outX methods. Override
     * this method in a subclass to handle the default case.
     *
     * @param  node   a parse tree node
     */
    protected void defaultOut(Node node) {
        currentIndent -= indentStep;
    }
}
