/*
 * @(#)SnmpObjectType.java
 *
 * This work is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * As a special exception, the copyright holders of this library 
 * give you permission to link this library with independent modules
 * to produce an executable, regardless of the license terms of 
 * these independent modules, and to copy and distribute the 
 * resulting executable under terms of your choice, provided that 
 * you also meet, for each linked independent module, the terms and 
 * conditions of the license of that module. An independent module 
 * is a module which is not derived from or based on this library. 
 * If you modify this library, you may extend this exception to your 
 * version of the library, but you are not obligated to do so. If 
 * you do not wish to do so, delete this exception statement from 
 * your version. 
 *
 * Copyright (c) 1999 Ericsson Telecom. All rights reserved.
 * Copyright (c) 2003 Per Cederberg. All rights reserved.
 */

package net.percederberg.mib.type;

import java.util.Vector;

/**
 * A class for handling type information for an SNMP object type.
 *
 * @version  1.1
 * @author   Per Cederberg, per@percederberg.net
 */
public class SnmpObjectType extends Type {

    /**
     * The access constant for no access. This value is used in both
     * SNMPv1 and SNMPv2.
     */
    public static final int NO_ACCESS = 0;

    /**
     * The access constant for notify access. This value is only used 
     * in SNMPv2.
     */
    public static final int NOTIFY_ACCESS = 1;

    /**
     * The access constant for read-only access. This value is used in 
     * both SNMPv1 and SNMPv2.
     */
    public static final int READ_ACCESS = 2;

    /**
     * The access constant for write-only access. This value is only
     * used in SNMPv1.
     */
    public static final int WRITE_ACCESS = 3;

    /**
     * The access constant for read-write access. This value is used 
     * in both SNMPv1 and SNMPv2.
     */
    public static final int READ_WRITE_ACCESS = 4;

    /**
     * The access constant for read-create access. This value is only
     * used in SNMPv2.
     */
    public static final int READ_CREATE_ACCESS = 5;

    /**
     * The mandatory status constant. This value is only used in 
     * SNMPv1.
     */
    public static final int MANDATORY_STATUS = 1;

    /**
     * The optional status constant. This value is only used in 
     * SNMPv1.
     */
    public static final int OPTIONAL_STATUS = 2;

    /**
     * The current status constant. This value is only used in 
     * SNMPv2.
     */
    public static final int CURRENT_STATUS = 3;
    
    /**
     * The deprecated status constant. This value is only used in 
     * SNMPv2.
     */
    public static final int DEPRECATED_STATUS = 4;

    /**
     * The obsolete status constant. This value is used in both SNMPv1
     * and SNMPv2.
     */
    public static final int OBSOLETE_STATUS = 5;

    /**
     * The basic type.
     */
    private Type baseType;

    /**
     * The access level value. This should be one of the access
     * constants.
     *
     * @see #NO_ACCESS
     * @see #NOTIFY_ACCESS
     * @see #READ_ACCESS
     * @see #WRITE_ACCESS
     * @see #READ_WRITE_ACCESS
     * @see #READ_CREATE_ACCESS
     */
    private int access;

    /**
     * The status value. This should be one of the status constants.
     *
     * @see #MANDATORY_STATUS
     * @see #OPTIONAL_STATUS
     * @see #CURRENT_STATUS
     * @see #DEPRECATED_STATUS
     * @see #OBSOLETE_STATUS
     */
    private int status;

    /**
     * The string description.
     */
    private String description;

    /**
     * The table indices.
     */
    private Vector index;

    /**
     * The default value
     */
    private Object defval;

    /**
     * Creates a new SNMP object type with the given parameters.
     * The description is set to an empty string.
     *
     * @param baseType    the basic type
     * @param access      an access constant value
     * @param status      a status constant value
     */
    public SnmpObjectType(Type baseType, int access, int status) {
        this(baseType, access, status, null, null, null);
    }

    /**
     * Creates a new SNMP object type with the given parameters.
     *
     * @param baseType    the basic type
     * @param access      an access constant value
     * @param status      a status constant value
     * @param descr       a string description (or null)
     * @param index       the table indices (or null)
     * @param defval      the default value (or null)
     */
    public SnmpObjectType(Type baseType,
                          int access,
                          int status,
                          String descr,
                          Vector index,
                          Object defval) {

        this.baseType = baseType;
        this.access = access;
        this.status = status;
        this.description = descr;
        this.index = index;
        this.defval = defval;
    }

    /**
     * Returns the access string.
     *
     * @return the access string
     */
    public String getAccess() {
        switch (access) {
        case NO_ACCESS:
            return "no access";
        case NOTIFY_ACCESS:
            return "notify";
        case READ_ACCESS:
            return "read";
        case WRITE_ACCESS:
            return "write";
        case READ_WRITE_ACCESS:
            return "read-write";
        case READ_CREATE_ACCESS:
            return "read-create";
        }
        return "*** UNKNOWN ***";
    }

    /**
     * Returns the basic type.
     *
     * @return the basic type
     */
    public Type getBaseType() {
        return baseType;
    }

    /**
     * Returns the default value.
     *
     * @return the default value
     */
    public Object getDefaultValue() {
        return defval;
    }

    /**
     * Returns the type description.
     *
     * @return the type description
     */
    public String getDescription() {
        if (description == null) {
            return "";
        } else {
            return description;
        }
    }

    /**
     * Returns the table indices (as a string).
     *
     * @return the table indices
     */
    public String getIndices() {
        if (index == null) {
            return "";
        } else {
            return index.toString();
        }
    }

    /**
     * Returns the status string.
     *
     * @return the status string
     */
    public String getStatus() {
        switch (status) {
        case MANDATORY_STATUS:
            return "mandatory";
        case OPTIONAL_STATUS:
            return "optional";
        case CURRENT_STATUS:
            return "current";
        case DEPRECATED_STATUS:
            return "deprecated";
        case OBSOLETE_STATUS:
            return "obsolete";
        }
        return "*** UNKNOWN ***";
    }

    /**
     * Checks if this type equals another.
     *
     * @param  obj       an object
     *
     * @return true if the types are equal, false otherwise
     */
    public boolean equals(Object obj) {
        return this == obj;
    }

    /**
     * Returns a description of this type.
     *
     * @return a description of the type
     */
    public String toString() {
        if (baseType != null) {
            return baseType.toString();
        } else {
            return "*** UNKNOWN TYPE ***";
        }
    }

    /**
     * Transfers the type information from this type to a type
     * converter object. The calling conventions declared in the
     * TypeConverter class are followed.
     *
     * @param   converter     a type converter
     */
    public void transferType(TypeConverter converter) {
        baseType.transferType(converter);
        if (access >= READ_ACCESS && access != WRITE_ACCESS) {
            converter.transferReadable(true);
        } else {
            converter.transferReadable(false);
        }
        if (access >= WRITE_ACCESS) {
            converter.transferWriteable(true);
        } else {
            converter.transferWriteable(false);
        }
        if (defval instanceof Integer) {
            converter.transferDefaultValue(((Integer) defval).intValue());
        } else {
            converter.transferDefaultValue(defval.toString());
        }
    }

}
